package threads.core.api;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.room.ColumnInfo;
import androidx.room.PrimaryKey;
import androidx.room.TypeConverters;

import java.util.Date;
import java.util.Objects;

import threads.core.MimeType;
import threads.core.THREADS;
import threads.ipfs.api.CID;
import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkNotNull;

@androidx.room.Entity
public class Thread extends Entity {

    @ColumnInfo(name = "thread")
    private final long thread;
    @NonNull
    @TypeConverters(Kind.class)
    @ColumnInfo(name = "kind")
    private final Kind kind;
    @NonNull
    @TypeConverters(Converter.class)
    @ColumnInfo(name = "senderPid")
    private final PID senderPid;
    @NonNull
    @ColumnInfo(name = "senderKey")
    private final String senderKey;
    @NonNull
    @ColumnInfo(name = "sesKey")
    private final String sesKey;
    @NonNull
    @TypeConverters(Converter.class)
    @ColumnInfo(name = "date")
    private Date date;
    @NonNull
    @ColumnInfo(name = "senderAlias")
    private String senderAlias;
    @PrimaryKey(autoGenerate = true)
    private long idx;
    @Nullable
    @ColumnInfo(name = "image")
    @TypeConverters(Converter.class)
    private CID image;
    @ColumnInfo(name = "markedFlag")
    private boolean markedFlag;
    @ColumnInfo(name = "unreadNotes")
    private int unreadNotes = 0;
    @Nullable
    @TypeConverters(Converter.class)
    @ColumnInfo(name = "cid")
    private CID cid;
    @NonNull
    @TypeConverters(Converter.class)
    @ColumnInfo(name = "expireDate")
    private Date expireDate;
    @NonNull
    @TypeConverters(Status.class)
    @ColumnInfo(name = "status")
    private Status status;
    @NonNull
    @TypeConverters(Members.class)
    @ColumnInfo(name = "members")
    private Members members = new Members();
    @NonNull
    @ColumnInfo(name = "mimeType")
    private String mimeType;
    @ColumnInfo(name = "pinned")
    private boolean pinned;

    @ColumnInfo(name = "publish")
    private boolean publish;
    @ColumnInfo(name = "request")
    private boolean request;

    Thread(@NonNull Status status,
           @NonNull PID senderPid,
           @NonNull String senderAlias,
           @NonNull String senderKey,
           @NonNull String sesKey,
           @NonNull Kind kind,
           @NonNull Date date,
           long thread) {
        this.thread = thread;
        this.senderPid = senderPid;
        this.senderAlias = senderAlias;
        this.senderKey = senderKey;

        this.sesKey = sesKey;
        this.kind = kind;
        this.expireDate = THREADS.getOffsetYearDate(-20);
        this.status = status;
        this.markedFlag = false;
        this.date = date;
        this.mimeType = MimeType.PLAIN_MIME_TYPE;
        this.pinned = false;
        this.publish = false;
        this.request = false;
    }

    public static Thread createThread(@NonNull Status status,
                                      @NonNull PID senderPid,
                                      @NonNull String senderAlias,
                                      @NonNull String senderKey,
                                      @NonNull String sesKey,
                                      @NonNull Kind kind,
                                      @NonNull Date date,
                                      long thread) {
        checkNotNull(status);
        checkNotNull(senderPid);
        checkNotNull(senderAlias);
        checkNotNull(senderKey);
        checkNotNull(sesKey);
        checkNotNull(kind);
        checkNotNull(date);
        return new Thread(status,
                senderPid, senderAlias, senderKey,
                sesKey, kind, date, thread);
    }

    public boolean isRequest() {
        return request;
    }

    public void setRequest(boolean request) {
        this.request = request;
    }

    public boolean isPublish() {
        return publish;
    }

    public void setPublish(boolean publish) {
        this.publish = publish;
    }

    public boolean isPinned() {
        return pinned;
    }

    public void setPinned(boolean pinned) {
        this.pinned = pinned;
    }

    public long getIdx() {
        return idx;
    }

    void setIdx(long idx) {
        this.idx = idx;
    }

    @NonNull
    public String getSenderBox() {
        return AddressType.getAddress(getSenderPid(), AddressType.INBOX);
    }

    @NonNull
    public String getSenderKey() {
        return senderKey;
    }

    @NonNull
    public String getMimeType() {
        return mimeType;
    }

    public void setMimeType(@NonNull String mimeType) {
        this.mimeType = mimeType;
    }

    @NonNull
    public PID getSenderPid() {
        return senderPid;
    }

    @NonNull
    public Date getExpireDate() {
        return expireDate;
    }

    public void setExpireDate(@NonNull Date expireDate) {
        this.expireDate = expireDate;
    }

    @NonNull
    public Status getStatus() {
        return status;
    }

    public void setStatus(@NonNull Status status) {
        checkNotNull(status);
        this.status = status;
    }

    @NonNull
    public Kind getKind() {
        return kind;
    }

    @NonNull
    public Members getMembers() {
        return members;
    }

    public void setMembers(@NonNull Members members) {
        this.members = members;
    }

    public boolean addMember(@NonNull PID pid) {
        checkNotNull(pid);
        return this.members.add(pid);
    }

    public boolean removeMember(@NonNull PID pid) {
        checkNotNull(pid);
        return this.members.remove(pid);
    }

    public boolean sameThread(@NonNull Thread o) {
        checkNotNull(o);
        if (this == o) return true;
        return Objects.equals(cid, o.getCid()) &&
                Objects.equals(senderPid, o.getSenderPid()) &&
                Objects.equals(image, o.getImage()) &&
                Objects.equals(date, o.getDate());
    }

    public boolean sameContent(@NonNull Thread o) {
        checkNotNull(o);
        if (this == o) return true;
        return unreadNotes == o.getUnreadNotes() &&
                markedFlag == o.getMarkedFlag() &&
                status == o.getStatus() &&
                pinned == o.isPinned() &&
                publish == o.isPublish() &&
                request == o.isRequest() &&
                Objects.equals(cid, o.getCid()) &&
                Objects.equals(senderAlias, o.getSenderAlias()) &&
                Objects.equals(image, o.getImage()) &&
                Objects.equals(date, o.getDate());
    }

    @Nullable
    public CID getImage() {
        return image;
    }

    public void setImage(@Nullable CID image) {
        this.image = image;
    }

    public boolean areItemsTheSame(@NonNull Thread thread) {
        checkNotNull(thread);
        return idx == thread.getIdx();

    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Thread thread = (Thread) o;
        return getIdx() == thread.getIdx();
    }

    @Override
    public int hashCode() {
        return Objects.hash(getIdx());
    }

    @Nullable
    public CID getCid() {
        return cid;
    }

    public void setCid(@Nullable CID cid) {
        this.cid = cid;
    }

    @NonNull
    public String getSenderAlias() {
        return senderAlias;
    }

    public void setSenderAlias(@NonNull String senderAlias) {
        checkNotNull(senderAlias);
        this.senderAlias = senderAlias;
    }

    public boolean getMarkedFlag() {
        return markedFlag;
    }

    public void setMarkedFlag(boolean markedFlag) {
        this.markedFlag = markedFlag;
    }

    public int getUnreadNotes() {
        return unreadNotes;
    }

    public void setUnreadNotes(int unreadNotes) {
        this.unreadNotes = unreadNotes;
    }

    public long getThread() {
        return thread;
    }

    @NonNull
    public String getSesKey() {
        return sesKey;
    }

    public void increaseUnreadMessagesNumber() {
        unreadNotes++;
    }

    @NonNull
    public Date getDate() {
        return date;
    }

    public void setDate(@NonNull Date date) {
        checkNotNull(date);
        this.date = date;
    }


    public boolean isEncrypted() {
        return !sesKey.isEmpty();
    }

}
