package threads.core;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import threads.core.api.Peer;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;
import threads.ipfs.api.PeerInfo;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class GatewayService {

    public static final String TAG = GatewayService.class.getSimpleName();


    public static int evaluatePeers(@NonNull Context context, boolean pubsubs) {
        IPFS ipfs = Singleton.getInstance(context).getIpfs();
        checkNotNull(ipfs);
        List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();
        int size = peers.size();
        for (threads.ipfs.api.Peer peer : peers) {
            if(pubsubs){
                if (peer.isAutonat() || peer.isRelay() || peer.isMeshSub() || peer.isFloodSub()) {
                    storePeer(context, peer);
                }
            } else {
                if (peer.isAutonat() || peer.isRelay()) {
                    storePeer(context, peer);
                }
            }
        }
        return size;
    }



    public synchronized static List<threads.core.api.Peer> getPeers(
            @NonNull Context context,
            int numRelays,
            int timeout,
            boolean protect) {

        checkNotNull(context);
        checkArgument(numRelays >= 0);
        checkArgument(timeout > 0);

        List<threads.core.api.Peer> result = new ArrayList<>();


        if (!Network.isConnected(context)) {
            return result;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();


        if (ipfs != null) {

            List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();

            peers.sort(threads.ipfs.api.Peer::compareTo);

            for (threads.ipfs.api.Peer peer : peers) {

                if (result.size() == numRelays) {
                    break;
                }

                if (peer.isRelay()) {

                    if (ipfs.isConnected(peer.getPid())) {

                        if (protect) {
                            ipfs.protectPeer(peer.getPid(), TAG);
                        }

                        result.add(storePeer(context, peer));


                    } else if (ipfs.swarmConnect(peer, timeout)) {

                        if (protect) {
                            ipfs.protectPeer(peer.getPid(), TAG);
                        }

                        result.add(storePeer(context, peer));

                    }

                }
            }

        }

        return result;
    }

    private static Peer storePeer(@NonNull Context context, @NonNull threads.ipfs.api.Peer peer) {

        // the given peer is connected (so rating will be dependent of peer
        int rating = 0;
        try {
            double latency = peer.getLatency();
            if (latency < 1000) {
                rating = (int) (1000 - latency);
            }

        } catch (Throwable e) {
            // ignore any exceptions here
        }

        // now add higher rating when peer has specific attributes
        try {
            int timeout = 5;
            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            if (ipfs != null) {
                PeerInfo info = ipfs.id(peer, timeout);
                if (info != null) {

                    String protocol = info.getProtocolVersion();
                    String agent = info.getAgentVersion();

                    if (protocol != null && protocol.equals("ipfs/0.1.0")) {
                        rating = rating + 100;
                    } else {
                        rating = rating - 100;
                    }
                    if (agent != null && agent.startsWith("go-ipfs/0.4.2")) {
                        rating = rating + 100;
                    }
                }
            }
        } catch (Throwable e) {
            // ignore any exceptions here
        }
        if (rating < 0) {
            rating = 0;
        }
        boolean isPubsub = peer.isFloodSub() || peer.isMeshSub();
        return storePeer(context, peer.getPid(),
                peer.getMultiAddress(), peer.isRelay(), peer.isAutonat(), isPubsub, rating);
    }


    public static threads.core.api.Peer storePeer(@NonNull Context context,
                                                  @NonNull PID pid,
                                                  @NonNull String multiAddress,
                                                  boolean isRelay,
                                                  boolean isAutonat,
                                                  boolean isPubsub,
                                                  int rating) {

        final THREADS threads = Singleton.getInstance(context).getThreads();

        threads.core.api.Peer peer = threads.getPeerByPID(pid);
        if (peer != null) {
            peer.setMultiAddress(multiAddress);
            peer.setRelay(isRelay);
            peer.setAutonat(isAutonat);
            peer.setPubsub(isPubsub);
            peer.setRating(rating);
            threads.updatePeer(peer);
            Log.e(TAG, "Update Peer  : " + peer.toString());
        } else {
            peer = threads.createPeer(pid, multiAddress);
            peer.setRelay(isRelay);
            peer.setAutonat(isAutonat);
            peer.setPubsub(isPubsub);
            peer.setRating(rating);
            threads.storePeer(peer);
            Log.e(TAG, "Store Peer  : " + peer.toString());
        }
        return peer;
    }

    public static void connectStoredAutonat(@NonNull Context context,
                                            int numConnections,
                                            int timeout,
                                            boolean cleanStoredPeers) {
        checkNotNull(context);
        checkArgument(numConnections >= 0);
        checkArgument(timeout > 0);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {

            List<threads.core.api.Peer> peers = threads.getAutonatPeers();

            peers.sort(Peer::compareTo);

            for (threads.core.api.Peer autonat : peers) {

                if (counter.get() == numConnections) {
                    break;
                }

                Log.e(TAG, "Stored Autonat : " + autonat.toString());

                if (ipfs.isConnected(PID.create(autonat.getPid()))) {
                    counter.incrementAndGet();
                } else {

                    String ma = autonat.getMultiAddress() + "/" +
                            IPFS.Style.p2p.name() + "/" + autonat.getPid();

                    if (ipfs.swarmConnect(ma, timeout)) {
                        counter.incrementAndGet();

                    } else {
                        if (cleanStoredPeers) {
                            threads.removePeer(autonat);
                        } else {
                            autonat.setRating(0);
                            threads.updatePeer(autonat);
                        }
                    }
                }
            }
        }
    }


    public static void connectStoredPubsub(@NonNull Context context,
                                            int numConnections,
                                            int timeout,
                                            boolean cleanStoredPeers) {
        checkNotNull(context);
        checkArgument(numConnections >= 0);
        checkArgument(timeout > 0);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {

            List<threads.core.api.Peer> peers = threads.getPubsubPeers();

            peers.sort(Peer::compareTo);

            for (threads.core.api.Peer autonat : peers) {

                if (counter.get() == numConnections) {
                    break;
                }

                Log.e(TAG, "Stored Pubsub : " + autonat.toString());

                if (ipfs.isConnected(PID.create(autonat.getPid()))) {
                    counter.incrementAndGet();
                } else {

                    String ma = autonat.getMultiAddress() + "/" +
                            IPFS.Style.p2p.name() + "/" + autonat.getPid();

                    if (ipfs.swarmConnect(ma, timeout)) {
                        counter.incrementAndGet();

                    } else {
                        if (cleanStoredPeers) {
                            threads.removePeer(autonat);
                        } else {
                            autonat.setRating(0);
                            threads.updatePeer(autonat);
                        }
                    }
                }
            }
        }
    }

    public static void connectStoredRelays(@NonNull Context context,
                                           int numConnections,
                                           int timeout,
                                           boolean cleanStoredPeers) {
        checkNotNull(context);
        checkArgument(numConnections >= 0);
        checkArgument(timeout > 0);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {

            List<threads.core.api.Peer> peers = threads.getRelayPeers();

            peers.sort(Peer::compareTo);

            for (threads.core.api.Peer relay : peers) {

                if (counter.get() == numConnections) {
                    break;
                }

                Log.e(TAG, "Stored Relay : " + relay.toString());

                if (ipfs.isConnected(PID.create(relay.getPid()))) {
                    counter.incrementAndGet();
                } else {

                    String ma = relay.getMultiAddress() + "/" +
                            IPFS.Style.p2p.name() + "/" + relay.getPid();

                    if (ipfs.swarmConnect(ma, timeout)) {
                        counter.incrementAndGet();

                    } else {
                        if (cleanStoredPeers) {
                            threads.removePeer(relay);
                        } else {
                            relay.setRating(0);
                            threads.updatePeer(relay);
                        }
                    }
                }
            }
        }
    }
}
