package threads.core;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import threads.core.api.Members;
import threads.core.api.Note;
import threads.core.api.Thread;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class ConnectService {

    private static final String TAG = ConnectService.class.getSimpleName();


    public static List<Future<Boolean>> connectMembersAsync(
            @NonNull Context context,
            @NonNull threads.core.api.Thread thread,
            @NonNull String aesKey,
            boolean supportDiscovery,
            int timeout) {
        checkNotNull(context);
        checkNotNull(thread);
        checkNotNull(aesKey);
        checkArgument(timeout > 0);

        boolean protectPeer = true;
        boolean protectRelay = true;

        ArrayList<Future<Boolean>> futures = new ArrayList<>();
        ExecutorService executorService = Executors.newFixedThreadPool(5);
        Members members = thread.getMembers();
        for (PID pid : members) {
            Future<Boolean> future = executorService.submit(() -> {
                try {
                    return IdentityService.connectPeer(context, pid,
                            aesKey, timeout, supportDiscovery, protectRelay, protectPeer);
                } catch (Throwable e) {
                    Log.e(TAG, e.getLocalizedMessage(), e);
                    return false;
                }
            });
            futures.add(future);
        }
        return futures;
    }


    public static boolean publishNote(@NonNull Context context,
                                      @NonNull Note note,
                                      @NonNull String aesKey) {

        checkNotNull(context);
        checkNotNull(note);
        checkNotNull(aesKey);

        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        Thread thread = threads.getThread(note);
        checkNotNull(thread);

        boolean notifyAll = true;
        Members members = thread.getMembers();
        for (PID pid : members) {

            try {

                boolean result = ipfs.isConnected(pid);
                if (result) {
                    result = threads.publishNote(ipfs, pid, note);
                }

                if (!result) {
                    notifyAll = false;
                }

            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
                return false;
            }

        }
        return notifyAll;
    }

    public static List<Future<Boolean>> connectMembersAsync(
            @NonNull Context context,
            @NonNull Thread thread,
            @NonNull String aesKey,
            boolean supportDiscovery) {

        checkNotNull(context);
        checkNotNull(thread);
        checkNotNull(aesKey);
        ArrayList<Future<Boolean>> futures = new ArrayList<>();
        ExecutorService executorService = Executors.newFixedThreadPool(5);

        boolean protectPeer = true;
        boolean protectRelay = true;

        Members members = thread.getMembers();
        for (PID pid : members) {
            Future<Boolean> future = executorService.submit(() -> {
                try {
                    return IdentityService.connectPeer(context,
                            pid, aesKey, supportDiscovery, protectRelay, protectPeer);
                } catch (Throwable e) {
                    Log.e(TAG, e.getLocalizedMessage(), e);
                    return false;
                }
            });
            futures.add(future);
        }
        return futures;
    }


    public static boolean connectMembers(@NonNull Context context,
                                         @NonNull Thread thread,
                                         @NonNull String aesKey,
                                         boolean supportDiscovery,
                                         boolean shortcut) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkNotNull(thread);

        boolean result = true;

        List<Future<Boolean>> futures = connectMembersAsync(
                context, thread, aesKey, supportDiscovery);
        for (Future<Boolean> future : futures) {
            try {
                if (!future.get()) {
                    result = false;
                    if (shortcut) {
                        return result;
                    }
                }
            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
                result = false;
            }
        }
        return result;
    }

    public static boolean connectMembers(@NonNull Context context,
                                         @NonNull threads.core.api.Thread thread,
                                         @NonNull String aesKey,
                                         boolean supportDiscovery,
                                         int timeout,
                                         boolean shortcut) {
        checkNotNull(context);
        checkNotNull(thread);
        checkNotNull(aesKey);
        checkArgument(timeout > 0);
        boolean result = true;
        List<Future<Boolean>> futures = connectMembersAsync(
                context, thread, aesKey, supportDiscovery, timeout);
        for (Future<Boolean> future : futures) {
            try {
                if (!future.get()) {
                    result = false;
                    if (shortcut) {
                        return result;
                    }
                }
            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
                result = false;
            }
        }
        return result;
    }
}
