package threads.core.api;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.room.ColumnInfo;
import androidx.room.PrimaryKey;
import androidx.room.TypeConverters;

import java.util.Objects;

import threads.ipfs.api.CID;
import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkNotNull;


@androidx.room.Entity
public class User extends Entity {
    @PrimaryKey
    @NonNull
    @ColumnInfo(name = "pid")
    private final String pid;
    @NonNull
    @ColumnInfo(name = "publicKey")
    private String publicKey;
    @NonNull
    @ColumnInfo(name = "alias")
    private String alias;
    @NonNull
    @TypeConverters(UserType.class)
    @ColumnInfo(name = "type")
    private UserType type;
    @NonNull
    @TypeConverters(UserStatus.class)
    @ColumnInfo(name = "status")
    private UserStatus status;
    @Nullable
    @ColumnInfo(name = "image")
    @TypeConverters(Converter.class)
    private CID image;
    @ColumnInfo(name = "blocked")
    private boolean blocked;

    User(@NonNull UserType type,
         @NonNull UserStatus status,
         @NonNull String alias,
         @NonNull String publicKey,
         @NonNull String pid,
         @Nullable CID image) {
        this.type = type;
        this.status = status;
        this.alias = alias;
        this.publicKey = publicKey;
        this.pid = pid;
        this.image = image;
        this.blocked = false;
    }

    @NonNull
    public static User createUser(@NonNull UserType type,
                                  @NonNull UserStatus status,
                                  @NonNull String alias,
                                  @NonNull String publicKey,
                                  @NonNull PID pid,
                                  @Nullable CID image) {
        checkNotNull(type);
        checkNotNull(status);
        checkNotNull(alias);
        checkNotNull(publicKey);
        checkNotNull(pid);
        return new User(type, status, alias, publicKey, pid.getPid(), image);
    }

    public boolean isBlocked() {
        return blocked;
    }

    public void setBlocked(boolean blocked) {
        this.blocked = blocked;
    }

    @NonNull
    public UserType getType() {
        return type;
    }

    public void setType(@NonNull UserType type) {
        checkNotNull(type);
        this.type = type;
    }

    @NonNull
    String getPid() {
        return pid;
    }


    @NonNull
    public UserStatus getStatus() {
        return status;
    }

    public void setStatus(@NonNull UserStatus status) {
        this.status = status;
    }

    @NonNull
    public String getAlias() {
        return alias;
    }

    public void setAlias(@NonNull String alias) {
        this.alias = alias;
    }

    @NonNull
    public String getPublicKey() {
        return publicKey;
    }

    public void setPublicKey(@NonNull String publicKey) {
        checkNotNull(publicKey);
        this.publicKey = publicKey;
    }

    @Nullable
    public CID getImage() {
        return image;
    }

    public void setImage(@Nullable CID image) {
        this.image = image;
    }

    @NonNull
    public String getSesKey() {
        return "";
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        User user = (User) o;
        return Objects.equals(pid, user.pid);
    }

    @Override
    public int hashCode() {
        return Objects.hash(pid);
    }

    public boolean areItemsTheSame(@NonNull User user) {
        checkNotNull(user);
        return this.pid.equals(user.pid);

    }

    public boolean sameContent(@NonNull User o) {
        checkNotNull(o);
        if (this == o) return true;
        return Objects.equals(status, o.getStatus()) &&
                Objects.equals(alias, o.getAlias()) &&
                Objects.equals(blocked, o.isBlocked());
    }

    public PID getPID() {
        return PID.create(pid);
    }
}
