package threads.core.api;

import androidx.lifecycle.LiveData;
import androidx.room.Dao;
import androidx.room.Delete;
import androidx.room.Insert;
import androidx.room.OnConflictStrategy;
import androidx.room.Query;
import androidx.room.TypeConverters;
import androidx.room.Update;

import java.util.Date;
import java.util.List;

import threads.ipfs.api.CID;
import threads.ipfs.api.PID;

@Dao
public interface ThreadDao {

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    long insertThread(Thread thread);

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    void insertThreads(Thread... threads);

    @Query("SELECT * FROM Thread")
    List<Thread> getThreads();

    @Query("SELECT * FROM Thread")
    LiveData<List<Thread>> getLiveDataThreads();

    @Query("DELETE FROM Thread")
    void clear();

    @Query("SELECT * FROM Thread WHERE date =:date")
    @TypeConverters({Converter.class})
    List<Thread> getThreadsByDate(Date date);

    @Query("SELECT * FROM Thread WHERE expireDate < :date")
    @TypeConverters({Converter.class})
    List<Thread> getExpiredThreads(Date date);

    @Query("SELECT * FROM Thread WHERE kind LIKE :kind AND status = :status")
    @TypeConverters({Kind.class, ThreadStatus.class})
    List<Thread> getThreadsByKindAndThreadStatus(Kind kind, ThreadStatus status);

    @Query("SELECT * FROM Thread WHERE pinned = :pinned")
    List<Thread> getThreadsByPinned(boolean pinned);

    @Query("SELECT mimeType FROM Thread WHERE idx = :idx")
    String getMimeType(long idx);

    @Query("SELECT cid FROM Thread WHERE idx = :idx")
    @TypeConverters({Converter.class})
    CID getCid(long idx);

    @Query("UPDATE Thread SET status = :status  WHERE idx = :idx")
    @TypeConverters({ThreadStatus.class})
    void setThreadStatus(long idx, ThreadStatus status);

    @Query("UPDATE Thread SET status = :status  WHERE idx IN (:idxs)")
    @TypeConverters({ThreadStatus.class})
    void setThreadsStatus(ThreadStatus status, long... idxs);

    @Query("UPDATE Thread SET senderAlias = :alias  WHERE idx = :idx")
    void setSenderAlias(long idx, String alias);

    @Query("UPDATE Thread SET pinned = :pinned  WHERE idx = :idx")
    void setPinned(long idx, boolean pinned);

    @Query("SELECT pinned FROM Thread WHERE idx = :idx")
    boolean isPinned(long idx);

    @Query("UPDATE Thread SET senderAlias = :alias  WHERE senderPid = :pid")
    @TypeConverters(Converter.class)
    void setSenderAlias(PID pid, String alias);

    @Query("UPDATE Thread SET status = :newStatus  WHERE status = :oldStatus")
    @TypeConverters({ThreadStatus.class})
    void setThreadStatus(ThreadStatus oldStatus, ThreadStatus newStatus);

    @Query("SELECT * FROM Thread WHERE status = :status")
    @TypeConverters({ThreadStatus.class})
    List<Thread> getThreadsByThreadStatus(ThreadStatus status);

    @Query("SELECT * FROM Thread WHERE cid = :cid")
    @TypeConverters({Converter.class})
    List<Thread> getThreadsByCid(CID cid);

    @Query("SELECT * FROM Thread WHERE cid = :cid AND thread = :thread")
    @TypeConverters({Converter.class})
    List<Thread> getThreadsByCidAndThread(CID cid, long thread);

    @Delete
    void removeThreads(Thread... threads);

    @Update(onConflict = OnConflictStrategy.REPLACE)
    void updateThreads(Thread... threads);

    @Query("UPDATE Thread SET unreadNotes = 0 WHERE idx IN (:idxs)")
    void resetUnreadNotes(long... idxs);


    @Query("UPDATE Thread SET unreadNotes = 0 WHERE thread IN (:threads)")
    void resetThreadUnreadNotes(long... threads);

    @Query("Select SUM(unreadNotes) FROM THREAD")
    int getUnreadNotes();

    @Query("SELECT * FROM Thread WHERE thread =:thread")
    List<Thread> getThreadsByThread(long thread);

    @Query("SELECT * FROM Thread WHERE idx =:idx")
    Thread getThreadByIdx(long idx);

    @Query("SELECT * FROM Thread WHERE idx IN(:idxs)")
    List<Thread> getThreadByIdxs(long... idxs);

    @Query("SELECT * FROM Thread WHERE senderPid =:senderPid")
    @TypeConverters({Converter.class})
    List<Thread> getThreadsBySenderPid(PID senderPid);

    @Query("SELECT * FROM Thread WHERE thread =:thread")
    LiveData<List<Thread>> getLiveDataThreadsByThread(long thread);

    @Query("UPDATE Thread SET unreadNotes = unreadNotes + 1  WHERE idx IN(:idxs)")
    void incrementUnreadNotesNumber(long... idxs);

    @Query("UPDATE Thread SET cid =:cid  WHERE idx = :idx")
    @TypeConverters({Converter.class})
    void setCid(long idx, CID cid);

    @Query("UPDATE Thread SET mimeType =:mimeType  WHERE idx = :idx")
    void setMimeType(long idx, String mimeType);

    @Query("UPDATE Thread SET bundle = :bundle WHERE idx = :idx")
    void setBundle(long idx, String bundle);

    @Query("UPDATE Thread SET hash = :hash WHERE idx = :idx")
    void setHash(long idx, String hash);

    @Query("UPDATE Thread SET image = :image WHERE idx = :idx")
    @TypeConverters({Converter.class})
    void setImage(long idx, CID image);

    @Query("SELECT status FROM Thread WHERE idx = :idx")
    @TypeConverters({ThreadStatus.class})
    ThreadStatus getThreadStatus(long idx);

    @Query("SELECT markedFlag FROM Thread WHERE idx = :idx")
    boolean getMarkedFlag(long idx);

    @Query("UPDATE Thread SET markedFlag = :flag WHERE idx = :idx")
    void setMarkedFlag(long idx, boolean flag);

    @Query("UPDATE Thread SET unreadNotes = :unreadNotes  WHERE idx = :idx")
    void setUnreadNotesNumber(long idx, int unreadNotes);
}
