package threads.core;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import threads.core.api.Addresses;
import threads.core.api.Content;
import threads.core.api.MultiAddresses;
import threads.core.api.Peer;
import threads.iota.IOTA;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;
import threads.ipfs.api.PeerInfo;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class IdentityService {

    private static final String TAG = IdentityService.class.getSimpleName();
    private static final String PROTOCOL = "/ipfs/";

    public static void identity(@NonNull Context context,
                                @NonNull String aesKey,
                                boolean addIdentity,
                                int timeout,
                                int numPeers,
                                boolean protectRelays) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkArgument(numPeers >= 0);
        ExecutorService executor = Executors.newSingleThreadExecutor();
        executor.submit(() -> {
            try {
                IdentityService.publishIdentity(
                        context, aesKey, addIdentity,
                        timeout, numPeers, protectRelays);
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        });
    }

    public static boolean publishIdentity(@NonNull Context context,
                                          @NonNull String aesKey,
                                          boolean addIdentity,
                                          int timeout,
                                          int numPeers,
                                          boolean protectPeers) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkArgument(numPeers >= 0);
        if (!Network.isConnected(context)) {
            return false;
        }


        final THREADS threads = Singleton.getInstance(context).getThreads();

        try {

            PID host = Preferences.getPID(context);
            if (host != null) {


                PeerInfo info = null;

                if (addIdentity) {
                    IPFS ipfs = Singleton.getInstance(context).getIpfs();
                    if (ipfs != null) {
                        info = ipfs.id();
                    }
                }

                threads.core.api.PeerInfo peer = threads.getPeerInfoByPID(host);
                if (peer != null) {
                    return updatePeerInfo(context, peer, info, aesKey,
                            timeout, numPeers, protectPeers);

                } else {
                    return createPeerInfo(context, info, host, aesKey,
                            timeout, numPeers, protectPeers);
                }
            }

        } catch (Throwable e) {
            Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
        }

        return false;
    }

    private static List<String> getMultiAddresses(@NonNull PeerInfo info) {
        checkNotNull(info);
        List<String> addresses = new ArrayList<>();
        for (String address : info.getMultiAddresses()) {
            if (!address.startsWith("/ip6/::1/") && !address.startsWith("/ip4/127.0.0.1/")) {
                String ending = PROTOCOL + info.getPID().getPid();
                if (address.endsWith(ending)) {
                    String smallAddress = address.substring(0, address.length() - ending.length());
                    addresses.add(smallAddress);
                }

            }
        }
        return addresses;
    }

    private static boolean createPeerInfo(@NonNull Context context,
                                          @Nullable PeerInfo info,
                                          @NonNull PID user,
                                          @NonNull String aesKey,
                                          int timeout,
                                          int numPeers,
                                          boolean protectRelays) {

        final THREADS threads = Singleton.getInstance(context).getThreads();

        threads.core.api.PeerInfo peerInfo = threads.createPeerInfo(user);


        List<Peer> relayPeers = GatewayService.getPeers(
                context, numPeers, timeout, protectRelays);
        for (Peer relay : relayPeers) {
            peerInfo.addAddress(relay.getPid(), relay.getMultiAddress());
        }

        String alias = threads.getUserAlias(user);
        peerInfo.addAdditional(Content.ALIAS, alias, false);

        if (info != null) {
            List<String> multiAddresses = getMultiAddresses(info);
            for (String address : multiAddresses) {
                peerInfo.addMultiAddresses(address);
            }
        }
        threads.storePeerInfo(peerInfo);

        return insertPeer(context, peerInfo, aesKey);


    }

    private static boolean insertPeer(@NonNull Context context,
                                      @NonNull threads.core.api.PeerInfo peer,
                                      @NonNull String aesKey) {

        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        checkNotNull(iota);
        threads.setHash(peer, null);

        long start = System.currentTimeMillis();

        boolean success = threads.insertPeerInfo(iota, peer, aesKey);

        long time = (System.currentTimeMillis() - start) / 1000;
        if (success) {
            Singleton.getInstance(context).getConsoleListener().info(
                    "Success store peer : " + time + " [s]");
        } else {
            Singleton.getInstance(context).getConsoleListener().error(
                    "Failed store peer : " + time + " [s]");
        }
        return success;
    }

    public static threads.core.api.PeerInfo getPeerInfo(@NonNull Context context,
                                                        @NonNull PID user,
                                                        @NonNull String aesKey) {
        checkNotNull(context);
        checkNotNull(user);
        checkNotNull(aesKey);
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        return threads.getPeer(iota, user, aesKey);
    }

    public static boolean connectPeer(@NonNull Context context,
                                      @NonNull PID peer,
                                      @NonNull String aesKey,
                                      @NonNull String protocol,
                                      boolean supportDiscovery,
                                      boolean protectRelay,
                                      boolean protectPeer) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkNotNull(peer);
        checkNotNull(protocol);
        final int timeout = Preferences.getConnectionTimeout(context);
        return connectPeer(context, peer, aesKey, protocol,
                timeout, supportDiscovery, protectRelay, protectPeer);
    }

    public static boolean connectPeer(@NonNull Context context,
                                      @NonNull PID user,
                                      @NonNull String aesKey,
                                      @NonNull String protocol,
                                      int timeout,
                                      boolean supportDiscovery,
                                      boolean protectRelay,
                                      boolean protectPeer) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkNotNull(user);
        checkNotNull(protocol);
        if (!Network.isConnected(context)) {
            return false;
        }


        if (supportDiscovery) {
            threads.core.api.PeerInfo peer = IdentityService.getPeerInfo(context, user, aesKey);
            if (peer != null) {

                if (connectPeer(context, peer, protocol, timeout, protectRelay, protectPeer)) {
                    return true;
                }

            }

        }
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            boolean connected = ipfs.swarmConnect(user, timeout);
            if (connected) {
                if (protectPeer) {
                    ipfs.protectPeer(user, GatewayService.TAG);
                }
            }
            return ipfs.isConnected(user);

        }
        return false;
    }

    public static boolean connectPeer(@NonNull Context context,
                                      @NonNull threads.core.api.PeerInfo peer,
                                      @NonNull String protocol,
                                      int timeout,
                                      boolean protectRelay,
                                      boolean protectPeer) {
        checkNotNull(context);
        checkNotNull(peer);

        final Singleton.ConsoleListener consoleListener =
                Singleton.getInstance(context).getConsoleListener();

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            Addresses addresses = peer.getAddresses();
            for (String relay : addresses.keySet()) {
                try {
                    String ma = addresses.get(relay);
                    checkNotNull(ma);

                    boolean relayConnected = ipfs.isConnected(PID.create(relay));
                    if (!relayConnected) {
                        relayConnected = ipfs.swarmConnect(
                                ma + "/" + IPFS.Style.ipfs.name() + "/" + relay,
                                timeout);
                    }
                    if (protectRelay && relayConnected) {
                        // be pessimistic and assume no relay and autonat
                        // anyway when it is, it they are the will be evaluated later
                        GatewayService.storePeer(context,
                                PID.create(relay), ma, false, false, 0);
                        ipfs.protectPeer(PID.create(relay), GatewayService.TAG);
                    }

                    if (relayConnected) {
                        ma = ma.concat("/" + protocol + "/" + relay +
                                "/p2p-circuit/" + protocol + "/" +
                                peer.getPID().getPid());
                        boolean connect = ipfs.swarmConnect(ma, timeout);
                        if (connect) {
                            if (protectPeer) {
                                ipfs.protectPeer(peer.getPID(), GatewayService.TAG);
                            }
                        }
                    }

                } catch (Throwable e) {
                    Log.e(TAG, e.getLocalizedMessage(), e);
                }
            }


            MultiAddresses multiAddresses = peer.getMultiAddresses();
            for (String address : multiAddresses) {

                try {
                    String ma = address.concat(PROTOCOL + peer.getPID().getPid());
                    consoleListener.debug("Try connect : " + ma);


                    if (ipfs.isConnected(peer.getPID())) {
                        if (protectPeer) {
                            ipfs.protectPeer(peer.getPID(), GatewayService.TAG);
                        }
                        break;
                    } else {
                        boolean connect = ipfs.swarmConnect(ma, timeout);

                        if (connect) {

                            if (protectPeer) {
                                ipfs.protectPeer(peer.getPID(), GatewayService.TAG);
                            }

                            consoleListener.info("Success connect : " + ma);
                            break;
                        }
                    }
                } catch (Throwable e) {
                    Log.e(TAG, e.getLocalizedMessage(), e);
                }
            }


            return ipfs.isConnected(peer.getPID());
        }
        return false;
    }


    private static boolean updatePeerInfo(
            @NonNull Context context,
            @NonNull threads.core.api.PeerInfo peer,
            @Nullable PeerInfo info,
            @NonNull String aesKey,
            int timeout,
            int numPeers,
            boolean protectPeers) {

        final THREADS threads = Singleton.getInstance(context).getThreads();

        peer.removeAddresses();

        List<Peer> peers = GatewayService.getPeers(context, numPeers, timeout, protectPeers);
        for (Peer relay : peers) {
            peer.addAddress(relay.getPid(), relay.getMultiAddress());
        }

        peer.removeMultiAddresses();

        if (info != null) {
            List<String> multiAddresses = getMultiAddresses(info);
            for (String address : multiAddresses) {
                peer.addMultiAddresses(address);
            }
        }

        threads.updatePeerInfo(peer);
        return insertPeer(context, peer, aesKey);
    }


}
