package threads.core;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import threads.core.api.Peer;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;
import threads.ipfs.api.PeerInfo;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class GatewayService {

    public static final String TAG = GatewayService.class.getSimpleName();


    public static int evaluatePeers(@NonNull Context context) {
        IPFS ipfs = Singleton.getInstance(context).getIpfs();
        checkNotNull(ipfs);
        List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();
        int size = peers.size();
        for (threads.ipfs.api.Peer peer : peers) {
            if (peer.isAutonat() || peer.isRelay()) {
                storePeer(context, peer);
            }
        }
        return size;
    }


    public synchronized static List<threads.core.api.Peer> getPeers(
            @NonNull Context context,
            int unmRelays,
            int timeout,
            boolean protect) {

        checkNotNull(context);
        checkArgument(unmRelays >= 0);
        checkArgument(timeout > 1000);

        List<threads.core.api.Peer> result = new ArrayList<>();


        if (!Network.isConnected(context)) {
            return result;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();



        if (ipfs != null) {

            List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();

            peers.sort(threads.ipfs.api.Peer::compareTo);

            for (threads.ipfs.api.Peer peer : peers) {

                if (result.size() == unmRelays) {
                    break;
                }

                if (peer.isRelay()) {

                    if (ipfs.isConnected(peer.getPid())) {

                        if (protect) {
                            ipfs.protectPeer(peer.getPid(), TAG);
                        }

                        result.add(storePeer(context, peer));


                    } else if (ipfs.swarmConnect(peer, timeout)) {

                        if (protect) {
                            ipfs.protectPeer(peer.getPid(), TAG);
                        }

                        result.add(storePeer(context, peer));

                    }

                }
            }

        }

        return result;
    }

    private static Peer storePeer(@NonNull Context context, @NonNull threads.ipfs.api.Peer peer) {

        // the given peer is connected (so rating will be dependent of peer
        int rating = 0;
        try {
            double latency = peer.getLatency();
            if (latency < 1000) {
                rating = (int) (1000 - latency);
            }

        } catch (Throwable e) {
            // ignore any exceptions here
        }

        // now add higher rating when peer has specific attributes
        try {
            int timeout = Preferences.getMaxThreshold(context);
            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            if (ipfs != null) {
                PeerInfo info = ipfs.id(peer, timeout);
                if (info != null) {
                    Log.e(TAG, "Info : " + info.toString());
                    String protocol = info.getProtocolVersion();
                    String agent = info.getAgentVersion();

                    if (protocol != null && protocol.equals("ipfs/0.1.0")) {
                        rating = rating + 100;
                    } else {
                        rating = rating - 100;
                    }
                    if (agent != null && agent.startsWith("go-ipfs/0.4.2")) {
                        rating = rating + 100;
                    }
                }
            }
        } catch (Throwable e) {
            // ignore any exceptions here
        }
        if (rating < 0) {
            rating = 0;
        }
        return storePeer(context, peer.getPid(),
                peer.getMultiAddress(), peer.isRelay(), peer.isAutonat(), rating);
    }


    public static threads.core.api.Peer storePeer(@NonNull Context context,
                                                  @NonNull PID pid,
                                                  @NonNull String multiAddress,
                                                  boolean isRelay,
                                                  boolean isAutonat,
                                                  int rating) {

        final THREADS threads = Singleton.getInstance(context).getThreads();

        threads.core.api.Peer relay = threads.getPeerByPID(pid);
        if (relay != null) {
            relay.setMultiAddress(multiAddress);
            relay.setRelay(isRelay);
            relay.setAutonat(isAutonat);
            relay.setRating(rating);
            threads.updatePeer(relay);
        } else {
            relay = threads.createPeer(pid, multiAddress);
            relay.setRelay(isRelay);
            relay.setAutonat(isAutonat);
            relay.setRating(rating);
            threads.storePeer(relay);
        }
        return relay;
    }

    public static void connectStoredAutonat(@NonNull Context context,
                                            int numConnections,
                                            int timeout,
                                            boolean cleanStoredPeers) {
        checkNotNull(context);
        checkArgument(numConnections >= 0);
        checkArgument(timeout > 1000);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {

            List<threads.core.api.Peer> peers = threads.getAutonatPeers();

            peers.sort(Peer::compareTo);

            for (threads.core.api.Peer autonat : peers) {

                if (counter.get() == numConnections) {
                    break;
                }

                Log.e(TAG, "Stored Autonat : " + autonat.toString());

                if (ipfs.isConnected(PID.create(autonat.getPid()))) {
                    counter.incrementAndGet();
                } else {

                    String ma = autonat.getMultiAddress() + "/" +
                            IPFS.Style.ipfs.name() + "/" + autonat.getPid();

                    if (ipfs.swarmConnect(ma, timeout)) {
                        counter.incrementAndGet();

                    } else {
                        if (cleanStoredPeers) {
                            threads.removePeer(autonat);
                        } else {
                            autonat.setRating(0);
                            threads.updatePeer(autonat);
                        }
                    }
                }
            }
        }
    }

    public static void connectStoredRelays(@NonNull Context context,
                                           int numConnections,
                                           int timeout,
                                           boolean cleanStoredPeers) {
        checkNotNull(context);
        checkArgument(numConnections >= 0);
        checkArgument(timeout > 1000);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {

            List<threads.core.api.Peer> peers = threads.getRelayPeers();

            peers.sort(Peer::compareTo);

            for (threads.core.api.Peer relay : peers) {

                if (counter.get() == numConnections) {
                    break;
                }

                Log.e(TAG, "Stored Relay : " + relay.toString());

                if (ipfs.isConnected(PID.create(relay.getPid()))) {
                    counter.incrementAndGet();
                } else {

                    String ma = relay.getMultiAddress() + "/" +
                            IPFS.Style.ipfs.name() + "/" + relay.getPid();

                    if (ipfs.swarmConnect(ma, timeout)) {
                        counter.incrementAndGet();

                    } else {
                        if (cleanStoredPeers) {
                            threads.removePeer(relay);
                        } else {
                            relay.setRating(0);
                            threads.updatePeer(relay);
                        }
                    }
                }
            }
        }
    }
}
