package threads.core.api;


import androidx.annotation.NonNull;
import androidx.room.ColumnInfo;
import androidx.room.PrimaryKey;

import java.util.Comparator;
import java.util.Objects;

import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkNotNull;

@androidx.room.Entity
public class Peer extends Entity implements Comparable<Peer> {

    public static Comparator<Peer> PeerComparator = (peer1, peer2) -> peer1.compareTo(peer2);
    @PrimaryKey
    @NonNull
    @ColumnInfo(name = "pid")
    private final String pid;
    @NonNull
    @ColumnInfo(name = "multiAddress")
    private String multiAddress;
    @ColumnInfo(name = "isRelay")
    private boolean isRelay;
    @ColumnInfo(name = "isAutonat")
    private boolean isAutonat;
    @ColumnInfo(name = "rating")
    private int rating;

    Peer(@NonNull String pid, @NonNull String multiAddress) {
        this.pid = pid;
        this.multiAddress = multiAddress;
        this.isRelay = false;
        this.isAutonat = false;
        this.rating = 0;
    }

    public boolean isAutonat() {
        return isAutonat;
    }

    public void setAutonat(boolean autonat) {
        isAutonat = autonat;
    }

    public static Peer createPeer(@NonNull PID pid, @NonNull String multiAddress) {
        checkNotNull(pid);
        checkNotNull(multiAddress);
        return new Peer(pid.getPid(), multiAddress);
    }

    public int getRating() {
        return rating;
    }

    public void setRating(int rating) {
        this.rating = rating;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Peer peer = (Peer) o;
        return pid.equals(peer.pid);
    }

    @Override
    public String toString() {
        return "Peer{" +
                "pid='" + pid + '\'' +
                ", multiAddress='" + multiAddress + '\'' +
                ", isRelay=" + isRelay +
                ", isAutonat=" + isAutonat +
                ", rating=" + rating +
                '}';
    }

    @Override
    public int hashCode() {
        return Objects.hash(pid);
    }

    @NonNull
    public String getPid() {
        return pid;
    }

    public boolean isRelay() {
        return isRelay;
    }

    public void setRelay(boolean relay) {
        isRelay = relay;
    }

    @NonNull
    public String getMultiAddress() {
        return multiAddress;
    }

    public void setMultiAddress(@NonNull String multiAddress) {
        checkNotNull(multiAddress);
        this.multiAddress = multiAddress;
    }

    @Override
    public int compareTo(@NonNull Peer peer) {
        return Double.compare(peer.rating, this.rating);
    }
}
