package threads.core;

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import threads.core.api.Addresses;
import threads.core.api.Content;
import threads.core.api.MultiAddresses;
import threads.core.api.Peer;
import threads.iota.IOTA;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;
import threads.ipfs.api.PeerInfo;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class IdentityService {

    private static final String TAG = IdentityService.class.getSimpleName();
    private static final String PROTOCOL = "/ipfs/";
    private static final String SUPPORT_PEER_DISCOVERY_KEY = "supportPeerDiscoveryKey";

    public static boolean isSupportPeerDiscovery(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(TAG, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(SUPPORT_PEER_DISCOVERY_KEY, true);
    }

    public static void setSupportPeerDiscovery(@NonNull Context context, boolean enable) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(TAG, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(SUPPORT_PEER_DISCOVERY_KEY, enable);
        editor.apply();
    }

    public static void identity(@NonNull Context context,
                                @NonNull String aesKey,
                                int numPeers,
                                boolean protectRelays,
                                boolean cleanStoredPeers) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkArgument(numPeers >= 0);
        ExecutorService executor = Executors.newSingleThreadExecutor();
        executor.submit(() -> {
            try {
                IdentityService.publishIdentity(
                        context, aesKey, numPeers, protectRelays, cleanStoredPeers);
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        });
    }

    public static boolean publishIdentity(@NonNull Context context,
                                          @NonNull String aesKey,
                                          int numPeers,
                                          boolean protectPeers,
                                          boolean cleanStoredPeers) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkArgument(numPeers >= 0);
        if (!Network.isConnected(context)) {
            return false;
        }

        if (!IdentityService.isSupportPeerDiscovery(context)) {
            return false;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        if (ipfs != null) {
            try {

                PID host = Preferences.getPID(context);
                if (host != null) {


                    PeerInfo info = ipfs.id();
                    checkNotNull(info);
                    List<String> addresses = getMultiAddresses(info);

                    threads.core.api.PeerInfo peer = threads.getPeerInfoByPID(host);
                    if (peer != null) {

                        boolean update = !peer.hasHash();
                        if (!update) {
                            MultiAddresses multiAddresses = peer.getMultiAddresses();
                            for (String address : addresses) {
                                if (!multiAddresses.contains(address)) {
                                    update = true;
                                    break;
                                }
                            }
                        }
                        if (update) {
                            return updatePeer(context, peer, aesKey,
                                    addresses, numPeers, protectPeers, cleanStoredPeers);
                        }

                    } else {
                        return createPeer(context, host, aesKey,
                                addresses, numPeers, protectPeers, cleanStoredPeers);
                    }
                }

            } catch (Throwable e) {
                Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
            }
        }
        return false;
    }

    private static List<String> getMultiAddresses(@NonNull PeerInfo info) {
        checkNotNull(info);
        List<String> addresses = new ArrayList<>();
        for (String address : info.getMultiAddresses()) {
            if (!address.startsWith("/ip6/::1/") && !address.startsWith("/ip4/127.0.0.1/")) {
                String ending = PROTOCOL + info.getPID().getPid();
                if (address.endsWith(ending)) {
                    String smallAddress = address.substring(0, address.length() - ending.length());
                    addresses.add(smallAddress);
                }

            }
        }
        return addresses;
    }

    private static boolean createPeer(@NonNull Context context,
                                      @NonNull PID user,
                                      @NonNull String aesKey,
                                      @NonNull List<String> addresses,
                                      int numPeers,
                                      boolean protectRelays,
                                      boolean cleanStoredPeers) {

        final THREADS threads = Singleton.getInstance(context).getThreads();

        threads.core.api.PeerInfo peer = threads.createPeerInfo(user);

        int timeout = Preferences.getMaxThreshold(context);
        List<Peer> relayPeers = GatewayService.getPeers(
                context, numPeers, timeout, protectRelays, cleanStoredPeers);
        for (Peer relay : relayPeers) {
            peer.addAddress(relay.getPid(), relay.getMultiAddress());
        }

        String alias = threads.getUserAlias(user);
        peer.addAdditional(Content.ALIAS, alias, false);

        for (String address : addresses) {
            peer.addMultiAddresses(address);
        }
        threads.storePeerInfo(peer);

        return insertPeer(context, peer, aesKey);


    }

    public static threads.core.api.PeerInfo getPeer(@NonNull Context context,
                                                    @NonNull PID user,
                                                    @NonNull String aesKey) {
        checkNotNull(context);
        checkNotNull(user);
        checkNotNull(aesKey);
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        if (IdentityService.isSupportPeerDiscovery(context)) {
            return threads.getPeer(iota, user, aesKey);
        }
        return null;
    }

    private static boolean insertPeer(@NonNull Context context,
                                      @NonNull threads.core.api.PeerInfo peer,
                                      @NonNull String aesKey) {

        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        checkNotNull(iota);
        threads.setHash(peer, null);

        long start = System.currentTimeMillis();

        boolean success = threads.insertPeerInfo(iota, peer, aesKey);

        long time = (System.currentTimeMillis() - start) / 1000;
        if (success) {
            Singleton.getInstance(context).getConsoleListener().info(
                    "Success store peer : " + time + " [s]");
        } else {
            Singleton.getInstance(context).getConsoleListener().error(
                    "Failed store peer : " + time + " [s]");
        }
        return success;
    }

    public static threads.core.api.PeerInfo getPeerInfo(@NonNull Context context,
                                                        @NonNull PID user,
                                                        @NonNull String aesKey) {
        checkNotNull(context);
        checkNotNull(user);
        checkNotNull(aesKey);
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IOTA iota = Singleton.getInstance(context).getIota();
        if (isSupportPeerDiscovery(context)) {
            return threads.getPeer(iota, user, aesKey);
        }
        return null;
    }

    public static boolean connectPeer(@NonNull Context context,
                                      @NonNull PID peer,
                                      @NonNull String aesKey,
                                      boolean protectRelay,
                                      boolean protectPeer) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkNotNull(peer);
        final int timeout = Preferences.getConnectionTimeout(context);
        return connectPeer(context, peer, aesKey, timeout, protectRelay, protectPeer);
    }

    public static boolean connectPeer(@NonNull Context context,
                                      @NonNull PID user,
                                      @NonNull String aesKey,
                                      int timeout,
                                      boolean protectRelay,
                                      boolean protectPeer) {
        checkNotNull(context);
        checkNotNull(aesKey);
        checkNotNull(user);

        if (!Network.isConnected(context)) {
            return false;
        }


        final IPFS ipfs = Singleton.getInstance(context).getIpfs();

        if (ipfs != null) {

            threads.core.api.PeerInfo peer = IdentityService.getPeer(context, user, aesKey);
            if (peer != null) {

                if (connectPeer(context, ipfs, peer, protectRelay, protectPeer)) {
                    return true;
                }

            }
            boolean connected = ipfs.swarmConnect(user, timeout);
            if (connected) {
                if (protectPeer) {
                    ipfs.protectPeer(user, GatewayService.TAG);
                }
            }
            return ipfs.isConnected(user);

        }
        return false;
    }

    private static boolean connectPeer(@NonNull Context context,
                                       @NonNull IPFS ipfs,
                                       @NonNull threads.core.api.PeerInfo peer,
                                       boolean protectRelay,
                                       boolean protectPeer) {
        checkNotNull(context);
        checkNotNull(peer);

        final int timeout = Preferences.getMaxThreshold(context);
        final boolean dialRelay = Preferences.isDialRelay(context);

        final Singleton.ConsoleListener consoleListener =
                Singleton.getInstance(context).getConsoleListener();


        Addresses addresses = peer.getAddresses();
        for (String relay : addresses.keySet()) {
            try {
                String ma = addresses.get(relay);
                checkNotNull(ma);

                boolean relayConnected = ipfs.isConnected(PID.create(relay));
                if (!relayConnected) {
                    relayConnected = ipfs.swarmConnect(
                            ma + "/" + IPFS.Style.ipfs.name() + "/" + relay,
                            timeout);
                }
                if (protectRelay && relayConnected) {
                    GatewayService.storePeer(context,
                            PID.create(relay), ma, true, true);
                    ipfs.protectPeer(PID.create(relay), GatewayService.TAG);
                }

                if (dialRelay && relayConnected) {
                    ma = ma.concat("/" + IPFS.Style.ipfs.name() + "/" + relay +
                            "/p2p-circuit/" + IPFS.Style.ipfs.name() + "/" +
                            peer.getPID().getPid());
                    boolean connect = ipfs.swarmConnect(ma, timeout);
                    if (connect) {
                        if (protectPeer) {
                            ipfs.protectPeer(peer.getPID(), GatewayService.TAG);
                        }
                    }
                }

            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
            }
        }


        MultiAddresses multiAddresses = peer.getMultiAddresses();
        for (String address : multiAddresses) {

            try {
                String ma = address.concat(PROTOCOL + peer.getPID().getPid());
                consoleListener.debug("Try connect : " + ma);


                if (ipfs.isConnected(peer.getPID())) {
                    if (protectPeer) {
                        ipfs.protectPeer(peer.getPID(), GatewayService.TAG);
                    }
                    break;
                } else {
                    boolean connect = ipfs.swarmConnect(ma, timeout);

                    if (connect) {

                        if (protectPeer) {
                            ipfs.protectPeer(peer.getPID(), GatewayService.TAG);
                        }

                        consoleListener.info("Success connect : " + ma);
                        break;
                    }
                }
            } catch (Throwable e) {
                Log.e(TAG, e.getLocalizedMessage(), e);
            }
        }


        return ipfs.isConnected(peer.getPID());
    }


    private static boolean updatePeer(@NonNull Context context,
                                      @NonNull threads.core.api.PeerInfo peer,
                                      @NonNull String aesKey,
                                      @NonNull List<String> multiAddresses,
                                      int numPeers,
                                      boolean protectPeers,
                                      boolean cleanStoredPeers) {

        final THREADS threads = Singleton.getInstance(context).getThreads();

        peer.removeAddresses();

        int timeout = Preferences.getMaxThreshold(context);
        List<Peer> relayPeers = GatewayService.getPeers(
                context, numPeers, timeout, protectPeers, cleanStoredPeers);
        for (Peer relay : relayPeers) {
            peer.addAddress(relay.getPid(), relay.getMultiAddress());
        }

        peer.removeMultiAddresses();

        for (String address : multiAddresses) {
            peer.addMultiAddresses(address);
        }


        threads.updatePeerInfo(peer);
        return insertPeer(context, peer, aesKey);
    }


}
