package threads.core;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import threads.ipfs.IPFS;
import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class GatewayService {

    public static final String TAG = GatewayService.class.getSimpleName();

    public static Comparator<threads.ipfs.api.Peer> PeerComparator = new Comparator<threads.ipfs.api.Peer>() {

        public int compare(threads.ipfs.api.Peer peer1, threads.ipfs.api.Peer peer2) {

            return peer1.compareTo(peer2);
        }

    };

    public static List<threads.core.api.Peer> getPeers(@NonNull Context context,
                                                       int unmRelays,
                                                       int timeout,
                                                       boolean protect,
                                                       boolean cleanStoredPeers) {
        checkNotNull(context);
        checkArgument(unmRelays >= 0);
        checkArgument(timeout > 1000);

        List<threads.core.api.Peer> result = new ArrayList<>();


        if (!Network.isConnected(context)) {
            return result;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();

        evaluateStoredPeers(context, unmRelays, timeout, cleanStoredPeers);


        if (ipfs != null) {

            List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();

            peers.sort(PeerComparator);

            for (threads.ipfs.api.Peer peer : peers) {

                if (result.size() == unmRelays) {
                    break;
                }

                if (peer.isRelay()) {

                    if (ipfs.isConnected(peer.getPid())) {

                        if (protect) {
                            ipfs.protectPeer(peer.getPid(), TAG);
                        }

                        result.add(storePeer(context, peer.getPid(),
                                peer.getMultiAddress(), true, true));


                    } else if (ipfs.swarmConnect(peer, timeout)) {

                        if (protect) {
                            ipfs.protectPeer(peer.getPid(), TAG);
                        }

                        result.add(storePeer(context, peer.getPid(),
                                peer.getMultiAddress(), true, true));

                    }

                }
            }

        }

        return result;
    }


    @Nullable
    public static threads.core.api.Peer getPeer(@NonNull Context context,
                                                @NonNull List<PID> ignore,
                                                int numPeers,
                                                int timeout,
                                                boolean protect,
                                                boolean cleanStoredPeers) {
        checkNotNull(context);
        checkNotNull(ignore);
        checkArgument(numPeers >= 0);
        checkArgument(timeout > 1000);

        if (!Network.isConnected(context)) {
            return null;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();

        evaluateStoredPeers(context, numPeers, timeout, cleanStoredPeers);


        if (ipfs != null) {

            List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();

            peers.sort(PeerComparator);

            for (threads.ipfs.api.Peer peer : peers) {

                // ignore list
                if (ignore.contains(peer.getPid())) {
                    continue;
                }

                if (peer.isRelay()) {


                    if (ipfs.isConnected(peer.getPid())) {

                        if (protect) {
                            ipfs.protectPeer(peer.getPid(), TAG);
                        }

                        return storePeer(context, peer.getPid(),
                                peer.getMultiAddress(), true, true);


                    } else if (ipfs.swarmConnect(peer, timeout)) {

                        if (protect) {
                            ipfs.protectPeer(peer.getPid(), TAG);
                        }

                        return storePeer(context, peer.getPid(),
                                peer.getMultiAddress(), true, true);

                    }

                }
            }
        }

        return null;
    }

    public static threads.core.api.Peer storePeer(@NonNull Context context,
                                                  @NonNull PID pid,
                                                  @NonNull String multiAddress,
                                                  boolean isRelay,
                                                  boolean connect) {


        final THREADS threads = Singleton.getInstance(context).getThreads();

        threads.core.api.Peer relay = threads.getPeerByPID(pid);
        if (relay != null) {
            relay.setMultiAddress(multiAddress);
            relay.setRelay(isRelay);
            relay.setConnected(connect);
            threads.updatePeer(relay);
        } else {
            relay = threads.createPeer(pid, multiAddress);
            relay.setRelay(isRelay);
            relay.setConnected(connect);
            threads.storePeer(relay);
        }
        return relay;
    }

    private static void evaluateStoredPeers(@NonNull Context context,
                                            int numConnections,
                                            int timeout,
                                            boolean cleanStoredPeers) {
        checkNotNull(context);
        checkArgument(numConnections >= 0);
        checkArgument(timeout > 1000);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {
            List<threads.core.api.Peer> relays = threads.getRelayPeers();
            for (threads.core.api.Peer relay : relays) {

                if (counter.get() == numConnections) {
                    break;
                }

                Log.e(TAG, "Stored Relay : " + relay.toString());

                if (ipfs.isConnected(PID.create(relay.getPid()))) {
                    relay.setConnected(true);
                    threads.updatePeer(relay);
                    counter.incrementAndGet();
                } else {

                    String ma = relay.getMultiAddress() + "/" +
                            IPFS.Style.ipfs.name() + "/" + relay.getPid();

                    if (ipfs.swarmConnect(ma, timeout)) {

                        relay.setConnected(true);
                        threads.updatePeer(relay);
                        counter.incrementAndGet();

                    } else {
                        if (cleanStoredPeers) {
                            threads.removePeer(relay);
                        }
                    }
                }
            }
        }
    }
}
