package threads.core;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.Comparator;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import threads.ipfs.IPFS;
import threads.ipfs.api.PID;
import threads.ipfs.api.PeerInfo;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class GatewayService {

    private static final String TAG = GatewayService.class.getSimpleName();
    public static Comparator<threads.ipfs.api.Peer> PeerComparator = new Comparator<threads.ipfs.api.Peer>() {

        public int compare(threads.ipfs.api.Peer peer1, threads.ipfs.api.Peer peer2) {

            return peer1.compareTo(peer2);
        }

    };

    @Nullable
    public static threads.core.api.Peer getPeer(@NonNull Context context,
                                                @NonNull List<PID> ignore,
                                                int numConnections,
                                                int timeout) {
        checkNotNull(context);
        checkNotNull(ignore);
        checkArgument(numConnections > 0);
        checkArgument(timeout > 1000);

        if (!Network.isConnected(context)) {
            return null;
        }

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();

        evaluateStoredRelays(context, numConnections, timeout);


        if (ipfs != null) {

            List<threads.ipfs.api.Peer> peers = ipfs.swarmPeers();

            peers.sort(PeerComparator);

            for (threads.ipfs.api.Peer peer : peers) {

                // ignore list
                if (ignore.contains(peer.getPid())) {
                    continue;
                }

                if (peer.isRelay()) {


                    if (ipfs.isConnected(peer.getPid())) {

                        // TODO remove

                        PeerInfo info = ipfs.id(peer, timeout);
                        if (info != null) {
                            Log.e(TAG, info.getAgentVersion() + "  " + info.getProtocolVersion());
                        }

                        // TODO END remove

                        return storePeer(context, peer.getPid(),
                                peer.getMultiAddress(), true, true);


                    } else if (ipfs.swarmConnect(peer, timeout)) {

                        // TODO protect the connection

                        return storePeer(context, peer.getPid(),
                                peer.getMultiAddress(), true, true);

                    }

                }
            }


            // now just return the fastest one (should not occeur very often)
            if (!peers.isEmpty()) {
                threads.ipfs.api.Peer peer = peers.get(0);
                return storePeer(context, peer.getPid(),
                        peer.getMultiAddress(), peer.isRelay(),
                        ipfs.isConnected(peer.getPid()));
            }
        }

        return null;
    }

    public static threads.core.api.Peer storePeer(@NonNull Context context,
                                                  @NonNull PID pid,
                                                  @NonNull String multiAddress,
                                                  boolean isRelay,
                                                  boolean connect) {


        final THREADS threads = Singleton.getInstance(context).getThreads();

        threads.core.api.Peer relay = threads.getPeerByPID(pid);
        if (relay != null) {
            relay.setMultiAddress(multiAddress);
            relay.setRelay(isRelay);
            relay.setConnected(connect);
            threads.updatePeer(relay);
        } else {
            relay = threads.createPeer(pid, multiAddress);
            relay.setRelay(isRelay);
            relay.setConnected(connect);
            threads.storePeer(relay);
        }
        return relay;
    }

    private static void evaluateStoredRelays(@NonNull Context context,
                                             int numConnections,
                                             int timeout) {
        checkNotNull(context);
        checkArgument(numConnections > 0);
        checkArgument(timeout > 1000);

        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        final THREADS threads = Singleton.getInstance(context).getThreads();
        final AtomicInteger counter = new AtomicInteger(0);

        if (ipfs != null) {
            List<threads.core.api.Peer> relays = threads.getRelayPeers();
            for (threads.core.api.Peer relay : relays) {

                if (counter.get() == numConnections) {
                    break;
                }

                // TODO what if already connected
                // TODO protect the peer

                Log.e(TAG, "Stored Relay : " + relay.toString());
                String ma = relay.getMultiAddress() + "/" + IPFS.Style.ipfs.name() + "/" + relay.getPid();

                if (ipfs.swarmConnect(ma, timeout)) {

                    // TODO Protect the connection

                    relay.setConnected(true);
                    threads.updatePeer(relay);

                    counter.incrementAndGet();
                } else {
                    threads.removePeer(relay);
                }
            }
        }
    }
}
