/*
 * Decompiled with CFR 0.152.
 */
package threads.core;

import android.content.Context;
import android.content.SharedPreferences;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.util.Preconditions;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import threads.core.Network;
import threads.core.Preferences;
import threads.core.Singleton;
import threads.core.THREADS;
import threads.core.api.Addresses;
import threads.core.api.Peer;
import threads.iota.IOTA;
import threads.ipfs.IPFS;
import threads.ipfs.api.PID;

public class PeerService {
    private static final int MAX_RELAYS = 3;
    private static final String TAG = PeerService.class.getSimpleName();
    private static final String SUPPORT_PEER_DISCOVERY_KEY = "supportPeerDiscoveryKey";

    public static boolean isSupportPeerDiscovery(@NonNull Context context) {
        Preconditions.checkNotNull((Object)context);
        SharedPreferences sharedPref = context.getSharedPreferences(TAG, 0);
        return sharedPref.getBoolean(SUPPORT_PEER_DISCOVERY_KEY, true);
    }

    public static void setSupportPeerDiscovery(@NonNull Context context, boolean enable) {
        Preconditions.checkNotNull((Object)context);
        SharedPreferences sharedPref = context.getSharedPreferences(TAG, 0);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(SUPPORT_PEER_DISCOVERY_KEY, enable);
        editor.apply();
    }

    public static void publishPeer(@NonNull Context context, @NonNull String aesKey) {
        Preconditions.checkNotNull((Object)context);
        Preconditions.checkNotNull((Object)aesKey);
        if (!Network.isConnected(context)) {
            return;
        }
        if (!PeerService.isSupportPeerDiscovery(context)) {
            return;
        }
        THREADS threads = Singleton.getInstance(context).getThreads();
        try {
            PID host = Preferences.getPID(context);
            if (host != null) {
                Peer peer = threads.getPeerByPID(host);
                if (peer != null) {
                    boolean isDirectPeerValid = PeerService.isDirectPeerValid(context, peer);
                    int numRelays = PeerService.updatePeerRelays(context, peer);
                    if (numRelays < 3 || !isDirectPeerValid || !peer.hasHash()) {
                        Addresses addresses = peer.getAddresses();
                        List<threads.ipfs.api.Peer> relays = PeerService.getRelayPeers(context, addresses.keySet());
                        threads.ipfs.api.Peer direct = null;
                        if (!isDirectPeerValid) {
                            direct = PeerService.getDirectPeer(context);
                        }
                        PeerService.updatePeer(context, peer, aesKey, relays, direct);
                    }
                } else {
                    List<threads.ipfs.api.Peer> relays = PeerService.getRelayPeers(context, new HashSet<String>());
                    threads.ipfs.api.Peer direct = PeerService.getDirectPeer(context);
                    PeerService.createPeer(context, host, aesKey, relays, direct);
                }
            }
        }
        catch (Throwable e) {
            Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
        }
    }

    private static boolean isDirectPeerValid(@NonNull Context context, @NonNull Peer peer) {
        String ma;
        Preconditions.checkNotNull((Object)context);
        Preconditions.checkNotNull((Object)peer);
        int thresholdMax = Preferences.getMaxThreshold(context);
        AtomicBoolean peerValid = new AtomicBoolean(false);
        IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null && !(ma = peer.getAdditional("peer")).isEmpty()) {
            String pid = peer.getAdditional("pid");
            if (!ipfs.isConnected(PID.create((String)pid))) {
                boolean connect = ipfs.swarmConnect(ma + "/" + IPFS.Style.ipfs.name() + "/" + pid, thresholdMax);
                if (connect) {
                    peerValid.set(true);
                } else {
                    Singleton.getInstance(context).getConsoleListener().debug("Direct peer not valid anymore : " + pid);
                }
            } else {
                peerValid.set(true);
            }
        }
        return peerValid.get();
    }

    private static int updatePeerRelays(@NonNull Context context, @NonNull Peer peer) {
        Preconditions.checkNotNull((Object)context);
        Preconditions.checkNotNull((Object)peer);
        int thresholdMax = Preferences.getMaxThreshold(context);
        AtomicInteger connected = new AtomicInteger(0);
        AtomicBoolean peerChanged = new AtomicBoolean(false);
        IPFS ipfs = Singleton.getInstance(context).getIpfs();
        Addresses addresses = peer.getAddresses();
        if (ipfs != null) {
            for (String relay : addresses.keySet()) {
                try {
                    String ma = (String)addresses.get(relay);
                    Preconditions.checkNotNull((Object)ma);
                    PID relayPID = PID.create((String)relay);
                    if (!ipfs.isConnected(relayPID)) {
                        boolean connect = ipfs.swarmConnect(ma + "/" + IPFS.Style.ipfs.name() + "/" + relay, thresholdMax);
                        if (connect) {
                            connected.incrementAndGet();
                            continue;
                        }
                        Singleton.getInstance(context).getConsoleListener().debug("Relay not valid anymore : " + relayPID);
                        peer.removeAddress(relay);
                        peerChanged.set(true);
                        continue;
                    }
                    connected.incrementAndGet();
                }
                catch (Throwable e) {
                    Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
                }
            }
        }
        if (peerChanged.get()) {
            Singleton.getInstance(context).getThreads().storePeer(peer);
        }
        return connected.get();
    }

    private static void createPeer(@NonNull Context context, @NonNull PID user, @NonNull String aesKey, @NonNull List<threads.ipfs.api.Peer> relays, @Nullable threads.ipfs.api.Peer direct) {
        THREADS threads = Singleton.getInstance(context).getThreads();
        try {
            Peer peer = threads.createPeer(user);
            String alias = threads.getUserAlias(user);
            peer.addAdditional("alias", alias, false);
            if (direct != null) {
                peer.addAdditional("pid", direct.getPid().getPid(), false);
                peer.addAdditional("peer", direct.getMultiAddress(), false);
            }
            for (threads.ipfs.api.Peer relay : relays) {
                peer.addAddress(relay.getPid(), relay.getMultiAddress());
            }
            threads.storePeer(peer);
            PeerService.insertPeer(context, peer, aesKey);
        }
        catch (Throwable e) {
            Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
        }
    }

    public static Peer getPeer(@NonNull Context context, @NonNull PID user, @NonNull String aesKey) {
        Preconditions.checkNotNull((Object)context);
        Preconditions.checkNotNull((Object)user);
        Preconditions.checkNotNull((Object)aesKey);
        THREADS threads = Singleton.getInstance(context).getThreads();
        IOTA iota = Singleton.getInstance(context).getIota();
        if (PeerService.isSupportPeerDiscovery(context)) {
            return threads.getPeer(iota, user, aesKey);
        }
        return null;
    }

    private static void insertPeer(@NonNull Context context, @NonNull Peer peer, @NonNull String aesKey) {
        THREADS threads = Singleton.getInstance(context).getThreads();
        IOTA iota = Singleton.getInstance(context).getIota();
        Preconditions.checkNotNull((Object)iota);
        threads.setHash(peer, null);
        long start = System.currentTimeMillis();
        boolean success = threads.insertPeer(iota, peer, aesKey);
        long time = (System.currentTimeMillis() - start) / 1000L;
        if (success) {
            Singleton.getInstance(context).getConsoleListener().info("Success store peer : " + time + " [s]");
        } else {
            Singleton.getInstance(context).getConsoleListener().error("Failed store peer : " + time + " [s]");
        }
    }

    private static void updatePeer(@NonNull Context context, @NonNull Peer peer, @NonNull String aesKey, @NonNull List<threads.ipfs.api.Peer> relays, @Nullable threads.ipfs.api.Peer direct) {
        THREADS threads = Singleton.getInstance(context).getThreads();
        boolean update = false;
        try {
            if (direct != null && !direct.getMultiAddress().equals(peer.getAdditional("peer"))) {
                update = true;
                peer.addAdditional("pid", direct.getPid().getPid(), false);
                peer.addAdditional("peer", direct.getMultiAddress(), false);
            }
            Addresses previous = peer.getAddresses();
            for (threads.ipfs.api.Peer relay : relays) {
                peer.addAddress(relay.getPid(), relay.getMultiAddress());
            }
            if (!previous.equals(peer.getAddresses())) {
                update = true;
            }
            if (update) {
                threads.updatePeer(peer);
                PeerService.insertPeer(context, peer, aesKey);
            }
        }
        catch (Throwable e) {
            Singleton.getInstance(context).getConsoleListener().debug("" + e.getLocalizedMessage());
        }
    }

    @Nullable
    private static threads.ipfs.api.Peer getDirectPeer(@NonNull Context context) {
        Preconditions.checkNotNull((Object)context);
        int thresholdMax = Preferences.getMaxThreshold(context);
        IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            List peers = ipfs.swarmPeers();
            for (threads.ipfs.api.Peer peer : peers) {
                if (peer.isRelay() || !ipfs.swarmConnect(peer, thresholdMax)) continue;
                Singleton.getInstance(context).getConsoleListener().debug("Direct Swarm Peer : " + peer.toString());
                return peer;
            }
        }
        return null;
    }

    @NonNull
    private static List<threads.ipfs.api.Peer> getRelayPeers(@NonNull Context context, @NonNull Set<String> ignoreList) {
        Preconditions.checkNotNull((Object)context);
        Preconditions.checkNotNull(ignoreList);
        ArrayList<threads.ipfs.api.Peer> relays = new ArrayList<threads.ipfs.api.Peer>();
        int thresholdMax = Preferences.getMaxThreshold(context);
        IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {
            List peers = ipfs.swarmPeers();
            for (threads.ipfs.api.Peer peer : peers) {
                if (!ignoreList.contains(peer.getPid().getPid()) && peer.isRelay() && ipfs.swarmConnect(peer, thresholdMax)) {
                    Singleton.getInstance(context).getConsoleListener().debug("Detect Swarm Relay Peer : " + peer.toString());
                    relays.add(peer);
                }
                if (relays.size() < 3) continue;
                break;
            }
        }
        return relays;
    }
}

