package threads.core;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.room.Room;
import androidx.room.migration.Migration;
import androidx.sqlite.db.SupportSQLiteDatabase;

import threads.core.api.EventsDatabase;
import threads.core.api.Message;
import threads.core.api.MessageKind;
import threads.core.api.PeersDatabase;
import threads.core.api.Server;
import threads.core.api.ThreadsDatabase;
import threads.iota.EntityService;
import threads.iota.IOTA;
import threads.ipfs.IPFS;
import threads.ipfs.api.AddressesConfig;
import threads.ipfs.api.ApiListener;
import threads.ipfs.api.ConnMgrConfig;
import threads.ipfs.api.DiscoveryConfig;
import threads.ipfs.api.ExperimentalConfig;
import threads.ipfs.api.PubsubConfig;
import threads.ipfs.api.ReproviderConfig;
import threads.ipfs.api.RoutingConfig;
import threads.ipfs.api.SwarmConfig;

import static androidx.core.util.Preconditions.checkNotNull;


public class Singleton {

    private static final Migration MIGRATION_61_62 = new Migration(61, 62) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("DROP TABLE TransactionHash");
        }
    };
    private static final Migration MIGRATION_62_63 = new Migration(62, 63) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("DROP TABLE BlockAccount");
            database.execSQL("ALTER TABLE User ADD COLUMN blocked INTEGER DEFAULT 0 NOT NULL");
        }
    };
    private static final Migration MIGRATION_63_64 = new Migration(63, 64) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("DROP TABLE Peer");
        }
    };

    private static final Migration MIGRATION_64_65 = new Migration(64, 65) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("ALTER TABLE Threads ADD COLUMN pinned INTEGER DEFAULT 0 NOT NULL");
        }
    };

    private static Singleton SINGLETON = null;
    private final IOTA iota;
    private final ThreadsDatabase threadsDatabase;
    private final EventsDatabase eventsDatabase;
    private final PeersDatabase peersDatabase;
    private final THREADS threads;
    private final ConsoleListener consoleListener = new ConsoleListener();
    @Nullable
    private IPFS ipfs = null;
    private Singleton(@NonNull Context context) {
        checkNotNull(context);


        threadsDatabase = Room.databaseBuilder(context,
                ThreadsDatabase.class,
                ThreadsDatabase.class.getSimpleName()).addMigrations(MIGRATION_61_62,
                MIGRATION_62_63, MIGRATION_63_64, MIGRATION_64_65).build();

        eventsDatabase =
                Room.inMemoryDatabaseBuilder(context, EventsDatabase.class).build();

        peersDatabase =
                Room.inMemoryDatabaseBuilder(context, PeersDatabase.class).build();

        EntityService entityService = EntityService.getInstance(context);

        threads = THREADS.createThreads(
                threadsDatabase, eventsDatabase, peersDatabase, entityService);

        consoleListener.setReport(Preferences.isReportMode(context));
        consoleListener.setDebug(Preferences.isDebugMode(context));

        IOTA.Builder iotaBuilder = new IOTA.Builder();
        Server server = Preferences.getTangleServer(context);
        iotaBuilder.protocol(server.getProtocol());
        iotaBuilder.host(server.getHost());
        iotaBuilder.port(server.getPort());
        iotaBuilder.timeout(Preferences.getTangleTimeout(context));
        iota = iotaBuilder.build();


        Integer quicPort = null;
        if (Preferences.isQUICEnabled(context)) {
            quicPort = Preferences.getSwarmPort(context);
        }

        AddressesConfig addresses = AddressesConfig.create(
                Preferences.getSwarmPort(context),
                null, null, quicPort);

        ExperimentalConfig experimental = ExperimentalConfig.create();
        experimental.setQUIC(Preferences.isQUICEnabled(context));
        experimental.setFilestoreEnabled(Preferences.isFilestoreEnabled(context));
        experimental.setPreferTLS(Preferences.isPreferTLS(context));


        PubsubConfig pubsub = PubsubConfig.create();
        pubsub.setRouter(Preferences.getPubsubRouter(context));


        SwarmConfig swarmConfig = SwarmConfig.create();
        swarmConfig.setDisableBandwidthMetrics(true);
        swarmConfig.setDisableNatPortMap(false);
        swarmConfig.setDisableRelay(false);
        swarmConfig.setEnableAutoRelay(Preferences.isAutoRelayEnabled(context));
        swarmConfig.setEnableAutoNATService(Preferences.isAutoNATServiceEnabled(context));
        swarmConfig.setEnableRelayHop(Preferences.isRelayHopEnabled(context));

        ConnMgrConfig mgr = swarmConfig.getConnMgr();
        mgr.setGracePeriod(Preferences.getGracePeriod(context));
        mgr.setHighWater(Preferences.getHighWater(context));
        mgr.setLowWater(Preferences.getLowWater(context));
        mgr.setType(Preferences.getConnMgrConfigType(context));

        DiscoveryConfig discoveryConfig = DiscoveryConfig.create();
        discoveryConfig.getMdns().setEnabled(Preferences.isMdnsEnabled(context));


        ReproviderConfig reproviderConfig = ReproviderConfig.create();
        reproviderConfig.setInterval(Preferences.getReproviderInterval(context));

        RoutingConfig routingConfig = RoutingConfig.create();
        routingConfig.setType(Preferences.getRoutingType(context));

        try {
            ipfs = IPFS.getInstance(context, this.consoleListener, addresses, experimental, pubsub,
                    discoveryConfig, swarmConfig, routingConfig, reproviderConfig, null);

            Preferences.setPID(context, ipfs.getPeerID());
        } catch (Throwable e) {
            Preferences.evaluateException(threads, Preferences.IPFS_INSTALL_FAILURE, e);
        }

    }

    @NonNull
    public static Singleton getInstance(@NonNull Context context) {
        checkNotNull(context);
        if (SINGLETON == null) {
            SINGLETON = new Singleton(context);
        }
        return SINGLETON;
    }

    @NonNull
    public PeersDatabase getPeersDatabase() {
        return peersDatabase;
    }

    @NonNull
    public ConsoleListener getConsoleListener() {
        return consoleListener;
    }

    @NonNull
    public IOTA getIota() {
        return iota;
    }

    @Nullable
    public IPFS getIpfs() {
        return ipfs;
    }

    @NonNull
    public EventsDatabase getEventsDatabase() {
        return eventsDatabase;
    }

    @NonNull
    public THREADS getThreads() {
        return threads;
    }

    @NonNull
    public ThreadsDatabase getThreadsDatabase() {
        return threadsDatabase;
    }


    public class ConsoleListener implements ApiListener {
        private boolean debug = false;
        private boolean report = false;

        public boolean isDebug() {
            return debug;
        }

        public void setDebug(boolean debug) {
            this.debug = debug;
        }

        public boolean isReport() {
            return report;
        }

        public void setReport(boolean report) {
            this.report = report;
        }


        @Override
        public void debug(@NonNull String message) {
            if (debug && report) {
                long timestamp = System.currentTimeMillis();
                new Thread(() -> {
                    Message em = threads.createMessage(MessageKind.DEBUG, message, timestamp);
                    threads.storeMessage(em);
                }).start();
            }
        }

        @Override
        public void info(@NonNull String message) {
            if (report) {
                long timestamp = System.currentTimeMillis();
                new Thread(() -> {
                    Message em = threads.createMessage(MessageKind.INFO, message, timestamp);
                    threads.storeMessage(em);
                }).start();
            }
        }

        @Override
        public void error(@NonNull String message) {
            if (report) {
                long timestamp = System.currentTimeMillis();
                new Thread(() -> {
                    Message em = threads.createMessage(MessageKind.ERROR, message, timestamp);
                    threads.storeMessage(em);
                }).start();
            }
        }

    }


}
