package threads.core.api;

import androidx.annotation.NonNull;

import java.util.Objects;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;


public class ServerInfo {

    @NonNull
    private final int port;
    @NonNull
    private final String host;
    @NonNull
    private final String protocol;
    private boolean online = true;
    private long milestone = 0L;
    @NonNull
    private String name = "N/A";
    @NonNull
    private String version = "N/A";
    private boolean supportPoW;

    ServerInfo(@NonNull String protocol,
               @NonNull String host,
               int port) {
        checkNotNull(protocol);
        checkNotNull(host);
        checkArgument(port > 0);
        this.protocol = protocol;
        this.host = host;
        this.port = port;
        this.supportPoW = false;
    }

    public static ServerInfo createServerInfo(@NonNull String protocol,
                                              @NonNull String host,
                                              int port) {
        return new ServerInfo(protocol, host, port);
    }

    public static ServerInfo createServerInfo(@NonNull Server server) {
        return createServerInfo(server.getProtocol(), server.getHost(), server.getPort());
    }

    @Override
    @NonNull
    public String toString() {
        return "ServerInfo{" +
                ", port='" + port + '\'' +
                ", host='" + host + '\'' +
                ", protocol='" + protocol + '\'' +
                ", online=" + online +
                ", milestone=" + milestone +
                ", name='" + name + '\'' +
                ", version='" + version + '\'' +
                ", supportPoW=" + supportPoW +
                '}';
    }


    public String getDisplayName() {
        return Server.getServer(this, false);
    }

    public boolean isSupportPoW() {
        return supportPoW;
    }

    public void setSupportPoW(boolean supportPoW) {
        this.supportPoW = supportPoW;
    }


    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ServerInfo that = (ServerInfo) o;
        return online == that.online &&
                milestone == that.milestone &&
                supportPoW == that.supportPoW &&
                Objects.equals(port, that.port) &&
                Objects.equals(host, that.host) &&
                Objects.equals(protocol, that.protocol) &&
                Objects.equals(name, that.name) &&
                Objects.equals(version, that.version);
    }

    @Override
    public int hashCode() {

        return Objects.hash(port, host, protocol, online, milestone, name, version, supportPoW);
    }

    @NonNull
    public int getPort() {
        return port;
    }

    @NonNull
    public String getHost() {
        return host;
    }

    @NonNull
    public String getProtocol() {
        return protocol;
    }

    public boolean isOnline() {
        return online;
    }

    public void setOnline(boolean online) {
        this.online = online;

    }

    @NonNull
    public String getName() {
        return name;
    }

    public void setName(@NonNull String name) {
        this.name = name;
    }

    @NonNull
    public String getVersion() {
        return version;
    }

    public void setVersion(@NonNull String version) {
        this.version = version;
    }

    public long getMilestone() {
        return milestone;
    }


    public void setMilestone(long milestone) {
        this.milestone = milestone;
    }


}
