package threads.core.api;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.Gson;

import java.util.Date;

import threads.iota.Entity;
import threads.ipfs.api.CID;
import threads.ipfs.api.Encryption;
import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkNotNull;

public class NoteRequestDecoder {
    private static final String TAG = NoteRequestDecoder.class.getSimpleName();

    @Nullable
    public static Thread convert(@NonNull Entity entity, @NonNull String privateKey) {
        checkNotNull(privateKey);
        checkNotNull(entity);
        Gson gson = new Gson();
        Content content = gson.fromJson(entity.getContent(), Content.class);
        Thread thread = convert(content, privateKey);
        if (thread != null) {
            thread.setHash(entity.getHash());
            thread.setBundle(entity.getBundle());
            thread.setTimestamp(entity.getTimestamp());
        }
        return null;
    }

    @Nullable
    public static Thread convert(@NonNull Content content, @NonNull String privateKey) {
        checkNotNull(privateKey);
        checkNotNull(content);
        try {
            // NOT ENCRYPTED
            String sesKey = "";

            // ENCRYPTED RSA
            String encSesKey = content.get(Content.SKEY);
            if (encSesKey != null) {
                sesKey = Encryption.decryptRSA(encSesKey, privateKey);
            }

            // NOT ENCRYPTED
            String senderPid = content.get(Content.PID);
            checkNotNull(senderPid);


            // NOT ENCRYPTED
            String senderKey = content.get(Content.PKEY);
            checkNotNull(senderKey);

            // NOT ENCRYPTED
            String date = content.get(Content.DATE);
            checkNotNull(date);

            // NOT ENCRYPTED
            String readOnlyString = content.get(Content.READ_ONLY);
            checkNotNull(readOnlyString);
            boolean isReadOnly = Boolean.valueOf(readOnlyString);


            // ENCRYPTED
            String senderAlias = content.get(Content.ALIAS);
            checkNotNull(senderAlias);
            senderAlias = Encryption.decrypt(senderAlias, sesKey);

            // ENCRYPTED
            String cid = content.get(Content.CID);
            checkNotNull(cid);
            cid = Encryption.decrypt(cid, sesKey);



            // ENCRYPTED
            String additions = content.get(Content.ADDS);
            checkNotNull(additions);
            additions = Encryption.decrypt(additions, sesKey);


            // ENCRYPTED
            CID image = null;
            String imgValue = content.get(Content.IMG);
            if (imgValue != null) {
                imgValue = Encryption.decrypt(imgValue, sesKey);
                image = CID.create(imgValue);
            }

            Thread thread = Thread.createThread(
                    ThreadStatus.REQUEST,
                    PID.create(senderPid),
                    senderAlias,
                    senderKey,
                    sesKey,
                    Kind.IN,
                    new Date(Long.valueOf(date)),
                    0L,
                    isReadOnly);
            thread.setCid(CID.create(cid));
            thread.setImage(image);
            thread.addMember(PID.create(senderPid));


            if (!additions.isEmpty()) {
                thread.setExternalAdditions(Additionals.toHashMap(additions));
            }


            return thread;
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
        return null;
    }

}
