package threads.core.api;

import androidx.annotation.NonNull;
import androidx.room.ColumnInfo;
import androidx.room.PrimaryKey;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.URL;
import java.util.Objects;

import javax.net.ssl.HttpsURLConnection;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;


@androidx.room.Entity
public class Server extends Entity {

    private static final int TIMEOUT_SERVER_CHECK = 5000;

    @NonNull
    @ColumnInfo(name = "host")
    private final String host;
    @ColumnInfo(name = "port")
    private final int port;
    @NonNull
    @ColumnInfo(name = "protocol")
    private final String protocol;
    @NonNull
    @ColumnInfo(name = "alias")
    private final String alias;
    @PrimaryKey(autoGenerate = true)
    private long idx;

    Server(@NonNull String protocol,
           @NonNull String host,
           int port,
           @NonNull String alias) {
        checkNotNull(protocol);
        checkNotNull(host);
        checkNotNull(alias);

        this.protocol = protocol;
        this.port = port;
        this.host = host;
        this.alias = alias;
    }

    @NonNull
    public static Server createServer(@NonNull String protocol,
                                      @NonNull String host,
                                      int port,
                                      @NonNull String alias) {

        checkNotNull(protocol);
        checkNotNull(host);
        checkArgument(port > 0);
        checkNotNull(alias);

        Server server = new Server(protocol, host, port, alias);
        server.setTimestamp(System.currentTimeMillis());
        return server;
    }


    public static String getServer(@NonNull ServerInfo server, boolean ipv6) {
        checkNotNull(server);
        String host = server.getHost();
        checkNotNull(host);
        String protocol = server.getProtocol();
        int port = server.getPort();
        if (ipv6) {
            host = "[" + host + "]";
        }
        if (80 == port) {
            return protocol + "://" + host;
        }

        return protocol + "://" + host + ":" + port;

    }

    @NonNull
    private static String getServer(@NonNull Server server, boolean ipv6) {
        checkNotNull(server);
        String host = server.getHost();
        checkNotNull(host);
        String protocol = server.getProtocol();
        int port = server.getPort();
        if (ipv6) {
            host = "[" + host + "]";
        }
        if (80 == port) {
            return protocol + "://" + host;
        }

        return protocol + "://" + host + ":" + port;

    }

    public static boolean isReachable(@NonNull Server server) {
        checkNotNull(server);
        boolean b = true;
        try {
            if (server.getProtocol().equalsIgnoreCase("http")) {
                InetSocketAddress sa = new InetSocketAddress(server.getHost(),
                        server.getPort());
                Socket ss = new Socket();
                ss.connect(sa, TIMEOUT_SERVER_CHECK);
                ss.close();
            } else {
                boolean ipv6 = false;
                InetAddress inetAddress = InetAddress.getByName(server.getHost());
                if (inetAddress instanceof Inet6Address) {
                    ipv6 = true;
                }
                URL url = new URL(getServer(server, ipv6));
                HttpsURLConnection urlConnection =
                        (HttpsURLConnection) url.openConnection();
                urlConnection.setConnectTimeout(TIMEOUT_SERVER_CHECK);

                urlConnection.connect(); // would throw an exception
            }
        } catch (Throwable e) {
            b = false;
        }
        return b;
    }

    public long getIdx() {
        return idx;
    }

    void setIdx(long idx) {
        this.idx = idx;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Server that = (Server) o;
        return idx == that.idx;
    }

    @Override
    public int hashCode() {
        return Objects.hash(idx);
    }


    @NonNull
    public String getAlias() {
        return alias;
    }


    @NonNull
    public String getHost() {
        return host;
    }

    @NonNull
    public int getPort() {
        return port;
    }

    @NonNull
    public String getProtocol() {
        return protocol;
    }

}
