package threads.core.api;

import androidx.annotation.NonNull;
import androidx.room.ColumnInfo;
import androidx.room.PrimaryKey;
import androidx.room.TypeConverters;

import java.util.Objects;

import threads.ipfs.api.PID;

import static androidx.core.util.Preconditions.checkNotNull;

@androidx.room.Entity
public class Peer extends Entity {

    @PrimaryKey
    @NonNull
    @ColumnInfo(name = "pid")
    private final String pid;

    @NonNull
    @TypeConverters(Addresses.class)
    @ColumnInfo(name = "addresses")
    private Addresses addresses = new Addresses();


    Peer(@NonNull String pid) {
        this.pid = pid;
    }

    public static Peer createPeer(@NonNull PID pid) {
        checkNotNull(pid);
        return new Peer(pid.getPid());
    }

    @NonNull
    String getPid() {
        return pid;
    }

    public PID getPID() {
        return PID.create(pid);
    }

    @NonNull
    public Addresses getAddresses() {
        return (Addresses) addresses.clone();
    }

    public void setAddresses(@NonNull Addresses addresses) {
        this.addresses = addresses;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Peer peer = (Peer) o;
        return Objects.equals(pid, peer.pid);
    }

    @Override
    public int hashCode() {
        return Objects.hash(pid);
    }

    public void addAddress(@NonNull String relayPid, @NonNull String address) {
        checkNotNull(relayPid);
        checkNotNull(address);
        this.addresses.put(relayPid, address);
    }

    public void removeAddress(@NonNull String relayPid) {
        checkNotNull(relayPid);
        this.addresses.remove(relayPid);
    }


    public void addAddress(@NonNull PID relay, @NonNull String address) {
        checkNotNull(relay);
        checkNotNull(address);
        this.addresses.put(relay.getPid(), address);
    }

    public void removeAddress(@NonNull PID relay) {
        checkNotNull(relay);
        this.addresses.remove(relay.getPid());
    }
}
