/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.validation.node;

import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.logging.Logger;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.shapes.TimestampShape;
import software.amazon.smithy.model.traits.TimestampFormatTrait;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.node.NodeValidatorPlugin;

final class TimestampFormatPlugin
implements NodeValidatorPlugin {
    private static final DateTimeFormatter HTTP_DATE = DateTimeFormatter.RFC_1123_DATE_TIME;
    private static final DateTimeFormatter DATE_TIME_Z = DateTimeFormatter.ISO_INSTANT;
    private static final Logger LOGGER = Logger.getLogger(TimestampFormatPlugin.class.getName());

    TimestampFormatPlugin() {
    }

    @Override
    public void apply(Shape shape, Node value, NodeValidatorPlugin.Context context, NodeValidatorPlugin.Emitter emitter) {
        if (shape instanceof TimestampShape) {
            boolean fromMemberWithTrait = context.getReferringMember().filter(member -> member.hasTrait(TimestampFormatTrait.ID)).isPresent();
            if (!fromMemberWithTrait) {
                this.validate(shape, shape.getTrait(TimestampFormatTrait.class).orElse(null), value, emitter);
            }
        } else if (shape instanceof MemberShape && shape.hasTrait(TimestampFormatTrait.ID)) {
            this.validate(shape, shape.expectTrait(TimestampFormatTrait.class), value, emitter);
        }
    }

    private void validate(Shape shape, TimestampFormatTrait trait, Node value, NodeValidatorPlugin.Emitter emitter) {
        if (trait == null) {
            this.defaultValidation(shape, value, emitter);
        } else {
            switch (trait.getValue()) {
                case "date-time": {
                    this.validateDatetime(shape, value, emitter);
                    break;
                }
                case "epoch-seconds": {
                    if (value.isNumberNode()) break;
                    emitter.accept(value, Severity.ERROR, String.format("Invalid %s value provided for a timestamp with a `%s` format.", new Object[]{value.getType(), trait.getValue()}));
                    break;
                }
                case "http-date": {
                    this.validateHttpDate(value, emitter);
                    break;
                }
                default: {
                    LOGGER.info(() -> "Unknown timestampFormat trait value: " + trait.getValue());
                }
            }
        }
    }

    private void defaultValidation(Shape shape, Node value, NodeValidatorPlugin.Emitter emitter) {
        if (!value.isNumberNode()) {
            if (value.isStringNode()) {
                this.validateDatetime(shape, value, emitter);
            } else {
                emitter.accept(value, "Invalid " + (Object)((Object)value.getType()) + " value provided for timestamp, `" + shape.getId() + "`. Expected a number that contains epoch seconds with optional millisecond precision, or a string that contains an RFC 3339 formatted timestamp (e.g., \"1985-04-12T23:20:50.52Z\")");
            }
        }
    }

    private void validateDatetime(Shape shape, Node value, NodeValidatorPlugin.Emitter emitter) {
        if (!value.isStringNode()) {
            emitter.accept(value, "Expected a string value for a date-time timestamp (e.g., \"1985-04-12T23:20:50.52Z\")");
            return;
        }
        String timestamp = value.expectStringNode().getValue();
        if (!timestamp.endsWith("Z") || !this.isValidFormat(timestamp, DATE_TIME_Z)) {
            emitter.accept(value, "Invalid string value, `" + timestamp + "`, provided for timestamp, `" + shape.getId() + "`. Expected an RFC 3339 formatted timestamp (e.g., \"1985-04-12T23:20:50.52Z\")");
        }
    }

    private void validateHttpDate(Node value, NodeValidatorPlugin.Emitter emitter) {
        if (!value.asStringNode().isPresent()) {
            emitter.accept(value, Severity.ERROR, this.createInvalidHttpDateMessage(value.getType().toString()));
        } else {
            String dateValue = value.asStringNode().get().getValue();
            if (!this.isValidFormat(dateValue, HTTP_DATE) || !dateValue.endsWith("GMT")) {
                emitter.accept(value, Severity.ERROR, this.createInvalidHttpDateMessage(dateValue));
            }
        }
    }

    private String createInvalidHttpDateMessage(String dateValue) {
        return String.format("Invalid value provided for %s formatted timestamp. Expected a string value that matches the IMF-fixdate production of RFC 9110 section-5.6.7. Found: %s", "http-date", dateValue);
    }

    private boolean isValidFormat(String value, DateTimeFormatter format) {
        try {
            format.parse(value);
            return true;
        }
        catch (DateTimeParseException e) {
            return false;
        }
    }
}

