/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.loader.CompositeModelFile;
import software.amazon.smithy.model.loader.FullyResolvedModelFile;
import software.amazon.smithy.model.loader.ImmutablePreludeModelFile;
import software.amazon.smithy.model.loader.ModelDiscovery;
import software.amazon.smithy.model.loader.ModelFile;
import software.amazon.smithy.model.loader.ModelImportException;
import software.amazon.smithy.model.loader.ModelLoader;
import software.amazon.smithy.model.loader.ModelValidator;
import software.amazon.smithy.model.loader.Prelude;
import software.amazon.smithy.model.loader.TraitContainer;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.traits.Trait;
import software.amazon.smithy.model.traits.TraitFactory;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidatedResult;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.model.validation.Validator;
import software.amazon.smithy.model.validation.ValidatorFactory;
import software.amazon.smithy.utils.FunctionalUtils;
import software.amazon.smithy.utils.Pair;

public final class ModelAssembler {
    public static final String ALLOW_UNKNOWN_TRAITS = "assembler.allowUnknownTraits";
    public static final String DISABLE_JAR_CACHE = "assembler.disableJarCache";
    private static final Logger LOGGER = Logger.getLogger(ModelAssembler.class.getName());
    private TraitFactory traitFactory;
    private ValidatorFactory validatorFactory;
    private boolean disableValidation;
    private final Map<String, Supplier<InputStream>> inputStreamModels = new HashMap<String, Supplier<InputStream>>();
    private final List<Validator> validators = new ArrayList<Validator>();
    private final List<Node> documentNodes = new ArrayList<Node>();
    private final List<Model> mergeModels = new ArrayList<Model>();
    private final List<Shape> shapes = new ArrayList<Shape>();
    private final List<Pair<ShapeId, Trait>> pendingTraits = new ArrayList<Pair<ShapeId, Trait>>();
    private final Map<String, Node> metadata = new HashMap<String, Node>();
    private final Map<String, Object> properties = new HashMap<String, Object>();
    private boolean disablePrelude;

    public ModelAssembler copy() {
        ModelAssembler assembler = new ModelAssembler();
        assembler.traitFactory = this.traitFactory;
        assembler.validatorFactory = this.validatorFactory;
        assembler.inputStreamModels.putAll(this.inputStreamModels);
        assembler.validators.addAll(this.validators);
        assembler.documentNodes.addAll(this.documentNodes);
        assembler.mergeModels.addAll(this.mergeModels);
        assembler.shapes.addAll(this.shapes);
        assembler.pendingTraits.addAll(this.pendingTraits);
        assembler.metadata.putAll(this.metadata);
        assembler.disablePrelude = this.disablePrelude;
        assembler.properties.putAll(this.properties);
        assembler.disableValidation = this.disableValidation;
        return assembler;
    }

    public ModelAssembler reset() {
        this.shapes.clear();
        this.pendingTraits.clear();
        this.metadata.clear();
        this.mergeModels.clear();
        this.inputStreamModels.clear();
        this.validators.clear();
        this.documentNodes.clear();
        this.disablePrelude = false;
        this.disableValidation = false;
        return this;
    }

    public ModelAssembler traitFactory(TraitFactory traitFactory) {
        this.traitFactory = Objects.requireNonNull(traitFactory);
        return this;
    }

    public ModelAssembler validatorFactory(ValidatorFactory validatorFactory) {
        this.validatorFactory = Objects.requireNonNull(validatorFactory);
        return this;
    }

    public ModelAssembler addValidator(Validator validator) {
        this.validators.add(Objects.requireNonNull(validator));
        return this;
    }

    public ModelAssembler addUnparsedModel(String sourceLocation, String model) {
        this.inputStreamModels.put(sourceLocation, () -> new ByteArrayInputStream(model.getBytes(StandardCharsets.UTF_8)));
        return this;
    }

    public ModelAssembler addDocumentNode(Node document) {
        this.documentNodes.add(Objects.requireNonNull(document));
        return this;
    }

    public ModelAssembler addImport(String importPath) {
        return this.addImport(Paths.get(Objects.requireNonNull(importPath, "importPath must not be null"), new String[0]));
    }

    public ModelAssembler addImport(Path importPath) {
        Objects.requireNonNull(importPath, "The importPath provided to ModelAssembler#addImport was null");
        if (Files.isDirectory(importPath, new LinkOption[0])) {
            try {
                Files.walk(importPath, FileVisitOption.FOLLOW_LINKS).filter(p -> !p.equals(importPath)).filter(p -> Files.isDirectory(p, new LinkOption[0]) || Files.isRegularFile(p, new LinkOption[0])).forEach(this::addImport);
            }
            catch (IOException e) {
                throw new ModelImportException("Error loading the contents of " + importPath, e);
            }
        } else if (Files.isRegularFile(importPath, new LinkOption[0])) {
            this.inputStreamModels.put(importPath.toString(), () -> {
                try {
                    return Files.newInputStream(importPath, new OpenOption[0]);
                }
                catch (IOException e) {
                    throw new ModelImportException("Unable to import Smithy model from " + importPath + ": " + e.getMessage(), e);
                }
            });
        } else {
            throw new ModelImportException("Cannot find import file: " + importPath);
        }
        return this;
    }

    public ModelAssembler addImport(URL url) {
        Objects.requireNonNull(url, "The provided url to ModelAssembler#addImport was null");
        String key = url.toExternalForm();
        if (key.startsWith("file:")) {
            try {
                key = Paths.get(url.toURI()).toString();
            }
            catch (URISyntaxException e) {
                key = key.substring(5);
            }
        }
        this.inputStreamModels.put(key, () -> {
            try {
                URLConnection connection = url.openConnection();
                if (this.properties.containsKey(DISABLE_JAR_CACHE)) {
                    connection.setUseCaches(false);
                }
                return connection.getInputStream();
            }
            catch (IOException | UncheckedIOException e) {
                throw new ModelImportException("Unable to open Smithy model import URL: " + url.toExternalForm(), e);
            }
        });
        return this;
    }

    public ModelAssembler disablePrelude() {
        this.disablePrelude = true;
        return this;
    }

    public ModelAssembler addShape(Shape shape) {
        this.shapes.add(shape);
        return this;
    }

    public ModelAssembler addShapes(Shape ... shapes) {
        for (Shape shape : shapes) {
            this.addShape(shape);
        }
        return this;
    }

    public ModelAssembler addTrait(ShapeId target, Trait trait) {
        this.pendingTraits.add((Pair<ShapeId, Trait>)Pair.of((Object)target, (Object)trait));
        return this;
    }

    public ModelAssembler addModel(Model model) {
        this.mergeModels.add(model);
        return this;
    }

    public ModelAssembler putMetadata(String name, Node value) {
        this.metadata.put(Objects.requireNonNull(name), Objects.requireNonNull(value));
        return this;
    }

    public ModelAssembler discoverModels(ClassLoader loader) {
        return this.addDiscoveredModels(ModelDiscovery.findModels(loader));
    }

    public ModelAssembler discoverModels() {
        return this.addDiscoveredModels(ModelDiscovery.findModels());
    }

    private ModelAssembler addDiscoveredModels(List<URL> urls) {
        for (URL url : urls) {
            LOGGER.fine(() -> "Discovered Smithy model: " + url);
            this.addImport(url);
        }
        return this;
    }

    public ModelAssembler putProperty(String setting, Object value) {
        this.properties.put(setting, value);
        return this;
    }

    public ModelAssembler removeProperty(String setting) {
        this.properties.remove(setting);
        return this;
    }

    public ModelAssembler disableValidation() {
        this.disableValidation = true;
        return this;
    }

    public ValidatedResult<Model> assemble() {
        if (this.traitFactory == null) {
            this.traitFactory = LazyTraitFactoryHolder.INSTANCE;
        }
        List<ModelFile> modelFiles = this.createModelFiles();
        try {
            CompositeModelFile files = new CompositeModelFile(this.traitFactory, modelFiles);
            TraitContainer traits = files.resolveShapes(files.shapeIds(), files::getShapeType);
            Model model = Model.builder().metadata(files.metadata()).addShapes(files.createShapes(traits)).build();
            return this.validate(model, traits, files.events());
        }
        catch (SourceException e) {
            ArrayList<ValidationEvent> events = new ArrayList<ValidationEvent>();
            events.add(ValidationEvent.fromSourceException(e));
            for (ModelFile modelFile : modelFiles) {
                events.addAll(modelFile.events());
            }
            return ValidatedResult.fromErrors(events);
        }
    }

    private List<ModelFile> createModelFiles() {
        ArrayList<ModelFile> modelFiles = new ArrayList<ModelFile>();
        if (!this.disablePrelude) {
            modelFiles.add(new ImmutablePreludeModelFile(Prelude.getPreludeModel()));
        }
        FullyResolvedModelFile assemblerModelFile = FullyResolvedModelFile.fromShapes(this.traitFactory, this.shapes);
        modelFiles.add(assemblerModelFile);
        this.metadata.forEach(assemblerModelFile::putMetadata);
        for (Pair<ShapeId, Trait> pair : this.pendingTraits) {
            assemblerModelFile.onTrait((ShapeId)pair.left, (Trait)pair.right);
        }
        for (Model model : this.mergeModels) {
            List<Shape> nonPrelude = model.shapes().filter(FunctionalUtils.not(Prelude::isPreludeShape)).collect(Collectors.toList());
            FullyResolvedModelFile resolvedFile = FullyResolvedModelFile.fromShapes(this.traitFactory, nonPrelude);
            model.getMetadata().forEach(resolvedFile::putMetadata);
            modelFiles.add(resolvedFile);
        }
        for (Node node : this.documentNodes) {
            try {
                modelFiles.add(ModelLoader.loadParsedNode(this.traitFactory, node));
            }
            catch (SourceException e) {
                assemblerModelFile.events().add(ValidationEvent.fromSourceException(e));
            }
        }
        for (Map.Entry entry : this.inputStreamModels.entrySet()) {
            try {
                ModelFile loaded = ModelLoader.load(this.traitFactory, this.properties, (String)entry.getKey(), (Supplier)entry.getValue());
                if (loaded == null) {
                    LOGGER.warning(() -> "No ModelLoader was able to load " + (String)entry.getKey());
                    continue;
                }
                modelFiles.add(loaded);
            }
            catch (SourceException e) {
                assemblerModelFile.events().add(ValidationEvent.fromSourceException(e));
            }
        }
        return modelFiles;
    }

    private ValidatedResult<Model> validate(Model model, TraitContainer traits, List<ValidationEvent> events) {
        this.validateTraits(model.getShapeIds(), traits, events);
        if (this.disableValidation) {
            return new ValidatedResult<Model>(model, events);
        }
        if (this.validatorFactory == null) {
            this.validatorFactory = LazyValidatorFactoryHolder.INSTANCE;
        }
        List<ValidationEvent> mergedEvents = ModelValidator.validate(model, this.validatorFactory, this.assembleValidators());
        mergedEvents.addAll(events);
        return new ValidatedResult<Model>(model, mergedEvents);
    }

    private void validateTraits(Set<ShapeId> ids, TraitContainer resolvedTraits, List<ValidationEvent> events) {
        Severity severity = this.areUnknownTraitsAllowed() ? Severity.WARNING : Severity.ERROR;
        for (Map.Entry<ShapeId, Map<ShapeId, Trait>> entry : resolvedTraits.traits().entrySet()) {
            ShapeId target = entry.getKey();
            for (Trait trait : entry.getValue().values()) {
                if (!ids.contains(trait.toShapeId())) {
                    events.add(ValidationEvent.builder().id("Model").severity(severity).sourceLocation(trait).shapeId(target).message(String.format("Unable to resolve trait `%s`. If this is a custom trait, then it must be defined before it can be used in a model.", trait.toShapeId())).build());
                }
                if (ids.contains(target)) continue;
                events.add(ValidationEvent.builder().id("Model").severity(Severity.ERROR).sourceLocation(trait).message(String.format("Trait `%s` applied to unknown shape `%s`", Trait.getIdiomaticTraitName(trait.toShapeId()), target)).build());
            }
        }
    }

    private boolean areUnknownTraitsAllowed() {
        Object allowUnknown = this.properties.get(ALLOW_UNKNOWN_TRAITS);
        return allowUnknown != null && (Boolean)allowUnknown != false;
    }

    private List<Validator> assembleValidators() {
        ArrayList<Validator> copiedValidators = new ArrayList<Validator>(this.validatorFactory.loadBuiltinValidators());
        copiedValidators.addAll(this.validators);
        return copiedValidators;
    }

    static final class LazyTraitFactoryHolder {
        static final TraitFactory INSTANCE = TraitFactory.createServiceFactory(ModelAssembler.class.getClassLoader());

        LazyTraitFactoryHolder() {
        }
    }

    private static final class LazyValidatorFactoryHolder {
        static final ValidatorFactory INSTANCE = ValidatorFactory.createServiceFactory(ModelAssembler.class.getClassLoader());

        private LazyValidatorFactoryHolder() {
        }
    }
}

