/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.CharBuffer;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.IdlStringLexer;
import software.amazon.smithy.model.loader.IdlToken;
import software.amazon.smithy.model.loader.LoaderUtils;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.loader.ParserUtils;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.SimpleParser;
import software.amazon.smithy.utils.SmithyBuilder;
import software.amazon.smithy.utils.SmithyUnstableApi;

@SmithyUnstableApi
final class IdlTokenizer
implements Iterator<IdlToken> {
    private static final int MAX_NESTING_LEVEL = 64;
    private final SimpleParser parser;
    private final String filename;
    private final Function<CharSequence, String> stringTable;
    private IdlToken currentTokenType;
    private int currentTokenStart = -1;
    private int currentTokenEnd = -1;
    private int currentTokenLine = -1;
    private int currentTokenColumn = -1;
    private Number currentTokenNumber;
    private CharSequence currentTokenStringSlice;
    private String currentTokenError;
    private final Deque<CharSequence> docCommentLines = new ArrayDeque<CharSequence>();
    private final Consumer<ValidationEvent> validationEventListener;

    private IdlTokenizer(Builder builder) {
        this.filename = builder.filename;
        this.stringTable = builder.stringTable;
        this.parser = new SimpleParser((CharSequence)SmithyBuilder.requiredState((String)"model", (Object)builder.model), 64);
        this.validationEventListener = builder.validationEventListener;
    }

    public static Builder builder() {
        return new Builder();
    }

    CharSequence getInput(int start, int end) {
        return CharBuffer.wrap(this.parser.input(), start, end);
    }

    public String internString(CharSequence sequence) {
        return this.stringTable.apply(sequence);
    }

    public void increaseNestingLevel() {
        try {
            this.parser.increaseNestingLevel();
        }
        catch (RuntimeException e) {
            throw this.syntax("Parser exceeded maximum allowed depth of 64", this.getCurrentTokenLocation());
        }
    }

    public void decreaseNestingLevel() {
        this.parser.decreaseNestingLevel();
    }

    public int getPosition() {
        return this.parser.position();
    }

    public int getLine() {
        return this.parser.line();
    }

    public int getColumn() {
        return this.parser.column();
    }

    public IdlToken getCurrentToken() {
        if (this.currentTokenType == null) {
            this.next();
        }
        return this.currentTokenType;
    }

    public int getCurrentTokenLine() {
        this.getCurrentToken();
        return this.currentTokenLine;
    }

    public int getCurrentTokenColumn() {
        this.getCurrentToken();
        return this.currentTokenColumn;
    }

    public int getCurrentTokenStart() {
        this.getCurrentToken();
        return this.currentTokenStart;
    }

    public int getCurrentTokenSpan() {
        this.getCurrentToken();
        return this.currentTokenEnd - this.currentTokenStart;
    }

    public SourceLocation getCurrentTokenLocation() {
        this.getCurrentToken();
        return new SourceLocation(this.filename, this.currentTokenLine, this.currentTokenColumn);
    }

    public CharSequence getCurrentTokenLexeme() {
        this.getCurrentToken();
        return CharBuffer.wrap(this.parser.input(), this.currentTokenStart, this.currentTokenEnd);
    }

    public CharSequence getCurrentTokenStringSlice() {
        this.getCurrentToken();
        if (this.currentTokenStringSlice != null) {
            return this.currentTokenStringSlice;
        }
        if (this.currentTokenType == IdlToken.IDENTIFIER) {
            return this.getCurrentTokenLexeme();
        }
        throw this.syntax("The current token must be string or identifier but found: " + this.currentTokenType.getDebug(this.getCurrentTokenLexeme()), this.getCurrentTokenLocation());
    }

    public Number getCurrentTokenNumberValue() {
        this.getCurrentToken();
        if (this.currentTokenNumber == null) {
            throw this.syntax("The current token must be number but found: " + this.currentTokenType.getDebug(this.getCurrentTokenLexeme()), this.getCurrentTokenLocation());
        }
        return this.currentTokenNumber;
    }

    public String getCurrentTokenError() {
        this.getCurrentToken();
        if (this.currentTokenType != IdlToken.ERROR) {
            throw this.syntax("The current token must be an error but found: " + this.currentTokenType.getDebug(this.getCurrentTokenLexeme()), this.getCurrentTokenLocation());
        }
        return this.currentTokenError == null ? "" : this.currentTokenError;
    }

    @Override
    public boolean hasNext() {
        return this.currentTokenType != IdlToken.EOF;
    }

    @Override
    public IdlToken next() {
        this.currentTokenStringSlice = null;
        this.currentTokenNumber = null;
        this.currentTokenColumn = this.parser.column();
        this.currentTokenLine = this.parser.line();
        this.currentTokenEnd = this.currentTokenStart = this.parser.position();
        char c = this.parser.peek();
        switch (c) {
            case '\u0000': {
                if (this.currentTokenType == IdlToken.EOF) {
                    throw new NoSuchElementException("Expected another token but reached EOF");
                }
                this.currentTokenEnd = this.parser.position();
                this.currentTokenType = IdlToken.EOF;
                return this.currentTokenType;
            }
            case '\t': 
            case ' ': {
                return this.tokenizeSpace();
            }
            case '\n': 
            case '\r': {
                return this.tokenizeNewline();
            }
            case ',': {
                return this.singleCharToken(IdlToken.COMMA);
            }
            case '@': {
                return this.singleCharToken(IdlToken.AT);
            }
            case '$': {
                return this.singleCharToken(IdlToken.DOLLAR);
            }
            case '.': {
                return this.singleCharToken(IdlToken.DOT);
            }
            case '{': {
                return this.singleCharToken(IdlToken.LBRACE);
            }
            case '}': {
                return this.singleCharToken(IdlToken.RBRACE);
            }
            case '[': {
                return this.singleCharToken(IdlToken.LBRACKET);
            }
            case ']': {
                return this.singleCharToken(IdlToken.RBRACKET);
            }
            case '(': {
                return this.singleCharToken(IdlToken.LPAREN);
            }
            case ')': {
                return this.singleCharToken(IdlToken.RPAREN);
            }
            case '#': {
                return this.singleCharToken(IdlToken.POUND);
            }
            case '=': {
                return this.singleCharToken(IdlToken.EQUAL);
            }
            case ':': {
                return this.parseColon();
            }
            case '\"': {
                return this.parseString();
            }
            case '/': {
                return this.parseComment();
            }
            case '-': 
            case '0': 
            case '1': 
            case '2': 
            case '3': 
            case '4': 
            case '5': 
            case '6': 
            case '7': 
            case '8': 
            case '9': {
                return this.parseNumber();
            }
            case 'A': 
            case 'B': 
            case 'C': 
            case 'D': 
            case 'E': 
            case 'F': 
            case 'G': 
            case 'H': 
            case 'I': 
            case 'J': 
            case 'K': 
            case 'L': 
            case 'M': 
            case 'N': 
            case 'O': 
            case 'P': 
            case 'Q': 
            case 'R': 
            case 'S': 
            case 'T': 
            case 'U': 
            case 'V': 
            case 'W': 
            case 'X': 
            case 'Y': 
            case 'Z': 
            case '_': 
            case 'a': 
            case 'b': 
            case 'c': 
            case 'd': 
            case 'e': 
            case 'f': 
            case 'g': 
            case 'h': 
            case 'i': 
            case 'j': 
            case 'k': 
            case 'l': 
            case 'm': 
            case 'n': 
            case 'o': 
            case 'p': 
            case 'q': 
            case 'r': 
            case 's': 
            case 't': 
            case 'u': 
            case 'v': 
            case 'w': 
            case 'x': 
            case 'y': 
            case 'z': {
                return this.parseIdentifier();
            }
        }
        this.currentTokenError = "Unexpected character: '" + (char)c + '\'';
        return this.singleCharToken(IdlToken.ERROR);
    }

    public void skipSpaces() {
        this.getCurrentToken();
        while (this.currentTokenType == IdlToken.SPACE) {
            this.next();
        }
    }

    public void skipWs() {
        this.getCurrentToken();
        while (this.currentTokenType.isWhitespace()) {
            this.next();
        }
    }

    public void skipWsAndDocs() {
        this.getCurrentToken();
        while (this.currentTokenType.isWhitespace() || this.currentTokenType == IdlToken.DOC_COMMENT) {
            this.next();
        }
    }

    public boolean doesCurrentIdentifierStartWith(char c) {
        this.getCurrentToken();
        return this.currentTokenType == IdlToken.IDENTIFIER && this.currentTokenEnd > this.currentTokenStart && this.parser.input().charAt(this.currentTokenStart) == c;
    }

    public CharSequence expectCurrentLexeme(CharSequence chars) {
        boolean isError;
        CharSequence lexeme = this.getCurrentTokenLexeme();
        boolean bl = isError = lexeme.length() != chars.length();
        if (!isError) {
            for (int i = 0; i < chars.length(); ++i) {
                if (lexeme.charAt(i) == chars.charAt(i)) continue;
                isError = true;
                break;
            }
        }
        if (isError) {
            throw this.syntax("Expected `" + chars + "`, but found `" + lexeme + "`", this.getCurrentTokenLocation());
        }
        return lexeme;
    }

    public void expect(IdlToken token) {
        this.getCurrentToken();
        if (this.currentTokenType != token) {
            throw this.syntax(this.createExpectMessage(token), this.getCurrentTokenLocation());
        }
    }

    public IdlToken expect(IdlToken ... tokens) {
        this.getCurrentToken();
        if (this.currentTokenType == IdlToken.ERROR) {
            throw this.syntax(this.createExpectMessage(tokens), this.getCurrentTokenLocation());
        }
        for (IdlToken token : tokens) {
            if (this.currentTokenType != token) continue;
            return token;
        }
        throw this.syntax(this.createExpectMessage(tokens), this.getCurrentTokenLocation());
    }

    private String createExpectMessage(IdlToken ... tokens) {
        StringBuilder result = new StringBuilder();
        if (this.currentTokenType == IdlToken.ERROR) {
            result.append(this.getCurrentTokenError());
        } else if (tokens.length == 1) {
            result.append("Expected ").append(tokens[0].getDebug()).append(" but found ").append(this.getCurrentToken().getDebug(this.getCurrentTokenLexeme()));
        } else {
            result.append("Expected one of ");
            for (IdlToken token : tokens) {
                result.append(token.getDebug()).append(", ");
            }
            result.delete(result.length() - 2, result.length());
            result.append("; but found ").append(this.getCurrentToken().getDebug(this.getCurrentTokenLexeme()));
        }
        return result.toString();
    }

    public void expectAndSkipSpaces() {
        if (this.getCurrentToken() != IdlToken.SPACE) {
            throw this.syntax("Expected one or more spaces, but found " + this.getCurrentToken().getDebug(this.getCurrentTokenLexeme()), this.getCurrentTokenLocation());
        }
        this.skipSpaces();
    }

    private ModelSyntaxException syntax(String message, SourceLocation location) {
        return new ModelSyntaxException("Syntax error at line " + location.getLine() + ", column " + location.getColumn() + ": " + message, location);
    }

    public void expectAndSkipWhitespace() {
        if (!this.getCurrentToken().isWhitespace()) {
            throw this.syntax("Expected one or more whitespace characters, but found " + this.getCurrentToken().getDebug(this.getCurrentTokenLexeme()), this.getCurrentTokenLocation());
        }
        this.skipWsAndDocs();
    }

    public void expectAndSkipBr() {
        this.getCurrentToken();
        while (this.currentTokenType.canSkipBeforeBr()) {
            this.next();
        }
        switch (this.getCurrentToken()) {
            case NEWLINE: 
            case COMMENT: 
            case DOC_COMMENT: {
                this.clearDocCommentLinesForBr();
                this.next();
                this.skipWs();
                break;
            }
            case EOF: {
                break;
            }
            default: {
                throw this.syntax("Expected a line break, but found " + this.getCurrentToken().getDebug(this.getCurrentTokenLexeme()), this.getCurrentTokenLocation());
            }
        }
    }

    private void clearDocCommentLinesForBr() {
        if (!this.docCommentLines.isEmpty()) {
            this.validationEventListener.accept(LoaderUtils.emitBadDocComment(this.getCurrentTokenLocation(), this.removePendingDocCommentLines()));
        }
    }

    private IdlToken singleCharToken(IdlToken type) {
        this.parser.skip();
        this.currentTokenEnd = this.parser.position();
        this.currentTokenType = type;
        return this.currentTokenType;
    }

    private IdlToken tokenizeNewline() {
        this.parser.skip();
        this.currentTokenEnd = this.parser.position();
        this.currentTokenType = IdlToken.NEWLINE;
        return this.currentTokenType;
    }

    private IdlToken tokenizeSpace() {
        this.parser.consumeWhile(c -> c == 32 || c == 9);
        this.currentTokenEnd = this.parser.position();
        this.currentTokenType = IdlToken.SPACE;
        return this.currentTokenType;
    }

    private IdlToken parseColon() {
        this.parser.skip();
        if (this.parser.peek() == '=') {
            this.parser.skip();
            this.currentTokenType = IdlToken.WALRUS;
        } else {
            this.currentTokenType = IdlToken.COLON;
        }
        this.currentTokenEnd = this.parser.position();
        return this.currentTokenType;
    }

    private IdlToken parseComment() {
        this.parser.expect('/');
        if (this.parser.peek() != '/') {
            this.currentTokenError = "Expected a '/' to follow '/' to form a comment.";
            return this.singleCharToken(IdlToken.ERROR);
        }
        this.parser.expect('/');
        IdlToken type = IdlToken.COMMENT;
        if (this.parser.peek() == '/') {
            this.parser.expect('/');
            type = IdlToken.DOC_COMMENT;
            if (this.parser.peek() == ' ') {
                this.parser.skip();
            }
            int lineStart = this.parser.position();
            this.parser.consumeRemainingCharactersOnLine();
            this.docCommentLines.add(this.getInput(lineStart, this.parser.position()));
        } else {
            this.parser.consumeRemainingCharactersOnLine();
        }
        if (this.parser.expect(new char[]{'\r', '\n', '\u0000'}) == '\r' && this.parser.peek() == '\n') {
            this.parser.skip();
        }
        this.currentTokenEnd = this.parser.position();
        this.currentTokenType = type;
        return this.currentTokenType;
    }

    String removePendingDocCommentLines() {
        if (this.docCommentLines.isEmpty()) {
            return null;
        }
        StringBuilder result = new StringBuilder();
        result.append(this.docCommentLines.removeFirst());
        while (!this.docCommentLines.isEmpty()) {
            result.append('\n').append(this.docCommentLines.removeFirst());
        }
        return result.toString();
    }

    private IdlToken parseNumber() {
        try {
            String lexeme = ParserUtils.parseNumber(this.parser);
            if (lexeme.contains("e") || lexeme.contains("E") || lexeme.contains(".")) {
                double value = Double.parseDouble(lexeme);
                this.currentTokenNumber = Double.isFinite(value) ? Double.valueOf(value) : new BigDecimal(lexeme);
            } else {
                try {
                    this.currentTokenNumber = Long.parseLong(lexeme);
                }
                catch (NumberFormatException e) {
                    this.currentTokenNumber = new BigInteger(lexeme);
                }
            }
            this.currentTokenEnd = this.parser.position();
            this.currentTokenType = IdlToken.NUMBER;
            return this.currentTokenType;
        }
        catch (RuntimeException e) {
            this.currentTokenEnd = this.parser.position();
            this.currentTokenError = e.getMessage().startsWith("Syntax error") ? e.getMessage().substring(e.getMessage().indexOf(58) + 1).trim() : e.getMessage();
            this.currentTokenType = IdlToken.ERROR;
            return this.currentTokenType;
        }
    }

    private IdlToken parseIdentifier() {
        try {
            ParserUtils.consumeIdentifier(this.parser);
            this.currentTokenType = IdlToken.IDENTIFIER;
        }
        catch (RuntimeException e) {
            this.currentTokenType = IdlToken.ERROR;
            this.currentTokenError = e.getMessage();
        }
        this.currentTokenEnd = this.parser.position();
        return this.currentTokenType;
    }

    private IdlToken parseString() {
        this.parser.skip();
        if (this.parser.peek() == '\"') {
            this.parser.skip();
            if (this.parser.peek() == '\"') {
                this.parser.skip();
                return this.parseTextBlock();
            }
            this.currentTokenEnd = this.parser.position();
            this.currentTokenStringSlice = "";
            this.currentTokenType = IdlToken.STRING;
            return this.currentTokenType;
        }
        try {
            this.currentTokenStringSlice = this.parseQuotedTextAndTextBlock(false);
            this.currentTokenEnd = this.parser.position();
            this.currentTokenType = IdlToken.STRING;
            return this.currentTokenType;
        }
        catch (RuntimeException e) {
            this.currentTokenEnd = this.parser.position();
            this.currentTokenError = "Error parsing quoted string: " + e.getMessage();
            this.currentTokenType = IdlToken.ERROR;
            return this.currentTokenType;
        }
    }

    private IdlToken parseTextBlock() {
        try {
            this.currentTokenStringSlice = this.parseQuotedTextAndTextBlock(true);
            this.currentTokenEnd = this.parser.position();
            this.currentTokenType = IdlToken.TEXT_BLOCK;
            return this.currentTokenType;
        }
        catch (RuntimeException e) {
            this.currentTokenEnd = this.parser.position();
            this.currentTokenError = "Error parsing text block: " + e.getMessage();
            this.currentTokenType = IdlToken.ERROR;
            return this.currentTokenType;
        }
    }

    private CharSequence parseQuotedTextAndTextBlock(boolean triple) {
        char next;
        int start = this.parser.position();
        while (!this.parser.eof() && ((next = this.parser.peek()) != '\"' || triple && (this.parser.peek(1) != '\"' || this.parser.peek(2) != '\"'))) {
            this.parser.skip();
            if (next != '\\') continue;
            this.parser.skip();
        }
        CharSequence result = this.parser.borrowSliceFrom(start);
        this.parser.expect('\"');
        if (triple) {
            this.parser.expect('\"');
            this.parser.expect('\"');
        }
        return IdlStringLexer.scanStringContents(result, triple);
    }

    public static final class Builder
    implements SmithyBuilder<IdlTokenizer> {
        private String filename = SourceLocation.NONE.getFilename();
        private CharSequence model;
        private Function<CharSequence, String> stringTable = CharSequence::toString;
        private Consumer<ValidationEvent> validationEventListener = event -> {};

        private Builder() {
        }

        public Builder filename(String filename) {
            this.filename = Objects.requireNonNull(filename);
            return this;
        }

        public Builder model(CharSequence model) {
            this.model = model;
            return this;
        }

        public Builder stringTable(Function<CharSequence, String> stringTable) {
            this.stringTable = Objects.requireNonNull(stringTable);
            return this;
        }

        Builder validationEventListener(Consumer<ValidationEvent> validationEventListener) {
            this.validationEventListener = Objects.requireNonNull(validationEventListener);
            return this;
        }

        public IdlTokenizer build() {
            return new IdlTokenizer(this);
        }
    }
}

