/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.ApplyMixin;
import software.amazon.smithy.model.loader.ApplyResourceBasedTargets;
import software.amazon.smithy.model.loader.AstModelLoader;
import software.amazon.smithy.model.loader.IdlNodeParser;
import software.amazon.smithy.model.loader.IdlReferenceResolver;
import software.amazon.smithy.model.loader.IdlShapeIdParser;
import software.amazon.smithy.model.loader.IdlToken;
import software.amazon.smithy.model.loader.IdlTokenizer;
import software.amazon.smithy.model.loader.IdlTraitParser;
import software.amazon.smithy.model.loader.LoadOperation;
import software.amazon.smithy.model.loader.LoaderUtils;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.loader.Version;
import software.amazon.smithy.model.node.ArrayNode;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.EnumShape;
import software.amazon.smithy.model.shapes.IntEnumShape;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.ResourceShape;
import software.amazon.smithy.model.shapes.ServiceShape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeType;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.ToShapeId;
import software.amazon.smithy.model.traits.DefaultTrait;
import software.amazon.smithy.model.traits.DocumentationTrait;
import software.amazon.smithy.model.traits.EnumValueTrait;
import software.amazon.smithy.model.traits.InputTrait;
import software.amazon.smithy.model.traits.OutputTrait;
import software.amazon.smithy.model.traits.UnitTypeTrait;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.ListUtils;

final class IdlModelLoader {
    private static final String PUT_KEY = "put";
    private static final String CREATE_KEY = "create";
    private static final String READ_KEY = "read";
    private static final String UPDATE_KEY = "update";
    private static final String DELETE_KEY = "delete";
    private static final String LIST_KEY = "list";
    private static final String RESOURCES_KEY = "resources";
    private static final String OPERATIONS_KEY = "operations";
    private static final String PROPERTIES_KEY = "properties";
    private static final String RENAME_KEY = "rename";
    private static final String COLLECTION_OPERATIONS_KEY = "collectionOperations";
    private static final String IDENTIFIERS_KEY = "identifiers";
    private static final String VERSION_KEY = "version";
    private static final String TYPE_KEY = "type";
    private static final String ERRORS_KEY = "errors";
    static final Collection<String> RESOURCE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "create", "read", "update", "delete", "list", "identifiers", "resources", "operations", "put", "properties", "collectionOperations"});
    static final List<String> SERVICE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "version", "operations", "resources", "rename", "errors"});
    private static final Set<String> SHAPE_TYPES = new HashSet<String>();
    private final String filename;
    private final IdlTokenizer tokenizer;
    private final Map<String, ShapeId> useShapes = new HashMap<String, ShapeId>();
    private final IdlReferenceResolver resolver;
    private Consumer<LoadOperation> operations;
    private Version modelVersion = Version.VERSION_1_0;
    private String namespace;
    private boolean emittedVersion = false;
    private String operationInputSuffix = "Input";
    private String operationOutputSuffix = "Output";

    IdlModelLoader(String filename, CharSequence model, Function<CharSequence, String> stringTable) {
        this.filename = filename;
        this.tokenizer = IdlTokenizer.builder().filename(filename).model(model).stringTable(stringTable).validationEventListener(this::emit).build();
        this.resolver = this::addForwardReference;
    }

    void parse(Consumer<LoadOperation> operationConsumer) {
        this.operations = operationConsumer;
        this.tokenizer.skipWsAndDocs();
        this.parseControlSection();
        if (!this.emittedVersion) {
            this.addOperation(new LoadOperation.ModelVersion(this.modelVersion, this.tokenizer.getCurrentTokenLocation()));
        }
        this.tokenizer.skipWsAndDocs();
        this.parseMetadataSection();
        this.parseShapeSection();
    }

    void emit(ValidationEvent event) {
        this.addOperation(new LoadOperation.Event(event));
    }

    void addOperation(LoadOperation operation) {
        this.operations.accept(operation);
    }

    public ModelSyntaxException syntax(String message) {
        return this.syntax(null, message);
    }

    ModelSyntaxException syntax(ShapeId shapeId, String message) {
        return ModelSyntaxException.builder().message(String.format("Syntax error at line %d, column %d: %s", this.tokenizer.getCurrentTokenLine(), this.tokenizer.getCurrentTokenColumn(), message)).sourceLocation(this.tokenizer.getCurrentTokenLocation()).shapeId(shapeId).build();
    }

    void addForwardReference(String id, BiFunction<ShapeId, ShapeType, ValidationEvent> receiver) {
        int memberPosition = id.indexOf(36);
        if (memberPosition > 0 && memberPosition < id.length() - 1) {
            this.addForwardReference(id.substring(0, memberPosition), (ShapeId resolved, ShapeType type) -> (ValidationEvent)receiver.apply(resolved.withMember(id.substring(memberPosition + 1)), (ShapeType)((Object)type)));
        } else {
            String resolved2 = this.useShapes.containsKey(id) ? this.useShapes.get(id).toString() : id;
            this.addOperation(new LoadOperation.ForwardReference(this.namespace, resolved2, receiver));
        }
    }

    void addForwardReference(String id, Consumer<ShapeId> consumer) {
        this.addForwardReference(id, (ShapeId resolved, ShapeType found) -> {
            consumer.accept((ShapeId)resolved);
            return null;
        });
    }

    String expectNamespace() {
        if (this.namespace == null) {
            throw new IllegalStateException("No namespace was set before trying to resolve a forward reference");
        }
        return this.namespace;
    }

    private void onDeferredTrait(ShapeId target, String traitName, Node traitValue, boolean isAnnotation) {
        this.addForwardReference(traitName, (ShapeId traitId, ShapeType type) -> {
            Node coerced = this.coerceTraitValue(traitValue, isAnnotation, (ShapeType)((Object)type));
            this.addOperation(new LoadOperation.ApplyTrait(this.modelVersion, traitValue.getSourceLocation(), this.expectNamespace(), target, (ShapeId)traitId, coerced));
            return null;
        });
    }

    private Node coerceTraitValue(Node value, boolean isAnnotation, ShapeType targetType) {
        if (isAnnotation && value.isNullNode()) {
            if (targetType == null || targetType == ShapeType.STRUCTURE || targetType == ShapeType.MAP) {
                return new ObjectNode(Collections.emptyMap(), value.getSourceLocation());
            }
            if (targetType == ShapeType.LIST || targetType == ShapeType.SET) {
                return new ArrayNode(Collections.emptyList(), value.getSourceLocation());
            }
        }
        return value;
    }

    private void parseControlSection() {
        HashSet<String> definedKeys = new HashSet<String>();
        while (this.tokenizer.getCurrentToken() == IdlToken.DOLLAR) {
            try {
                this.tokenizer.next();
                this.tokenizer.expect(IdlToken.IDENTIFIER, IdlToken.STRING);
                String key = this.tokenizer.internString(this.tokenizer.getCurrentTokenStringSlice());
                this.tokenizer.next();
                this.tokenizer.skipSpaces();
                this.tokenizer.expect(IdlToken.COLON);
                this.tokenizer.next();
                this.tokenizer.skipSpaces();
                if (!definedKeys.add(key)) {
                    throw this.syntax(String.format("Duplicate control statement `%s`", key));
                }
                Node value = IdlNodeParser.expectAndSkipNode(this.tokenizer, this.resolver);
                switch (key) {
                    case "version": {
                        this.onVersion(value);
                        break;
                    }
                    case "operationInputSuffix": {
                        this.operationInputSuffix = value.expectStringNode().getValue();
                        break;
                    }
                    case "operationOutputSuffix": {
                        this.operationOutputSuffix = value.expectStringNode().getValue();
                        break;
                    }
                    default: {
                        this.emit(ValidationEvent.builder().id("Model").sourceLocation(value).severity(Severity.WARNING).message(String.format("Unknown control statement `%s` with value `%s", key, Node.printJson(value))).build());
                    }
                }
                this.tokenizer.expectAndSkipBr();
            }
            catch (ModelSyntaxException e) {
                this.errorRecovery(e);
            }
        }
    }

    private void onVersion(Node value) {
        String parsedVersion;
        Version resolvedVersion;
        if (!value.isStringNode()) {
            value.expectStringNode(() -> "The $version control statement must have a string value, but found " + Node.printJson(value));
        }
        if ((resolvedVersion = Version.fromString(parsedVersion = value.expectStringNode().getValue())) == null) {
            throw this.syntax("Unsupported Smithy version number: " + parsedVersion);
        }
        this.emittedVersion = true;
        this.modelVersion = resolvedVersion;
        this.addOperation(new LoadOperation.ModelVersion(this.modelVersion, value.getSourceLocation()));
    }

    private void parseMetadataSection() {
        while (this.tokenizer.doesCurrentIdentifierStartWith('m')) {
            try {
                this.tokenizer.expectCurrentLexeme("metadata");
                this.tokenizer.next();
                this.tokenizer.expectAndSkipSpaces();
                this.tokenizer.expect(IdlToken.IDENTIFIER, IdlToken.STRING);
                String key = this.tokenizer.internString(this.tokenizer.getCurrentTokenStringSlice());
                this.tokenizer.next();
                this.tokenizer.skipSpaces();
                this.tokenizer.expect(IdlToken.EQUAL);
                this.tokenizer.next();
                this.tokenizer.skipSpaces();
                Node value = IdlNodeParser.expectAndSkipNode(this.tokenizer, this.resolver);
                this.operations.accept(new LoadOperation.PutMetadata(this.modelVersion, key, value));
                this.tokenizer.expectAndSkipBr();
            }
            catch (ModelSyntaxException e) {
                this.errorRecovery(e);
            }
        }
    }

    private void parseShapeSection() {
        if (this.tokenizer.doesCurrentIdentifierStartWith('n')) {
            this.tokenizer.expectCurrentLexeme("namespace");
            this.tokenizer.next();
            this.tokenizer.expectAndSkipSpaces();
            this.namespace = this.tokenizer.internString(IdlShapeIdParser.expectAndSkipShapeIdNamespace(this.tokenizer));
            this.tokenizer.expectAndSkipBr();
            SourceLocation possibleDocCommentLocation = this.tokenizer.getCurrentTokenLocation();
            this.tokenizer.skipWsAndDocs();
            this.parseUseSection();
            this.parseFirstShapeStatement(possibleDocCommentLocation);
            this.parseSubsequentShapeStatements();
        } else if (this.tokenizer.hasNext()) {
            throw this.syntax("Expected a namespace definition but found " + this.tokenizer.getCurrentToken().getDebug(this.tokenizer.getCurrentTokenLexeme()));
        }
    }

    private void parseUseSection() {
        String keyword;
        while (this.tokenizer.getCurrentToken() == IdlToken.IDENTIFIER && (keyword = this.tokenizer.internString(this.tokenizer.getCurrentTokenLexeme())).equals("use")) {
            this.tokenizer.next();
            this.tokenizer.expectAndSkipSpaces();
            SourceLocation idLocation = this.tokenizer.getCurrentTokenLocation();
            String idString = this.tokenizer.internString(IdlShapeIdParser.expectAndSkipAbsoluteShapeId(this.tokenizer));
            ShapeId id = ShapeId.from(idString);
            if (id.hasMember()) {
                throw new ModelSyntaxException("Use statements cannot use members", idLocation);
            }
            if (this.useShapes.containsKey(id.getName())) {
                ShapeId previous = this.useShapes.get(id.getName());
                String message = String.format("Cannot use name `%s` because it conflicts with `%s`", id, previous);
                throw new ModelSyntaxException(message, idLocation);
            }
            this.useShapes.put(id.getName(), id);
            this.addForwardReference(idString, (ShapeId resolved, ShapeType type) -> {
                if (type != null) {
                    return null;
                }
                return ValidationEvent.builder().id("Model").severity(Severity.WARNING).sourceLocation(idLocation).message("Use statement refers to undefined shape: " + id).build();
            });
            this.tokenizer.expectAndSkipBr();
            this.tokenizer.skipWsAndDocs();
        }
    }

    private void parseApplyStatement(List<IdlTraitParser.Result> traits) {
        List<IdlTraitParser.Result> traitsToApply;
        SourceLocation foundDocComments = null;
        for (IdlTraitParser.Result trait : traits) {
            if (trait.getTraitType() != IdlTraitParser.TraitType.DOC_COMMENT) {
                throw this.syntax("Traits applied to apply statement");
            }
            foundDocComments = trait.getValue().getSourceLocation();
        }
        if (foundDocComments != null) {
            LoaderUtils.emitBadDocComment(foundDocComments, null);
        }
        this.tokenizer.expectCurrentLexeme("apply");
        this.tokenizer.next();
        this.tokenizer.expectAndSkipWhitespace();
        String target = this.tokenizer.internString(IdlShapeIdParser.expectAndSkipShapeId(this.tokenizer));
        this.tokenizer.expectAndSkipWhitespace();
        if (IdlToken.AT == this.tokenizer.expect(IdlToken.AT, IdlToken.LBRACE)) {
            traitsToApply = Collections.singletonList(IdlTraitParser.expectAndSkipTrait(this.tokenizer, this.resolver));
        } else {
            this.tokenizer.next();
            this.tokenizer.skipWsAndDocs();
            traitsToApply = IdlTraitParser.expectAndSkipTraits(this.tokenizer, this.resolver);
            this.tokenizer.skipWsAndDocs();
            this.tokenizer.expect(IdlToken.RBRACE);
            this.tokenizer.next();
        }
        this.addForwardReference(target, (ShapeId id) -> {
            for (IdlTraitParser.Result trait : traitsToApply) {
                String traitNameString = this.tokenizer.internString(trait.getTraitName());
                this.onDeferredTrait((ShapeId)id, traitNameString, trait.getValue(), trait.getTraitType() == IdlTraitParser.TraitType.ANNOTATION);
            }
        });
        this.tokenizer.expectAndSkipBr();
    }

    private void parseFirstShapeStatement(SourceLocation possibleDocCommentLocation) {
        if (this.tokenizer.getCurrentToken() != IdlToken.EOF) {
            try {
                this.tokenizer.skipWsAndDocs();
                String docLines = this.tokenizer.removePendingDocCommentLines();
                List<IdlTraitParser.Result> traits = IdlTraitParser.parseDocsAndTraitsBeforeShape(this.tokenizer, this.resolver);
                if (docLines != null) {
                    traits.add(new IdlTraitParser.Result(DocumentationTrait.ID.toString(), new StringNode(docLines, possibleDocCommentLocation), IdlTraitParser.TraitType.DOC_COMMENT));
                }
                if (this.parseShapeDefinition(traits, docLines != null)) {
                    this.parseShapeOrApply(traits);
                }
            }
            catch (ModelSyntaxException e) {
                this.errorRecovery(e);
            }
        }
    }

    private void parseSubsequentShapeStatements() {
        while (this.tokenizer.hasNext()) {
            try {
                boolean hasDocComment;
                boolean bl = hasDocComment = this.tokenizer.getCurrentToken() == IdlToken.DOC_COMMENT;
                List<IdlTraitParser.Result> traits = IdlTraitParser.parseDocsAndTraitsBeforeShape(this.tokenizer, this.resolver);
                if (!this.parseShapeDefinition(traits, hasDocComment)) continue;
                this.parseShapeOrApply(traits);
            }
            catch (ModelSyntaxException e) {
                this.errorRecovery(e);
            }
        }
    }

    private void errorRecovery(ModelSyntaxException e) {
        IdlToken token;
        if (!this.tokenizer.hasNext()) {
            throw e;
        }
        this.emit(ValidationEvent.fromSourceException(e));
        do {
            token = this.tokenizer.next();
        } while ((this.tokenizer.getCurrentTokenColumn() != 1 || !this.isErrorRecoveryToken(token)) && this.tokenizer.hasNext());
    }

    private boolean isErrorRecoveryToken(IdlToken token) {
        switch (token) {
            case IDENTIFIER: 
            case DOC_COMMENT: 
            case AT: 
            case DOLLAR: {
                return true;
            }
        }
        return false;
    }

    private boolean parseShapeDefinition(List<IdlTraitParser.Result> traits, boolean hasDocComment) {
        if (this.tokenizer.getCurrentToken() != IdlToken.EOF) {
            return true;
        }
        if (hasDocComment) {
            if (traits.size() == 1) {
                this.emit(LoaderUtils.emitBadDocComment(this.tokenizer.getCurrentTokenLocation(), traits.get(0).getValue().expectStringNode().getValue()));
                return false;
            }
            return true;
        }
        return !traits.isEmpty();
    }

    private void parseShapeOrApply(List<IdlTraitParser.Result> traits) {
        SourceLocation location = this.tokenizer.getCurrentTokenLocation();
        this.tokenizer.expect(IdlToken.IDENTIFIER);
        String shapeType = this.tokenizer.internString(this.tokenizer.getCurrentTokenLexeme());
        if (shapeType.equals("apply")) {
            this.parseApplyStatement(traits);
            return;
        }
        ShapeType type = ShapeType.fromString(shapeType).orElseThrow(() -> this.syntax("Unknown shape type: " + shapeType));
        this.tokenizer.next();
        this.tokenizer.expectAndSkipSpaces();
        ShapeId id = this.parseShapeName();
        switch (type) {
            case STRING: 
            case BLOB: 
            case BOOLEAN: 
            case DOCUMENT: 
            case BYTE: 
            case SHORT: 
            case INTEGER: 
            case LONG: 
            case FLOAT: 
            case DOUBLE: 
            case BIG_DECIMAL: 
            case BIG_INTEGER: 
            case TIMESTAMP: {
                this.parseSimpleShape(id, location, type.createBuilderForType());
                break;
            }
            case LIST: 
            case SET: 
            case MAP: 
            case UNION: 
            case STRUCTURE: {
                this.parseAggregateShape(id, location, type.createBuilderForType());
                break;
            }
            case ENUM: {
                this.parseEnumShape(id, location, EnumShape.builder());
                break;
            }
            case INT_ENUM: {
                this.parseEnumShape(id, location, IntEnumShape.builder());
                break;
            }
            case SERVICE: {
                this.parseServiceStatement(id, location);
                break;
            }
            case RESOURCE: {
                this.parseResourceStatement(id, location);
                break;
            }
            case OPERATION: {
                this.parseOperationStatement(id, location);
                break;
            }
            default: {
                throw this.syntax("Shape type unknown: " + shapeType);
            }
        }
        this.addTraits(id, traits);
        this.tokenizer.expectAndSkipBr();
    }

    private void addTraits(ShapeId id, List<IdlTraitParser.Result> traits) {
        for (IdlTraitParser.Result result : traits) {
            String traitName = this.tokenizer.internString(result.getTraitName());
            this.onDeferredTrait(id, traitName, result.getValue(), result.getTraitType() == IdlTraitParser.TraitType.ANNOTATION);
        }
    }

    private ShapeId parseShapeName() {
        int line = this.tokenizer.getCurrentTokenLine();
        int column = this.tokenizer.getCurrentTokenColumn();
        this.tokenizer.expect(IdlToken.IDENTIFIER);
        String name = this.tokenizer.internString(this.tokenizer.getCurrentTokenStringSlice());
        ShapeId id = ShapeId.fromRelative(this.expectNamespace(), name);
        if (this.useShapes.containsKey(name)) {
            ShapeId previous = this.useShapes.get(name);
            String message = String.format("Shape name `%s` conflicts with imported shape `%s`", name, previous);
            throw new ModelSyntaxException(message, this.filename, line, column);
        }
        this.tokenizer.next();
        return id;
    }

    private void parseSimpleShape(ShapeId id, SourceLocation location, AbstractShapeBuilder<?, ?> builder) {
        LoadOperation.DefineShape operation = this.createShape((AbstractShapeBuilder<?, ?>)((AbstractShapeBuilder)builder.source(location)).id(id));
        this.parseMixins(operation);
        this.addOperation(operation);
    }

    LoadOperation.DefineShape createShape(AbstractShapeBuilder<?, ?> builder) {
        return new LoadOperation.DefineShape(this.modelVersion, builder);
    }

    private void parseMixins(LoadOperation.DefineShape operation) {
        this.tokenizer.skipSpaces();
        if (!this.tokenizer.doesCurrentIdentifierStartWith('w')) {
            return;
        }
        this.tokenizer.expect(IdlToken.IDENTIFIER);
        this.tokenizer.expectCurrentLexeme("with");
        if (!this.modelVersion.supportsMixins()) {
            throw this.syntax(operation.toShapeId(), "Mixins can only be used with Smithy version 2 or later. Attempted to use mixins with version `" + (Object)((Object)this.modelVersion) + "`.");
        }
        this.tokenizer.next();
        this.tokenizer.skipWsAndDocs();
        this.tokenizer.expect(IdlToken.LBRACKET);
        this.tokenizer.next();
        this.tokenizer.skipWsAndDocs();
        do {
            String target = this.tokenizer.internString(IdlShapeIdParser.expectAndSkipShapeId(this.tokenizer));
            this.addForwardReference(target, (ShapeId resolved) -> {
                operation.addDependency((ShapeId)resolved);
                operation.addModifier(new ApplyMixin((ShapeId)resolved));
            });
            this.tokenizer.skipWsAndDocs();
        } while (this.tokenizer.getCurrentToken() != IdlToken.RBRACKET);
        this.tokenizer.expect(IdlToken.RBRACKET);
        this.tokenizer.next();
    }

    private void parseEnumShape(ShapeId id, SourceLocation location, AbstractShapeBuilder<?, ?> builder) {
        LoadOperation.DefineShape operation = this.createShape((AbstractShapeBuilder<?, ?>)((AbstractShapeBuilder)builder.id(id)).source(location));
        this.parseMixins(operation);
        this.tokenizer.skipWsAndDocs();
        this.tokenizer.expect(IdlToken.LBRACE);
        this.tokenizer.next();
        this.tokenizer.skipWs();
        while (this.tokenizer.getCurrentToken() != IdlToken.EOF && this.tokenizer.getCurrentToken() != IdlToken.RBRACE) {
            List<IdlTraitParser.Result> memberTraits = IdlTraitParser.parseDocsAndTraitsBeforeShape(this.tokenizer, this.resolver);
            SourceLocation memberLocation = this.tokenizer.getCurrentTokenLocation();
            this.tokenizer.expect(IdlToken.IDENTIFIER);
            String memberName = this.tokenizer.internString(this.tokenizer.getCurrentTokenLexeme());
            MemberShape.Builder memberBuilder = ((MemberShape.Builder)((MemberShape.Builder)MemberShape.builder().id(id.withMember(memberName))).source(memberLocation)).target(UnitTypeTrait.UNIT);
            operation.addMember(memberBuilder);
            this.addTraits(memberBuilder.getId(), memberTraits);
            this.tokenizer.next();
            this.tokenizer.skipSpaces();
            if (this.tokenizer.getCurrentToken() == IdlToken.EQUAL) {
                this.tokenizer.next();
                this.tokenizer.skipSpaces();
                Node value = IdlNodeParser.expectAndSkipNode(this.tokenizer, this.resolver);
                memberBuilder.addTrait(new EnumValueTrait.Provider().createTrait(memberBuilder.getId(), value));
                this.tokenizer.expectAndSkipBr();
                continue;
            }
            this.tokenizer.skipWs();
        }
        this.tokenizer.expect(IdlToken.RBRACE);
        this.tokenizer.next();
        this.operations.accept(operation);
    }

    private void parseAggregateShape(ShapeId id, SourceLocation location, AbstractShapeBuilder<?, ?> builder) {
        LoadOperation.DefineShape operation = this.createShape((AbstractShapeBuilder<?, ?>)((AbstractShapeBuilder)builder.id(id)).source(location));
        this.parseForResource(operation);
        this.parseMixins(operation);
        this.parseMembers(operation);
        this.operations.accept(operation);
    }

    private void parseMembers(LoadOperation.DefineShape op) {
        HashSet<String> definedMembers = new HashSet<String>();
        this.tokenizer.skipWsAndDocs();
        this.tokenizer.expect(IdlToken.LBRACE);
        this.tokenizer.next();
        this.tokenizer.skipWs();
        while (this.tokenizer.hasNext() && this.tokenizer.getCurrentToken() != IdlToken.RBRACE) {
            this.parseMember(op, definedMembers);
            this.tokenizer.skipWs();
        }
        this.tokenizer.expect(IdlToken.RBRACE);
        this.tokenizer.next();
    }

    private void parseMember(LoadOperation.DefineShape operation, Set<String> defined) {
        boolean isTargetElided;
        ShapeId parent = operation.toShapeId();
        List<IdlTraitParser.Result> memberTraits = IdlTraitParser.parseDocsAndTraitsBeforeShape(this.tokenizer, this.resolver);
        SourceLocation memberLocation = this.tokenizer.getCurrentTokenLocation();
        if (this.tokenizer.getCurrentToken() == IdlToken.RBRACE && memberTraits.size() == 1 && memberTraits.get(0).getTraitType() == IdlTraitParser.TraitType.DOC_COMMENT) {
            IdlTraitParser.Result danglingDocComment = memberTraits.get(0);
            SourceLocation sourceLocation = danglingDocComment.getValue().getSourceLocation();
            String value = danglingDocComment.getValue().toString();
            this.emit(LoaderUtils.emitBadDocComment(sourceLocation, value));
            return;
        }
        boolean bl = isTargetElided = this.tokenizer.getCurrentToken() == IdlToken.DOLLAR;
        if (isTargetElided) {
            this.tokenizer.expect(IdlToken.DOLLAR);
            this.tokenizer.next();
        }
        this.tokenizer.expect(IdlToken.IDENTIFIER);
        String memberName = this.tokenizer.internString(this.tokenizer.getCurrentTokenLexeme());
        if (defined.contains(memberName)) {
            throw this.syntax(parent, "Duplicate member of `" + parent + "`: '" + memberName + '\'');
        }
        defined.add(memberName);
        ShapeId memberId = parent.withMember(memberName);
        if (isTargetElided && !this.modelVersion.supportsTargetElision()) {
            throw this.syntax(memberId, "Members can only elide targets in IDL version 2 or later");
        }
        MemberShape.Builder memberBuilder = (MemberShape.Builder)((MemberShape.Builder)MemberShape.builder().id(memberId)).source(memberLocation);
        this.tokenizer.next();
        if (!isTargetElided) {
            this.tokenizer.skipSpaces();
            this.tokenizer.expect(IdlToken.COLON);
            this.tokenizer.next();
            this.tokenizer.skipSpaces();
            String target = this.tokenizer.internString(IdlShapeIdParser.expectAndSkipShapeId(this.tokenizer));
            this.addForwardReference(target, memberBuilder::target);
        }
        this.tokenizer.skipSpaces();
        if (this.tokenizer.getCurrentToken() == IdlToken.EQUAL) {
            if (!this.modelVersion.isDefaultSupported()) {
                throw this.syntax("@default assignment is only supported in IDL version 2 or later");
            }
            this.tokenizer.expect(IdlToken.EQUAL);
            this.tokenizer.next();
            this.tokenizer.skipSpaces();
            Node node = IdlNodeParser.expectAndSkipNode(this.tokenizer, this.resolver);
            memberBuilder.addTrait(new DefaultTrait(node));
            this.tokenizer.expectAndSkipBr();
        }
        operation.addMember(memberBuilder);
        this.addTraits(memberBuilder.getId(), memberTraits);
    }

    private void parseForResource(LoadOperation.DefineShape operation) {
        this.tokenizer.skipSpaces();
        if (!this.tokenizer.doesCurrentIdentifierStartWith('f')) {
            return;
        }
        this.tokenizer.expectCurrentLexeme("for");
        if (!this.modelVersion.supportsTargetElision()) {
            throw this.syntax(operation.toShapeId(), "Structures can only be bound to resources with Smithy version 2 or later. Attempted to bind a structure to a resource with version `" + (Object)((Object)this.modelVersion) + "`.");
        }
        this.tokenizer.next();
        this.tokenizer.expectAndSkipSpaces();
        String forTarget = this.tokenizer.internString(IdlShapeIdParser.expectAndSkipShapeId(this.tokenizer));
        this.addForwardReference(forTarget, (ShapeId shapeId) -> {
            operation.addDependency((ShapeId)shapeId);
            operation.addModifier(new ApplyResourceBasedTargets((ShapeId)shapeId));
        });
    }

    private void parseServiceStatement(ShapeId id, SourceLocation location) {
        ServiceShape.Builder builder = (ServiceShape.Builder)((ServiceShape.Builder)new ServiceShape.Builder().id(id)).source(location);
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.parseMixins(operation);
        this.tokenizer.skipWsAndDocs();
        this.tokenizer.expect(IdlToken.LBRACE);
        ObjectNode shapeNode = IdlNodeParser.expectAndSkipNode(this.tokenizer, this.resolver).expectObjectNode();
        LoaderUtils.checkForAdditionalProperties(shapeNode, id, SERVICE_PROPERTY_NAMES).ifPresent(this::emit);
        shapeNode.getStringMember(VERSION_KEY).map(StringNode::getValue).ifPresent(builder::version);
        this.optionalIdList(shapeNode, OPERATIONS_KEY, builder::addOperation);
        this.optionalIdList(shapeNode, RESOURCES_KEY, builder::addResource);
        this.optionalIdList(shapeNode, ERRORS_KEY, builder::addError);
        AstModelLoader.loadServiceRenameIntoBuilder(builder, shapeNode);
        this.operations.accept(operation);
    }

    private void optionalId(ObjectNode node, String name, Consumer<ShapeId> consumer) {
        if (node.getMember(name).isPresent()) {
            this.addForwardReference(node.expectStringMember(name).getValue(), consumer);
        }
    }

    private void optionalIdList(ObjectNode node, String name, Consumer<ShapeId> consumer) {
        if (node.getMember(name).isPresent()) {
            ArrayNode value = node.expectArrayMember(name);
            for (StringNode element : value.getElementsAs(StringNode.class)) {
                this.addForwardReference(element.getValue(), consumer);
            }
        }
    }

    private void parseResourceStatement(ShapeId id, SourceLocation location) {
        ResourceShape.Builder builder = (ResourceShape.Builder)((ResourceShape.Builder)ResourceShape.builder().id(id)).source(location);
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.parseMixins(operation);
        this.tokenizer.skipWsAndDocs();
        this.tokenizer.expect(IdlToken.LBRACE);
        ObjectNode shapeNode = IdlNodeParser.expectAndSkipNode(this.tokenizer, this.resolver).expectObjectNode();
        LoaderUtils.checkForAdditionalProperties(shapeNode, id, RESOURCE_PROPERTY_NAMES).ifPresent(this::emit);
        this.optionalId(shapeNode, PUT_KEY, builder::put);
        this.optionalId(shapeNode, CREATE_KEY, builder::create);
        this.optionalId(shapeNode, READ_KEY, builder::read);
        this.optionalId(shapeNode, UPDATE_KEY, builder::update);
        this.optionalId(shapeNode, DELETE_KEY, builder::delete);
        this.optionalId(shapeNode, LIST_KEY, builder::list);
        this.optionalIdList(shapeNode, OPERATIONS_KEY, builder::addOperation);
        this.optionalIdList(shapeNode, RESOURCES_KEY, builder::addResource);
        this.optionalIdList(shapeNode, COLLECTION_OPERATIONS_KEY, builder::addCollectionOperation);
        shapeNode.getObjectMember(IDENTIFIERS_KEY).ifPresent(ids -> {
            for (Map.Entry<StringNode, Node> entry : ids.getMembers().entrySet()) {
                String name = entry.getKey().getValue();
                StringNode target = entry.getValue().expectStringNode();
                this.addForwardReference(target.getValue(), (ShapeId targetId) -> builder.addIdentifier(name, (ToShapeId)targetId));
            }
        });
        shapeNode.getObjectMember(PROPERTIES_KEY).ifPresent(properties -> {
            if (!this.modelVersion.supportsResourceProperties()) {
                throw this.syntax(id, "Resource properties can only be used with Smithy version 2 or later. Attempted to use resource properties with version `" + (Object)((Object)this.modelVersion) + "`.");
            }
            for (Map.Entry<StringNode, Node> entry : properties.getMembers().entrySet()) {
                String name = entry.getKey().getValue();
                StringNode target = entry.getValue().expectStringNode();
                this.addForwardReference(target.getValue(), (ShapeId targetId) -> builder.addProperty(name, (ToShapeId)targetId));
            }
        });
        this.operations.accept(operation);
    }

    private void parseOperationStatement(ShapeId id, SourceLocation location) {
        OperationShape.Builder builder = (OperationShape.Builder)((OperationShape.Builder)OperationShape.builder().id(id)).source(location);
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.parseMixins(operation);
        this.tokenizer.skipWsAndDocs();
        this.tokenizer.expect(IdlToken.LBRACE);
        this.tokenizer.next();
        this.tokenizer.skipWsAndDocs();
        HashSet<String> defined = new HashSet<String>();
        while (this.tokenizer.hasNext() && this.tokenizer.getCurrentToken() != IdlToken.RBRACE) {
            this.tokenizer.expect(IdlToken.IDENTIFIER);
            String key = this.tokenizer.internString(this.tokenizer.getCurrentTokenLexeme());
            if (!defined.add(key)) {
                throw this.syntax(id, String.format("Duplicate operation %s property for `%s`", key, id));
            }
            this.tokenizer.next();
            this.tokenizer.skipWsAndDocs();
            switch (key) {
                case "input": {
                    IdlToken nextInput = this.tokenizer.expect(IdlToken.COLON, IdlToken.WALRUS);
                    this.tokenizer.next();
                    IdlTraitParser.Result inputTrait = new IdlTraitParser.Result(InputTrait.ID.toString(), Node.objectNode(), IdlTraitParser.TraitType.ANNOTATION);
                    this.parseInlineableOperationMember(id, nextInput, this.operationInputSuffix, builder::input, inputTrait);
                    break;
                }
                case "output": {
                    IdlToken nextOutput = this.tokenizer.expect(IdlToken.COLON, IdlToken.WALRUS);
                    this.tokenizer.next();
                    IdlTraitParser.Result outputTrait = new IdlTraitParser.Result(OutputTrait.ID.toString(), Node.objectNode(), IdlTraitParser.TraitType.ANNOTATION);
                    this.parseInlineableOperationMember(id, nextOutput, this.operationOutputSuffix, builder::output, outputTrait);
                    break;
                }
                case "errors": {
                    this.tokenizer.expect(IdlToken.COLON);
                    this.tokenizer.next();
                    this.parseIdList(builder::addError);
                    break;
                }
                default: {
                    throw this.syntax(id, String.format("Unknown property %s for %s", key, id));
                }
            }
            this.tokenizer.expectAndSkipWhitespace();
            this.tokenizer.skipWsAndDocs();
        }
        this.tokenizer.expect(IdlToken.RBRACE);
        this.tokenizer.next();
        this.operations.accept(operation);
    }

    private void parseInlineableOperationMember(ShapeId id, IdlToken token, String suffix, Consumer<ShapeId> consumer, IdlTraitParser.Result defaultTrait) {
        if (token == IdlToken.WALRUS) {
            if (!this.modelVersion.supportsInlineOperationIO()) {
                throw this.syntax(id, "Inlined operation inputs and outputs can only be used with Smithy version 2 or later. Attempted to use inlined IO with version `" + (Object)((Object)this.modelVersion) + "`.");
            }
            this.tokenizer.removePendingDocCommentLines();
            this.tokenizer.skipWs();
            consumer.accept(this.parseInlineStructure(id.getName() + suffix, defaultTrait));
        } else {
            this.tokenizer.skipWsAndDocs();
            String target = this.tokenizer.internString(IdlShapeIdParser.expectAndSkipShapeId(this.tokenizer));
            this.addForwardReference(target, consumer);
        }
    }

    private ShapeId parseInlineStructure(String name, IdlTraitParser.Result defaultTrait) {
        List<IdlTraitParser.Result> traits = IdlTraitParser.parseDocsAndTraitsBeforeShape(this.tokenizer, this.resolver);
        if (defaultTrait != null) {
            traits.add(defaultTrait);
        }
        ShapeId id = ShapeId.fromRelative(this.expectNamespace(), name);
        SourceLocation location = this.tokenizer.getCurrentTokenLocation();
        StructureShape.Builder builder = (StructureShape.Builder)((StructureShape.Builder)StructureShape.builder().id(id)).source(location);
        this.parseAggregateShape(id, location, builder);
        this.addTraits(id, traits);
        return id;
    }

    private void parseIdList(Consumer<ShapeId> consumer) {
        this.tokenizer.increaseNestingLevel();
        this.tokenizer.skipWsAndDocs();
        this.tokenizer.expect(IdlToken.LBRACKET);
        this.tokenizer.next();
        this.tokenizer.skipWsAndDocs();
        while (this.tokenizer.hasNext() && this.tokenizer.getCurrentToken() != IdlToken.RBRACKET) {
            this.tokenizer.expect(IdlToken.IDENTIFIER);
            String target = this.tokenizer.internString(IdlShapeIdParser.expectAndSkipShapeId(this.tokenizer));
            this.addForwardReference(target, consumer);
            this.tokenizer.skipWsAndDocs();
        }
        this.tokenizer.expect(IdlToken.RBRACKET);
        this.tokenizer.next();
        this.tokenizer.decreaseNestingLevel();
    }

    static {
        for (ShapeType type : ShapeType.values()) {
            if (type == ShapeType.MEMBER) continue;
            SHAPE_TYPES.add(type.toString());
        }
    }
}

