/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.validation.validators;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.knowledge.NeighborProviderIndex;
import software.amazon.smithy.model.neighbor.NeighborProvider;
import software.amazon.smithy.model.neighbor.Relationship;
import software.amazon.smithy.model.neighbor.RelationshipDirection;
import software.amazon.smithy.model.neighbor.RelationshipType;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeType;
import software.amazon.smithy.model.traits.DeprecatedTrait;
import software.amazon.smithy.model.traits.MixinTrait;
import software.amazon.smithy.model.traits.TraitDefinition;
import software.amazon.smithy.model.validation.AbstractValidator;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.FunctionalUtils;
import software.amazon.smithy.utils.MapUtils;
import software.amazon.smithy.utils.SetUtils;
import software.amazon.smithy.utils.StringUtils;

public final class TargetValidator
extends AbstractValidator {
    private static final int MAX_EDIT_DISTANCE_FOR_SUGGESTIONS = 2;
    private static final Set<ShapeType> INVALID_MEMBER_TARGETS = SetUtils.of((Object[])new ShapeType[]{ShapeType.SERVICE, ShapeType.RESOURCE, ShapeType.OPERATION, ShapeType.MEMBER});
    private static final Map<RelationshipType, String> RELATIONSHIP_TYPE_DEPRECATION_MAPPINGS = MapUtils.of((Object)((Object)RelationshipType.MEMBER_TARGET), (Object)"Member targets a deprecated shape", (Object)((Object)RelationshipType.RESOURCE), (Object)"Binds a deprecated resource", (Object)((Object)RelationshipType.OPERATION), (Object)"Binds a deprecated operation", (Object)((Object)RelationshipType.IDENTIFIER), (Object)"Resource identifier targets a deprecated shape", (Object)((Object)RelationshipType.PROPERTY), (Object)"Resource property targets a deprecated shape", (Object)((Object)RelationshipType.INPUT), (Object)"Operation input targets a deprecated shape", (Object)((Object)RelationshipType.OUTPUT), (Object)"Operation output targets a deprecated shape", (Object)((Object)RelationshipType.ERROR), (Object)"Operation error targets a deprecated shape", (Object)((Object)RelationshipType.MIXIN), (Object)"Applies a deprecated mixin");

    @Override
    public List<ValidationEvent> validate(Model model) {
        ArrayList<ValidationEvent> events = new ArrayList<ValidationEvent>();
        NeighborProvider neighborProvider = NeighborProviderIndex.of(model).getProvider();
        for (Shape shape : model.toSet()) {
            this.validateShape(model, shape, neighborProvider.getNeighbors(shape), events);
        }
        return events;
    }

    private void validateShape(Model model, Shape shape, List<Relationship> relationships, List<ValidationEvent> mutableEvents) {
        for (Relationship relationship : relationships) {
            if (relationship.getNeighborShape().isPresent()) {
                this.validateTarget(model, shape, relationship.getNeighborShape().get(), relationship, mutableEvents);
                continue;
            }
            mutableEvents.add(this.unresolvedTarget(model, shape, relationship));
        }
    }

    private void validateTarget(Model model, Shape shape, Shape target, Relationship rel, List<ValidationEvent> events) {
        RelationshipType relType = rel.getRelationshipType();
        if (relType != RelationshipType.MIXIN && relType.getDirection() == RelationshipDirection.DIRECTED) {
            if (target.hasTrait(TraitDefinition.class)) {
                events.add(this.error(shape, String.format("Found a %s reference to trait definition `%s`. Trait definitions cannot be targeted by members or referenced by shapes in any other context other than applying them as traits.", new Object[]{relType, rel.getNeighborShapeId()})));
                return;
            }
            if (!target.isMemberShape() && target.hasTrait(MixinTrait.class)) {
                events.add(this.error(shape, String.format("Illegal %s reference to mixin `%s`; shapes marked with the mixin trait can only be referenced to apply them as a mixin.", new Object[]{relType, rel.getNeighborShapeId()})));
                return;
            }
        }
        this.validateDeprecatedTargets(shape, target, relType, events);
        switch (relType) {
            case PROPERTY: 
            case MEMBER_TARGET: {
                if (!INVALID_MEMBER_TARGETS.contains((Object)target.getType())) break;
                events.add(this.error(shape, String.format("Members cannot target %s shapes, but found %s", new Object[]{target.getType(), target})));
                break;
            }
            case MAP_KEY: {
                target.asMemberShape().ifPresent(m -> this.validateMapKey(shape, m.getTarget(), model, events));
                break;
            }
            case RESOURCE: {
                if (target.getType() == ShapeType.RESOURCE) break;
                events.add(this.badType(shape, target, relType, ShapeType.RESOURCE));
                break;
            }
            case OPERATION: {
                if (target.getType() == ShapeType.OPERATION) break;
                events.add(this.badType(shape, target, relType, ShapeType.OPERATION));
                break;
            }
            case INPUT: 
            case OUTPUT: {
                if (target.getType() != ShapeType.STRUCTURE) {
                    events.add(this.badType(shape, target, relType, ShapeType.STRUCTURE));
                    break;
                }
                if (!target.findTrait("error").isPresent()) break;
                events.add(this.inputOutputWithErrorTrait(shape, target, rel.getRelationshipType()));
                break;
            }
            case ERROR: {
                if (target.getType() != ShapeType.STRUCTURE) {
                    events.add(this.badType(shape, target, relType, ShapeType.STRUCTURE));
                    break;
                }
                if (target.findTrait("error").isPresent()) break;
                events.add(this.errorNoTrait(shape, target.getId()));
                break;
            }
            case IDENTIFIER: {
                this.validateIdentifier(shape, target, events);
                break;
            }
            case CREATE: 
            case READ: 
            case UPDATE: 
            case DELETE: 
            case LIST: {
                if (target.getType() == ShapeType.OPERATION) break;
                events.add(this.error(shape, String.format("Resource %s lifecycle operation must target an operation, but found %s", relType.toString().toLowerCase(Locale.US), target)));
                break;
            }
            case MIXIN: {
                if (target.hasTrait(MixinTrait.class)) break;
                events.add(this.error(shape, String.format("Attempted to use %s as a mixin, but it is not marked with the mixin trait", target.getId())));
                break;
            }
        }
    }

    private void validateDeprecatedTargets(Shape shape, Shape target, RelationshipType relType, List<ValidationEvent> events) {
        if (!target.hasTrait(DeprecatedTrait.class)) {
            return;
        }
        String relLabel = RELATIONSHIP_TYPE_DEPRECATION_MAPPINGS.get((Object)relType);
        if (relLabel == null) {
            return;
        }
        StringBuilder builder = new StringBuilder(relLabel).append(", ").append(target.getId());
        DeprecatedTrait deprecatedTrait = target.expectTrait(DeprecatedTrait.class);
        deprecatedTrait.getMessage().ifPresent(message -> builder.append(". ").append((String)message));
        deprecatedTrait.getSince().ifPresent(since -> builder.append(" (since ").append((String)since).append(')'));
        events.add(ValidationEvent.builder().id("DeprecatedShape." + target.getId()).severity(Severity.WARNING).shape(shape).message(builder.toString()).build());
    }

    private void validateMapKey(Shape shape, ShapeId target, Model model, List<ValidationEvent> events) {
        model.getShape(target).filter(FunctionalUtils.not(Shape::isStringShape)).ifPresent(resolved -> {
            String message = String.format("Map key member targets %s, but is expected to target a string", resolved);
            events.add(this.error(shape, message));
        });
    }

    private void validateIdentifier(Shape shape, Shape target, List<ValidationEvent> events) {
        if (target.getType() != ShapeType.STRING && target.getType() != ShapeType.ENUM) {
            events.add(this.badType(shape, target, RelationshipType.IDENTIFIER, ShapeType.STRING));
        }
    }

    private ValidationEvent unresolvedTarget(Model model, Shape shape, Relationship rel) {
        String suggestionText;
        Collection<String> suggestions = this.computeTargetSuggestions(model, rel.getNeighborShapeId());
        String string = suggestionText = !suggestions.isEmpty() ? ". Did you mean " + String.join((CharSequence)", ", suggestions) + "?" : "";
        if (rel.getRelationshipType() == RelationshipType.MEMBER_TARGET) {
            return this.error(shape, String.format("member shape targets an unresolved shape `%s`%s", rel.getNeighborShapeId(), suggestionText));
        }
        String indefiniteArticle = TargetValidator.isUppercaseVowel(rel.getRelationshipType().toString().charAt(0)) ? "an" : "a";
        return this.error(shape, String.format("%s shape has %s `%s` relationship to an unresolved shape `%s`%s", new Object[]{shape.getType(), indefiniteArticle, rel.getRelationshipType().toString().toLowerCase(Locale.US), rel.getNeighborShapeId(), suggestionText}));
    }

    private Collection<String> computeTargetSuggestions(Model model, ShapeId target) {
        String targetString = target.toString();
        int floor = Integer.MAX_VALUE;
        TreeSet<String> candidates = new TreeSet<String>();
        for (Shape shape : model.toSet()) {
            String idString = shape.getId().toString();
            int distance = StringUtils.levenshteinDistance((CharSequence)targetString, (CharSequence)idString, (int)2);
            if (distance == floor) {
                candidates.add(idString);
                continue;
            }
            if (distance <= -1 || distance >= floor) continue;
            floor = distance;
            candidates.clear();
            candidates.add(idString);
        }
        return candidates;
    }

    private static boolean isUppercaseVowel(char c) {
        return c == 'A' || c == 'E' || c == 'I' || c == 'O' || c == 'U';
    }

    private ValidationEvent badType(Shape shape, Shape target, RelationshipType rel, ShapeType valid) {
        return this.error(shape, String.format("%s shape `%s` relationships must target a %s shape, but found %s", new Object[]{shape.getType(), rel.toString().toLowerCase(Locale.US), valid, target}));
    }

    private ValidationEvent inputOutputWithErrorTrait(Shape shape, Shape target, RelationshipType rel) {
        String descriptor = rel == RelationshipType.INPUT ? "input" : "output";
        return this.error(shape, String.format("Operation %s targets an invalid structure `%s` that is marked with the `error` trait.", descriptor, target.getId()));
    }

    private ValidationEvent errorNoTrait(Shape shape, ShapeId target) {
        return this.error(shape, String.format("`%s` cannot be bound as an error because it is not marked with the `error` trait.", target));
    }
}

