/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.validation.testrunner;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidatedResult;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.IoUtils;

public final class SmithyTestCase {
    private static final Pattern EVENT_PATTERN = Pattern.compile("^\\[(?<severity>SUPPRESSED|NOTE|WARNING|DANGER|ERROR)] (?<shape>[^ ]+): ?(?<message>.*) \\| (?<id>[^)]+)");
    private final List<ValidationEvent> expectedEvents;
    private final String modelLocation;

    public SmithyTestCase(String modelLocation, List<ValidationEvent> expectedEvents) {
        this.modelLocation = Objects.requireNonNull(modelLocation);
        this.expectedEvents = Collections.unmodifiableList(expectedEvents);
    }

    public static SmithyTestCase fromModelFile(String modelLocation) {
        String errorFileLocation = SmithyTestCase.inferErrorFileLocation(modelLocation);
        List<ValidationEvent> expectedEvents = SmithyTestCase.loadExpectedEvents(errorFileLocation);
        return new SmithyTestCase(modelLocation, expectedEvents);
    }

    public List<ValidationEvent> getExpectedEvents() {
        return this.expectedEvents;
    }

    public String getModelLocation() {
        return this.modelLocation;
    }

    public Result createResult(ValidatedResult<Model> validatedResult) {
        List<ValidationEvent> actualEvents = validatedResult.getValidationEvents();
        List<ValidationEvent> unmatchedEvents = this.getExpectedEvents().stream().filter(expectedEvent -> actualEvents.stream().noneMatch(actualEvent -> SmithyTestCase.compareEvents(expectedEvent, actualEvent))).collect(Collectors.toList());
        List<ValidationEvent> extraEvents = actualEvents.stream().filter(actualEvent -> this.getExpectedEvents().stream().noneMatch(expectedEvent -> SmithyTestCase.compareEvents(expectedEvent, actualEvent))).filter(event -> event.getSeverity() != Severity.SUPPRESSED).filter(event -> !this.isModelDeprecationEvent((ValidationEvent)event)).collect(Collectors.toList());
        return new Result(this.getModelLocation(), unmatchedEvents, extraEvents);
    }

    private static boolean compareEvents(ValidationEvent expected, ValidationEvent actual) {
        String normalizedActualMessage = actual.getMessage();
        if (actual.getSuppressionReason().isPresent()) {
            normalizedActualMessage = normalizedActualMessage + " (" + actual.getSuppressionReason().get() + ")";
        }
        String comparedMessage = expected.getMessage().replace("\n", "\\n").replace("\r", "\\n");
        return expected.getSeverity() == actual.getSeverity() && actual.containsId(expected.getId()) && expected.getShapeId().equals(actual.getShapeId()) && normalizedActualMessage.startsWith(comparedMessage);
    }

    private boolean isModelDeprecationEvent(ValidationEvent event) {
        return event.getId().equals("ModelDeprecation") || event.getId().equals("DeprecatedTrait") || event.getId().equals("DeprecatedShape");
    }

    private static String inferErrorFileLocation(String modelLocation) {
        int extensionPosition = modelLocation.lastIndexOf(".");
        if (extensionPosition == -1) {
            throw new IllegalArgumentException("Invalid Smithy model file: " + modelLocation);
        }
        return modelLocation.substring(0, extensionPosition) + ".errors";
    }

    private static List<ValidationEvent> loadExpectedEvents(String errorsFileLocation) {
        String contents = IoUtils.readUtf8File((String)errorsFileLocation);
        return Arrays.stream(contents.split(System.lineSeparator())).filter(line -> !line.trim().isEmpty()).map(SmithyTestCase::parseValidationEvent).collect(Collectors.toList());
    }

    static ValidationEvent parseValidationEvent(String event) {
        Matcher matcher = EVENT_PATTERN.matcher(event);
        if (!matcher.find()) {
            throw new IllegalArgumentException("Invalid validation event: " + event);
        }
        SourceLocation location = new SourceLocation("/", 0, 0);
        ValidationEvent.Builder builder = ValidationEvent.builder().severity(Severity.fromString(matcher.group("severity")).get()).sourceLocation(location).id(matcher.group("id")).message(matcher.group("message"));
        if (!matcher.group("shape").equals("-")) {
            builder.shapeId(ShapeId.from(matcher.group("shape")));
        }
        return builder.build();
    }

    public static final class Result {
        private final String modelLocation;
        private final Collection<ValidationEvent> unmatchedEvents;
        private final Collection<ValidationEvent> extraEvents;

        Result(String modelLocation, Collection<ValidationEvent> unmatchedEvents, Collection<ValidationEvent> extraEvents) {
            this.modelLocation = modelLocation;
            this.unmatchedEvents = Collections.unmodifiableCollection(new TreeSet<ValidationEvent>(unmatchedEvents));
            this.extraEvents = Collections.unmodifiableCollection(new TreeSet<ValidationEvent>(extraEvents));
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("=======================\nModel Validation Result\n=======================\n").append(this.getModelLocation()).append('\n');
            if (!this.getUnmatchedEvents().isEmpty()) {
                builder.append("\nDid not match the following events\n----------------------------------\n");
                for (ValidationEvent event : this.getUnmatchedEvents()) {
                    builder.append(event.toString().replace("\n", "\\n")).append('\n');
                }
                builder.append('\n');
            }
            if (!this.getExtraEvents().isEmpty()) {
                builder.append("\nEncountered unexpected events\n-----------------------------\n");
                for (ValidationEvent event : this.getExtraEvents()) {
                    builder.append(event.toString().replace("\n", "\\n")).append("\n");
                }
                builder.append('\n');
            }
            return builder.toString();
        }

        public boolean isInvalid() {
            return !this.unmatchedEvents.isEmpty() || !this.extraEvents.isEmpty();
        }

        public String getModelLocation() {
            return this.modelLocation;
        }

        public Collection<ValidationEvent> getUnmatchedEvents() {
            return this.unmatchedEvents;
        }

        public Collection<ValidationEvent> getExtraEvents() {
            return this.extraEvents;
        }

        public Result unwrap() {
            if (this.isInvalid()) {
                throw new Error(this);
            }
            return this;
        }
    }

    public static final class Error
    extends RuntimeException {
        public final Result result;

        Error(Result result) {
            super(result.toString());
            this.result = result;
        }
    }
}

