/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.loader.ApplyMixin;
import software.amazon.smithy.model.loader.LoadOperation;
import software.amazon.smithy.model.loader.LoaderUtils;
import software.amazon.smithy.model.loader.Version;
import software.amazon.smithy.model.node.ArrayNode;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.BigDecimalShape;
import software.amazon.smithy.model.shapes.BigIntegerShape;
import software.amazon.smithy.model.shapes.BlobShape;
import software.amazon.smithy.model.shapes.BooleanShape;
import software.amazon.smithy.model.shapes.ByteShape;
import software.amazon.smithy.model.shapes.CollectionShape;
import software.amazon.smithy.model.shapes.DocumentShape;
import software.amazon.smithy.model.shapes.DoubleShape;
import software.amazon.smithy.model.shapes.EnumShape;
import software.amazon.smithy.model.shapes.FloatShape;
import software.amazon.smithy.model.shapes.IntEnumShape;
import software.amazon.smithy.model.shapes.IntegerShape;
import software.amazon.smithy.model.shapes.ListShape;
import software.amazon.smithy.model.shapes.LongShape;
import software.amazon.smithy.model.shapes.MapShape;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.ResourceShape;
import software.amazon.smithy.model.shapes.ServiceShape;
import software.amazon.smithy.model.shapes.SetShape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShortShape;
import software.amazon.smithy.model.shapes.StringShape;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.TimestampShape;
import software.amazon.smithy.model.shapes.UnionShape;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.SetUtils;

final class AstModelLoader {
    private static final String METADATA = "metadata";
    private static final String MEMBERS = "members";
    private static final String SHAPES = "shapes";
    private static final String TRAITS = "traits";
    private static final String TYPE = "type";
    private static final String TARGET = "target";
    private static final String ERRORS = "errors";
    private static final String MIXINS = "mixins";
    private static final List<String> TOP_LEVEL_PROPERTIES = ListUtils.of((Object[])new String[]{"smithy", "shapes", "metadata"});
    private static final List<String> APPLY_PROPERTIES = ListUtils.of((Object)"type", (Object)"traits");
    private static final List<String> SIMPLE_PROPERTY_NAMES = ListUtils.of((Object)"type", (Object)"traits");
    private static final List<String> NAMED_MEMBER_SHAPE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "members", "traits", "mixins"});
    private static final List<String> COLLECTION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "member", "traits"});
    private static final List<String> MAP_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "key", "value", "traits"});
    private static final Set<String> MEMBER_PROPERTIES = SetUtils.of((Object[])new String[]{"target", "traits"});
    private static final Set<String> REFERENCE_PROPERTIES = SetUtils.of((Object)"target");
    private static final Set<String> OPERATION_PROPERTY_NAMES = SetUtils.of((Object[])new String[]{"type", "input", "output", "errors", "traits"});
    private static final Set<String> RESOURCE_PROPERTIES = SetUtils.of((Object[])new String[]{"type", "create", "read", "update", "delete", "list", "put", "identifiers", "resources", "operations", "collectionOperations", "properties", "traits"});
    private static final Set<String> SERVICE_PROPERTIES = SetUtils.of((Object[])new String[]{"type", "version", "operations", "resources", "rename", "errors", "traits"});
    private final Version modelVersion;
    private final ObjectNode model;
    private Consumer<LoadOperation> operations;

    AstModelLoader(Version modelVersion, ObjectNode model) {
        this.modelVersion = modelVersion;
        this.model = model;
    }

    void parse(Consumer<LoadOperation> consumer) {
        this.operations = consumer;
        LoaderUtils.checkForAdditionalProperties(this.model, null, TOP_LEVEL_PROPERTIES).ifPresent(this::emit);
        StringNode versionNode = this.model.expectStringMember("smithy");
        consumer.accept(new LoadOperation.ModelVersion(this.modelVersion, versionNode.getSourceLocation()));
        this.loadMetadata();
        this.loadShapes();
    }

    private void emit(ValidationEvent event) {
        this.operations.accept(new LoadOperation.Event(event));
    }

    private void loadMetadata() {
        try {
            this.model.getObjectMember(METADATA).ifPresent(metadata -> {
                for (Map.Entry<String, Node> entry : metadata.getStringMap().entrySet()) {
                    this.operations.accept(new LoadOperation.PutMetadata(this.modelVersion, entry.getKey(), entry.getValue()));
                }
            });
        }
        catch (SourceException e) {
            this.emit(ValidationEvent.fromSourceException(e));
        }
    }

    private void loadShapes() {
        this.model.getObjectMember(SHAPES).ifPresent(shapes -> {
            for (Map.Entry<StringNode, Node> entry : shapes.getMembers().entrySet()) {
                ShapeId id = entry.getKey().expectShapeId();
                ObjectNode definition = entry.getValue().expectObjectNode();
                String type = definition.expectStringMember(TYPE).getValue();
                try {
                    LoadOperation.DefineShape defineShape = this.loadShape(id, type, definition);
                    if (defineShape == null) continue;
                    this.operations.accept(defineShape);
                }
                catch (SourceException e) {
                    ValidationEvent event = ValidationEvent.fromSourceException(e).toBuilder().shapeId(id).build();
                    this.emit(event);
                }
            }
        });
    }

    private LoadOperation.DefineShape loadShape(ShapeId id, String type, ObjectNode value) {
        switch (type) {
            case "blob": {
                return this.loadSimpleShape(id, value, BlobShape.builder());
            }
            case "boolean": {
                return this.loadSimpleShape(id, value, BooleanShape.builder());
            }
            case "byte": {
                return this.loadSimpleShape(id, value, ByteShape.builder());
            }
            case "short": {
                return this.loadSimpleShape(id, value, ShortShape.builder());
            }
            case "integer": {
                return this.loadSimpleShape(id, value, IntegerShape.builder());
            }
            case "intEnum": {
                return this.loadNamedMemberShape(id, value, IntEnumShape.builder());
            }
            case "long": {
                return this.loadSimpleShape(id, value, LongShape.builder());
            }
            case "float": {
                return this.loadSimpleShape(id, value, FloatShape.builder());
            }
            case "double": {
                return this.loadSimpleShape(id, value, DoubleShape.builder());
            }
            case "document": {
                return this.loadSimpleShape(id, value, DocumentShape.builder());
            }
            case "bigDecimal": {
                return this.loadSimpleShape(id, value, BigDecimalShape.builder());
            }
            case "bigInteger": {
                return this.loadSimpleShape(id, value, BigIntegerShape.builder());
            }
            case "string": {
                return this.loadSimpleShape(id, value, StringShape.builder());
            }
            case "enum": {
                return this.loadNamedMemberShape(id, value, EnumShape.builder());
            }
            case "timestamp": {
                return this.loadSimpleShape(id, value, TimestampShape.builder());
            }
            case "list": {
                return this.loadCollection(id, value, ListShape.builder());
            }
            case "set": {
                return this.loadCollection(id, value, SetShape.builder());
            }
            case "map": {
                return this.loadMap(id, value);
            }
            case "resource": {
                return this.loadResource(id, value);
            }
            case "service": {
                return this.loadService(id, value);
            }
            case "structure": {
                return this.loadNamedMemberShape(id, value, StructureShape.builder());
            }
            case "union": {
                return this.loadNamedMemberShape(id, value, UnionShape.builder());
            }
            case "operation": {
                return this.loadOperation(id, value);
            }
            case "apply": {
                LoaderUtils.checkForAdditionalProperties(value, id, APPLY_PROPERTIES).ifPresent(this::emit);
                value.getObjectMember(TRAITS).ifPresent(traits -> this.applyTraits(id, (ObjectNode)traits));
                return null;
            }
        }
        throw new SourceException("Invalid shape `type`: " + type, value);
    }

    private void applyTraits(ShapeId id, ObjectNode traits) {
        for (Map.Entry<StringNode, Node> traitNode : traits.getMembers().entrySet()) {
            ShapeId traitId = traitNode.getKey().expectShapeId();
            this.operations.accept(new LoadOperation.ApplyTrait(this.modelVersion, traitNode.getKey().getSourceLocation(), id.getNamespace(), id, traitId, traitNode.getValue()));
        }
    }

    private void applyShapeTraits(ShapeId id, ObjectNode node) {
        node.getObjectMember(TRAITS).ifPresent(traits -> this.applyTraits(id, (ObjectNode)traits));
    }

    private void loadMember(LoadOperation.DefineShape operation, ShapeId id, ObjectNode targetNode) {
        LoaderUtils.checkForAdditionalProperties(targetNode, id, MEMBER_PROPERTIES).ifPresent(this::emit);
        MemberShape.Builder builder = (MemberShape.Builder)((MemberShape.Builder)MemberShape.builder().source(targetNode.getSourceLocation())).id(id);
        ShapeId target = targetNode.expectStringMember(TARGET).expectShapeId();
        builder.target(target);
        this.applyShapeTraits(id, targetNode);
        operation.addMember(builder);
    }

    private void loadOptionalMember(LoadOperation.DefineShape operation, ShapeId id, ObjectNode node, String member) {
        node.getObjectMember(member).ifPresent(targetNode -> this.loadMember(operation, id, (ObjectNode)targetNode));
    }

    private LoadOperation.DefineShape loadCollection(ShapeId id, ObjectNode node, CollectionShape.Builder<?, ?> builder) {
        LoaderUtils.checkForAdditionalProperties(node, id, COLLECTION_PROPERTY_NAMES).ifPresent(this::emit);
        this.applyShapeTraits(id, node);
        builder.id(id).source(node.getSourceLocation());
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.loadOptionalMember(operation, id.withMember("member"), node, "member");
        this.addMixins(operation, node);
        return operation;
    }

    LoadOperation.DefineShape createShape(AbstractShapeBuilder<?, ?> builder) {
        return new LoadOperation.DefineShape(this.modelVersion, builder);
    }

    private LoadOperation.DefineShape loadMap(ShapeId id, ObjectNode node) {
        LoaderUtils.checkForAdditionalProperties(node, id, MAP_PROPERTY_NAMES).ifPresent(this::emit);
        MapShape.Builder builder = (MapShape.Builder)MapShape.builder().id(id).source(node.getSourceLocation());
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.loadOptionalMember(operation, id.withMember("key"), node, "key");
        this.loadOptionalMember(operation, id.withMember("value"), node, "value");
        this.addMixins(operation, node);
        this.applyShapeTraits(id, node);
        return operation;
    }

    private LoadOperation.DefineShape loadOperation(ShapeId id, ObjectNode node) {
        LoaderUtils.checkForAdditionalProperties(node, id, OPERATION_PROPERTY_NAMES).ifPresent(this::emit);
        this.applyShapeTraits(id, node);
        OperationShape.Builder builder = ((OperationShape.Builder)((OperationShape.Builder)OperationShape.builder().id(id)).source(node.getSourceLocation())).addErrors(this.loadOptionalTargetList(id, node, ERRORS));
        this.loadOptionalTarget(id, node, "input").ifPresent(builder::input);
        this.loadOptionalTarget(id, node, "output").ifPresent(builder::output);
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.addMixins(operation, node);
        return operation;
    }

    private LoadOperation.DefineShape loadResource(ShapeId id, ObjectNode node) {
        LoaderUtils.checkForAdditionalProperties(node, id, RESOURCE_PROPERTIES).ifPresent(this::emit);
        this.applyShapeTraits(id, node);
        ResourceShape.Builder builder = (ResourceShape.Builder)((ResourceShape.Builder)ResourceShape.builder().id(id)).source(node.getSourceLocation());
        this.loadOptionalTarget(id, node, "put").ifPresent(builder::put);
        this.loadOptionalTarget(id, node, "create").ifPresent(builder::create);
        this.loadOptionalTarget(id, node, "read").ifPresent(builder::read);
        this.loadOptionalTarget(id, node, "update").ifPresent(builder::update);
        this.loadOptionalTarget(id, node, "delete").ifPresent(builder::delete);
        this.loadOptionalTarget(id, node, "list").ifPresent(builder::list);
        builder.operations(this.loadOptionalTargetList(id, node, "operations"));
        builder.collectionOperations(this.loadOptionalTargetList(id, node, "collectionOperations"));
        builder.resources(this.loadOptionalTargetList(id, node, "resources"));
        node.getObjectMember("identifiers").ifPresent(ids -> {
            for (Map.Entry<StringNode, Node> entry : ids.getMembers().entrySet()) {
                String name = entry.getKey().getValue();
                ShapeId target = this.loadReferenceBody(id, entry.getValue());
                builder.addIdentifier(name, target);
            }
        });
        node.getObjectMember("properties").ifPresent(properties -> {
            if (!this.modelVersion.supportsResourceProperties()) {
                this.emit(ValidationEvent.builder().sourceLocation(properties.getSourceLocation()).id("Model").severity(Severity.ERROR).message("Resource properties can only be used with Smithy version 2 or later. Attempted to use resource properties with version `" + (Object)((Object)this.modelVersion) + "`.").build());
            }
            for (Map.Entry<StringNode, Node> entry : properties.getMembers().entrySet()) {
                String name = entry.getKey().getValue();
                ShapeId target = this.loadReferenceBody(id, entry.getValue());
                builder.addProperty(name, target);
            }
        });
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.addMixins(operation, node);
        return operation;
    }

    private LoadOperation.DefineShape loadService(ShapeId id, ObjectNode node) {
        LoaderUtils.checkForAdditionalProperties(node, id, SERVICE_PROPERTIES).ifPresent(this::emit);
        this.applyShapeTraits(id, node);
        ServiceShape.Builder builder = (ServiceShape.Builder)((ServiceShape.Builder)new ServiceShape.Builder().id(id)).source(node.getSourceLocation());
        node.getStringMember("version").map(StringNode::getValue).ifPresent(builder::version);
        builder.operations(this.loadOptionalTargetList(id, node, "operations"));
        builder.resources(this.loadOptionalTargetList(id, node, "resources"));
        AstModelLoader.loadServiceRenameIntoBuilder(builder, node);
        builder.addErrors(this.loadOptionalTargetList(id, node, ERRORS));
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.addMixins(operation, node);
        return operation;
    }

    static void loadServiceRenameIntoBuilder(ServiceShape.Builder builder, ObjectNode node) {
        node.getObjectMember("rename").ifPresent(rename -> {
            for (Map.Entry<StringNode, Node> entry : rename.getMembers().entrySet()) {
                ShapeId fromId = entry.getKey().expectShapeId();
                String toName = entry.getValue().expectStringNode().getValue();
                builder.putRename(fromId, toName);
            }
        });
    }

    private LoadOperation.DefineShape loadSimpleShape(ShapeId id, ObjectNode node, AbstractShapeBuilder<?, ?> builder) {
        LoaderUtils.checkForAdditionalProperties(node, id, SIMPLE_PROPERTY_NAMES).ifPresent(this::emit);
        this.applyShapeTraits(id, node);
        ((AbstractShapeBuilder)builder.id(id)).source(node.getSourceLocation());
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.addMixins(operation, node);
        return operation;
    }

    private LoadOperation.DefineShape loadNamedMemberShape(ShapeId id, ObjectNode node, AbstractShapeBuilder<?, ?> builder) {
        LoaderUtils.checkForAdditionalProperties(node, id, NAMED_MEMBER_SHAPE_PROPERTY_NAMES).ifPresent(this::emit);
        ((AbstractShapeBuilder)builder.id(id)).source(node.getSourceLocation());
        LoadOperation.DefineShape operation = this.createShape(builder);
        this.finishLoadingNamedMemberShapeMembers(operation, node);
        return operation;
    }

    private void finishLoadingNamedMemberShapeMembers(LoadOperation.DefineShape operation, ObjectNode node) {
        this.applyShapeTraits(operation.toShapeId(), node);
        ObjectNode memberObject = node.getObjectMember(MEMBERS).orElse(Node.objectNode());
        for (Map.Entry<String, Node> entry : memberObject.getStringMap().entrySet()) {
            this.loadMember(operation, operation.toShapeId().withMember(entry.getKey()), entry.getValue().expectObjectNode());
        }
        this.addMixins(operation, node);
    }

    private void addMixins(LoadOperation.DefineShape operation, ObjectNode node) {
        ArrayNode mixins = node.getArrayMember(MIXINS).orElse(Node.arrayNode());
        for (ObjectNode mixin : mixins.getElementsAs(ObjectNode.class)) {
            ShapeId mixinId = this.loadReferenceBody(operation.toShapeId(), mixin);
            operation.addDependency(mixinId);
            operation.addModifier(new ApplyMixin(mixinId));
        }
    }

    private Optional<ShapeId> loadOptionalTarget(ShapeId id, ObjectNode node, String member) {
        return node.getObjectMember(member).map(r -> this.loadReferenceBody(id, (Node)r));
    }

    private ShapeId loadReferenceBody(ShapeId id, Node reference) {
        ObjectNode referenceObject = reference.expectObjectNode();
        LoaderUtils.checkForAdditionalProperties(referenceObject, id, REFERENCE_PROPERTIES).ifPresent(this::emit);
        return referenceObject.expectStringMember(TARGET).expectShapeId();
    }

    private List<ShapeId> loadOptionalTargetList(ShapeId id, ObjectNode node, String member) {
        return node.getArrayMember(member).map(array -> {
            ArrayList<ShapeId> ids = new ArrayList<ShapeId>(array.size());
            for (Node element : array.getElements()) {
                ids.add(this.loadReferenceBody(id, element));
            }
            return ids;
        }).orElseGet(Collections::emptyList);
    }
}

