/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.function.Supplier;
import java.util.logging.Logger;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.loader.LoaderVisitor;
import software.amazon.smithy.model.loader.ModelDiscovery;
import software.amazon.smithy.model.loader.ModelImportException;
import software.amazon.smithy.model.loader.ModelLoader;
import software.amazon.smithy.model.loader.ModelValidator;
import software.amazon.smithy.model.loader.Prelude;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.traits.TraitFactory;
import software.amazon.smithy.model.validation.ValidatedResult;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.model.validation.Validator;
import software.amazon.smithy.model.validation.ValidatorFactory;

public final class ModelAssembler {
    public static final String ALLOW_UNKNOWN_TRAITS = "assembler.allowUnknownTraits";
    public static final String DISABLE_JAR_CACHE = "assembler.disableJarCache";
    private static final Logger LOGGER = Logger.getLogger(ModelAssembler.class.getName());
    private TraitFactory traitFactory;
    private ValidatorFactory validatorFactory;
    private boolean disableValidation;
    private final Map<String, Supplier<InputStream>> inputStreamModels = new TreeMap<String, Supplier<InputStream>>((a, b) -> {
        boolean isJsonA = a.endsWith(".json");
        boolean isJsonB = b.endsWith(".json");
        if (isJsonA) {
            if (!isJsonB) {
                return -1;
            }
        } else if (isJsonB) {
            return 1;
        }
        return a.compareTo((String)b);
    });
    private final List<Validator> validators = new ArrayList<Validator>();
    private final List<Node> documentNodes = new ArrayList<Node>();
    private final List<Model> mergeModels = new ArrayList<Model>();
    private final List<AbstractShapeBuilder<?, ?>> shapes = new ArrayList();
    private final Map<String, Node> metadata = new HashMap<String, Node>();
    private final Map<String, Object> properties = new HashMap<String, Object>();
    private boolean disablePrelude;

    public ModelAssembler copy() {
        ModelAssembler assembler = new ModelAssembler();
        assembler.traitFactory = this.traitFactory;
        assembler.validatorFactory = this.validatorFactory;
        assembler.inputStreamModels.putAll(this.inputStreamModels);
        assembler.validators.addAll(this.validators);
        assembler.documentNodes.addAll(this.documentNodes);
        assembler.mergeModels.addAll(this.mergeModels);
        assembler.shapes.addAll(this.shapes);
        assembler.metadata.putAll(this.metadata);
        assembler.disablePrelude = this.disablePrelude;
        assembler.properties.putAll(this.properties);
        assembler.disableValidation = this.disableValidation;
        return assembler;
    }

    public ModelAssembler reset() {
        this.shapes.clear();
        this.metadata.clear();
        this.mergeModels.clear();
        this.inputStreamModels.clear();
        this.validators.clear();
        this.documentNodes.clear();
        this.disablePrelude = false;
        this.disableValidation = false;
        return this;
    }

    public ModelAssembler traitFactory(TraitFactory traitFactory) {
        this.traitFactory = Objects.requireNonNull(traitFactory);
        return this;
    }

    public ModelAssembler validatorFactory(ValidatorFactory validatorFactory) {
        this.validatorFactory = Objects.requireNonNull(validatorFactory);
        return this;
    }

    public ModelAssembler addValidator(Validator validator) {
        this.validators.add(Objects.requireNonNull(validator));
        return this;
    }

    public ModelAssembler addUnparsedModel(String sourceLocation, String model) {
        this.inputStreamModels.put(sourceLocation, () -> new ByteArrayInputStream(model.getBytes(Charset.forName("UTF-8"))));
        return this;
    }

    public ModelAssembler addDocumentNode(Node document) {
        this.documentNodes.add(Objects.requireNonNull(document));
        return this;
    }

    public ModelAssembler addImport(String importPath) {
        return this.addImport(Paths.get(Objects.requireNonNull(importPath, "importPath must not be null"), new String[0]));
    }

    public ModelAssembler addImport(Path importPath) {
        Objects.requireNonNull(importPath, "The importPath provided to ModelAssembler#addImport was null");
        if (Files.isDirectory(importPath, new LinkOption[0])) {
            try {
                Files.walk(importPath, FileVisitOption.FOLLOW_LINKS).filter(p -> !p.equals(importPath)).filter(p -> Files.isDirectory(p, new LinkOption[0]) || Files.isRegularFile(p, new LinkOption[0])).forEach(this::addImport);
            }
            catch (IOException e) {
                throw new ModelImportException("Error loading the contents of " + importPath, e);
            }
        } else if (Files.isRegularFile(importPath, new LinkOption[0])) {
            this.inputStreamModels.put(importPath.toString(), () -> {
                try {
                    return Files.newInputStream(importPath, new OpenOption[0]);
                }
                catch (IOException e) {
                    throw new ModelImportException("Unable to import Smithy model from " + importPath + ": " + e.getMessage(), e);
                }
            });
        } else {
            throw new ModelImportException("Cannot find import file: " + importPath);
        }
        return this;
    }

    public ModelAssembler addImport(URL url) {
        Objects.requireNonNull(url, "The provided url to ModelAssembler#addImport was null");
        this.inputStreamModels.put(url.toExternalForm(), () -> {
            try {
                URLConnection connection = url.openConnection();
                if (this.properties.containsKey(DISABLE_JAR_CACHE)) {
                    connection.setUseCaches(false);
                }
                return connection.getInputStream();
            }
            catch (IOException | UncheckedIOException e) {
                throw new ModelImportException("Unable to open Smithy model import URL: " + url.toExternalForm(), e);
            }
        });
        return this;
    }

    public ModelAssembler disablePrelude() {
        this.disablePrelude = true;
        return this;
    }

    public ModelAssembler addShape(Shape shape) {
        this.shapes.add((AbstractShapeBuilder<?, ?>)Shape.shapeToBuilder(shape));
        return this;
    }

    public ModelAssembler addShapes(Shape ... shapes) {
        for (Shape shape : shapes) {
            this.addShape(shape);
        }
        return this;
    }

    public ModelAssembler addModel(Model model) {
        this.mergeModels.add(model);
        return this;
    }

    public ModelAssembler putMetadata(String name, Node value) {
        this.metadata.put(Objects.requireNonNull(name), Objects.requireNonNull(value));
        return this;
    }

    public ModelAssembler discoverModels(ClassLoader loader) {
        return this.addDiscoveredModels(ModelDiscovery.findModels(loader));
    }

    public ModelAssembler discoverModels() {
        return this.addDiscoveredModels(ModelDiscovery.findModels());
    }

    private ModelAssembler addDiscoveredModels(List<URL> urls) {
        for (URL url : urls) {
            LOGGER.fine(() -> "Discovered Smithy model: " + url);
            this.addImport(url);
        }
        return this;
    }

    public ModelAssembler putProperty(String setting, Object value) {
        this.properties.put(setting, value);
        return this;
    }

    public ModelAssembler removeProperty(String setting) {
        this.properties.remove(setting);
        return this;
    }

    public ModelAssembler disableValidation() {
        this.disableValidation = true;
        return this;
    }

    public ValidatedResult<Model> assemble() {
        LoaderVisitor visitor = this.createLoaderVisitor();
        try {
            return this.doAssemble(visitor);
        }
        catch (SourceException e) {
            ValidatedResult<Model> modelResult = visitor.onEnd();
            ArrayList<ValidationEvent> events = new ArrayList<ValidationEvent>(modelResult.getValidationEvents());
            events.add(ValidationEvent.fromSourceException(e));
            return new ValidatedResult<Model>(modelResult.getResult().orElse(null), events);
        }
    }

    private LoaderVisitor createLoaderVisitor() {
        if (this.traitFactory == null) {
            this.traitFactory = LazyTraitFactoryHolder.INSTANCE;
        }
        return new LoaderVisitor(this.traitFactory, this.properties);
    }

    private ValidatedResult<Model> doAssemble(LoaderVisitor visitor) {
        if (!this.disablePrelude) {
            ModelAssembler.mergeModelIntoVisitor(Prelude.getPreludeModel(), visitor);
        }
        this.shapes.forEach(visitor::onShape);
        this.metadata.forEach(visitor::onMetadata);
        for (Model model : this.mergeModels) {
            ModelAssembler.mergeModelIntoVisitor(model, visitor);
        }
        for (Node node : this.documentNodes) {
            ModelLoader.loadParsedNode(node, visitor);
        }
        for (Map.Entry entry : this.inputStreamModels.entrySet()) {
            if (ModelLoader.load((String)entry.getKey(), (Supplier)entry.getValue(), visitor)) continue;
            LOGGER.warning(() -> "No ModelLoader was able to load " + (String)modelEntry.getKey());
        }
        ValidatedResult<Model> modelResult = visitor.onEnd();
        return !modelResult.getResult().isPresent() ? modelResult : this.validate(modelResult.getResult().get(), modelResult.getValidationEvents());
    }

    private static void mergeModelIntoVisitor(Model model, LoaderVisitor visitor) {
        model.getMetadata().forEach(visitor::onMetadata);
        model.shapes().forEach(visitor::onShape);
    }

    private ValidatedResult<Model> validate(Model model, List<ValidationEvent> modelResultEvents) {
        if (this.disableValidation) {
            return new ValidatedResult<Model>(model, modelResultEvents);
        }
        if (this.validatorFactory == null) {
            this.validatorFactory = LazyValidatorFactoryHolder.INSTANCE;
        }
        List<ValidationEvent> events = ModelValidator.validate(model, this.validatorFactory, this.assembleValidators());
        events.addAll(modelResultEvents);
        return new ValidatedResult<Model>(model, events);
    }

    private List<Validator> assembleValidators() {
        ArrayList<Validator> copiedValidators = new ArrayList<Validator>(this.validatorFactory.loadBuiltinValidators());
        copiedValidators.addAll(this.validators);
        return copiedValidators;
    }

    static final class LazyTraitFactoryHolder {
        static final TraitFactory INSTANCE = TraitFactory.createServiceFactory(ModelAssembler.class.getClassLoader());

        LazyTraitFactoryHolder() {
        }
    }

    private static final class LazyValidatorFactoryHolder {
        static final ValidatorFactory INSTANCE = ValidatorFactory.createServiceFactory(ModelAssembler.class.getClassLoader());

        private LazyValidatorFactoryHolder() {
        }
    }
}

