/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.loader.LoaderVisitor;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.BigDecimalShape;
import software.amazon.smithy.model.shapes.BigIntegerShape;
import software.amazon.smithy.model.shapes.BlobShape;
import software.amazon.smithy.model.shapes.BooleanShape;
import software.amazon.smithy.model.shapes.ByteShape;
import software.amazon.smithy.model.shapes.CollectionShape;
import software.amazon.smithy.model.shapes.DocumentShape;
import software.amazon.smithy.model.shapes.DoubleShape;
import software.amazon.smithy.model.shapes.FloatShape;
import software.amazon.smithy.model.shapes.IntegerShape;
import software.amazon.smithy.model.shapes.ListShape;
import software.amazon.smithy.model.shapes.LongShape;
import software.amazon.smithy.model.shapes.MapShape;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.ResourceShape;
import software.amazon.smithy.model.shapes.ServiceShape;
import software.amazon.smithy.model.shapes.SetShape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShortShape;
import software.amazon.smithy.model.shapes.StringShape;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.TimestampShape;
import software.amazon.smithy.model.shapes.UnionShape;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.SetUtils;

enum AstModelLoader {
    INSTANCE;

    private static final String METADATA = "metadata";
    private static final String MEMBERS = "members";
    private static final String SHAPES = "shapes";
    private static final String TRAITS = "traits";
    private static final String TYPE = "type";
    private static final String TARGET = "target";
    private static final List<String> TOP_LEVEL_PROPERTIES;
    private static final List<String> APPLY_PROPERTIES;
    private static final List<String> SIMPLE_PROPERTY_NAMES;
    private static final List<String> STRUCTURE_AND_UNION_PROPERTY_NAMES;
    private static final List<String> COLLECTION_PROPERTY_NAMES;
    private static final List<String> MAP_PROPERTY_NAMES;
    private static final Set<String> MEMBER_PROPERTIES;
    private static final Set<String> REFERENCE_PROPERTIES;
    private static final Set<String> OPERATION_PROPERTY_NAMES;
    private static final Set<String> RESOURCE_PROPERTIES;
    private static final Set<String> SERVICE_PROPERTIES;

    void load(ObjectNode model, LoaderVisitor visitor) {
        visitor.checkForAdditionalProperties(model, null, TOP_LEVEL_PROPERTIES);
        this.loadMetadata(model, visitor);
        this.loadShapes(model, visitor);
    }

    private void loadMetadata(ObjectNode model, LoaderVisitor visitor) {
        try {
            model.getObjectMember(METADATA).ifPresent(metadata -> {
                for (Map.Entry<String, Node> entry : metadata.getStringMap().entrySet()) {
                    visitor.onMetadata(entry.getKey(), entry.getValue());
                }
            });
        }
        catch (SourceException e) {
            visitor.onError(ValidationEvent.fromSourceException(e));
        }
    }

    private void loadShapes(ObjectNode model, LoaderVisitor visitor) {
        model.getObjectMember(SHAPES).ifPresent(shapes -> {
            for (Map.Entry<StringNode, Node> entry : shapes.getMembers().entrySet()) {
                ShapeId id = entry.getKey().expectShapeId();
                ObjectNode definition = entry.getValue().expectObjectNode();
                String type = definition.expectStringMember(TYPE).getValue();
                try {
                    this.loadShape(id, type, definition, visitor);
                }
                catch (SourceException e) {
                    ValidationEvent event = ValidationEvent.fromSourceException(e).toBuilder().shapeId(id).build();
                    visitor.onError(event);
                }
            }
        });
    }

    private void loadShape(ShapeId id, String type, ObjectNode value, LoaderVisitor visitor) {
        switch (type) {
            case "blob": {
                this.loadSimpleShape(id, value, BlobShape.builder(), visitor);
                break;
            }
            case "boolean": {
                this.loadSimpleShape(id, value, BooleanShape.builder(), visitor);
                break;
            }
            case "byte": {
                this.loadSimpleShape(id, value, ByteShape.builder(), visitor);
                break;
            }
            case "short": {
                this.loadSimpleShape(id, value, ShortShape.builder(), visitor);
                break;
            }
            case "integer": {
                this.loadSimpleShape(id, value, IntegerShape.builder(), visitor);
                break;
            }
            case "long": {
                this.loadSimpleShape(id, value, LongShape.builder(), visitor);
                break;
            }
            case "float": {
                this.loadSimpleShape(id, value, FloatShape.builder(), visitor);
                break;
            }
            case "double": {
                this.loadSimpleShape(id, value, DoubleShape.builder(), visitor);
                break;
            }
            case "document": {
                this.loadSimpleShape(id, value, DocumentShape.builder(), visitor);
                break;
            }
            case "bigDecimal": {
                this.loadSimpleShape(id, value, BigDecimalShape.builder(), visitor);
                break;
            }
            case "bigInteger": {
                this.loadSimpleShape(id, value, BigIntegerShape.builder(), visitor);
                break;
            }
            case "string": {
                this.loadSimpleShape(id, value, StringShape.builder(), visitor);
                break;
            }
            case "timestamp": {
                this.loadSimpleShape(id, value, TimestampShape.builder(), visitor);
                break;
            }
            case "list": {
                this.loadCollection(id, value, ListShape.builder(), visitor);
                break;
            }
            case "set": {
                this.loadCollection(id, value, SetShape.builder(), visitor);
                break;
            }
            case "map": {
                this.loadMap(id, value, visitor);
                break;
            }
            case "resource": {
                this.loadResource(id, value, visitor);
                break;
            }
            case "service": {
                this.loadService(id, value, visitor);
                break;
            }
            case "structure": {
                this.loadStructure(id, value, visitor);
                break;
            }
            case "union": {
                this.loadUnion(id, value, visitor);
                break;
            }
            case "operation": {
                this.loadOperation(id, value, visitor);
                break;
            }
            case "apply": {
                visitor.checkForAdditionalProperties(value, id, APPLY_PROPERTIES);
                this.applyTraits(id, value.expectObjectMember(TRAITS), visitor);
                break;
            }
            default: {
                throw new SourceException("Invalid shape `type`: " + type, value);
            }
        }
    }

    private void applyTraits(ShapeId id, ObjectNode traits, LoaderVisitor visitor) {
        for (Map.Entry<StringNode, Node> traitNode : traits.getMembers().entrySet()) {
            ShapeId traitId = traitNode.getKey().expectShapeId();
            visitor.onTrait(id, traitId, traitNode.getValue());
        }
    }

    private void applyShapeTraits(ShapeId id, ObjectNode node, LoaderVisitor visitor) {
        node.getObjectMember(TRAITS).ifPresent(traits -> this.applyTraits(id, (ObjectNode)traits, visitor));
    }

    private void loadMember(LoaderVisitor visitor, ShapeId id, ObjectNode targetNode) {
        visitor.checkForAdditionalProperties(targetNode, id, MEMBER_PROPERTIES);
        MemberShape.Builder builder = (MemberShape.Builder)((MemberShape.Builder)MemberShape.builder().source(targetNode.getSourceLocation())).id(id);
        ShapeId target = targetNode.expectStringMember(TARGET).expectShapeId();
        builder.target(target);
        this.applyShapeTraits(id, targetNode, visitor);
        visitor.onShape(builder);
    }

    private void loadCollection(ShapeId id, ObjectNode node, CollectionShape.Builder builder, LoaderVisitor visitor) {
        visitor.checkForAdditionalProperties(node, id, COLLECTION_PROPERTY_NAMES);
        this.applyShapeTraits(id, node, visitor);
        this.loadMember(visitor, id.withMember("member"), node.expectObjectMember("member"));
        visitor.onShape((AbstractShapeBuilder)((AbstractShapeBuilder)builder.id(id)).source(node.getSourceLocation()));
    }

    private void loadMap(ShapeId id, ObjectNode node, LoaderVisitor visitor) {
        visitor.checkForAdditionalProperties(node, id, MAP_PROPERTY_NAMES);
        this.loadMember(visitor, id.withMember("key"), node.expectObjectMember("key"));
        this.loadMember(visitor, id.withMember("value"), node.expectObjectMember("value"));
        this.applyShapeTraits(id, node, visitor);
        visitor.onShape((AbstractShapeBuilder)((MapShape.Builder)MapShape.builder().id(id)).source(node.getSourceLocation()));
    }

    private void loadOperation(ShapeId id, ObjectNode node, LoaderVisitor visitor) {
        visitor.checkForAdditionalProperties(node, id, OPERATION_PROPERTY_NAMES);
        this.applyShapeTraits(id, node, visitor);
        OperationShape.Builder builder = ((OperationShape.Builder)((OperationShape.Builder)OperationShape.builder().id(id)).source(node.getSourceLocation())).addErrors(this.loadOptionalTargetList(visitor, id, node, "errors"));
        this.loadOptionalTarget(visitor, id, node, "input").ifPresent(builder::input);
        this.loadOptionalTarget(visitor, id, node, "output").ifPresent(builder::output);
        visitor.onShape(builder);
    }

    private void loadResource(ShapeId id, ObjectNode node, LoaderVisitor visitor) {
        visitor.checkForAdditionalProperties(node, id, RESOURCE_PROPERTIES);
        this.applyShapeTraits(id, node, visitor);
        ResourceShape.Builder builder = (ResourceShape.Builder)((ResourceShape.Builder)ResourceShape.builder().id(id)).source(node.getSourceLocation());
        this.loadOptionalTarget(visitor, id, node, "put").ifPresent(builder::put);
        this.loadOptionalTarget(visitor, id, node, "create").ifPresent(builder::create);
        this.loadOptionalTarget(visitor, id, node, "read").ifPresent(builder::read);
        this.loadOptionalTarget(visitor, id, node, "update").ifPresent(builder::update);
        this.loadOptionalTarget(visitor, id, node, "delete").ifPresent(builder::delete);
        this.loadOptionalTarget(visitor, id, node, "list").ifPresent(builder::list);
        builder.operations(this.loadOptionalTargetList(visitor, id, node, "operations"));
        builder.collectionOperations(this.loadOptionalTargetList(visitor, id, node, "collectionOperations"));
        builder.resources(this.loadOptionalTargetList(visitor, id, node, "resources"));
        node.getObjectMember("identifiers").ifPresent(ids -> {
            for (Map.Entry<StringNode, Node> entry : ids.getMembers().entrySet()) {
                String name = entry.getKey().getValue();
                ShapeId target = this.loadReferenceBody(visitor, id, entry.getValue());
                builder.addIdentifier(name, target);
            }
        });
        visitor.onShape(builder);
    }

    private void loadService(ShapeId id, ObjectNode node, LoaderVisitor visitor) {
        visitor.checkForAdditionalProperties(node, id, SERVICE_PROPERTIES);
        this.applyShapeTraits(id, node, visitor);
        ServiceShape.Builder builder = (ServiceShape.Builder)((ServiceShape.Builder)new ServiceShape.Builder().id(id)).source(node.getSourceLocation());
        builder.version(node.expectStringMember("version").getValue());
        builder.operations(this.loadOptionalTargetList(visitor, id, node, "operations"));
        builder.resources(this.loadOptionalTargetList(visitor, id, node, "resources"));
        visitor.onShape(builder);
    }

    private void loadSimpleShape(ShapeId id, ObjectNode node, AbstractShapeBuilder builder, LoaderVisitor visitor) {
        visitor.checkForAdditionalProperties(node, id, SIMPLE_PROPERTY_NAMES);
        this.applyShapeTraits(id, node, visitor);
        visitor.onShape((AbstractShapeBuilder)((AbstractShapeBuilder)builder.id(id)).source(node.getSourceLocation()));
    }

    private void loadStructure(ShapeId id, ObjectNode node, LoaderVisitor visitor) {
        visitor.checkForAdditionalProperties(node, id, STRUCTURE_AND_UNION_PROPERTY_NAMES);
        visitor.onShape((AbstractShapeBuilder)((StructureShape.Builder)StructureShape.builder().id(id)).source(node.getSourceLocation()));
        this.finishLoadingStructOrUnionMembers(id, node, visitor);
    }

    private void loadUnion(ShapeId id, ObjectNode node, LoaderVisitor visitor) {
        visitor.checkForAdditionalProperties(node, id, STRUCTURE_AND_UNION_PROPERTY_NAMES);
        visitor.onShape((AbstractShapeBuilder)((UnionShape.Builder)UnionShape.builder().id(id)).source(node.getSourceLocation()));
        this.finishLoadingStructOrUnionMembers(id, node, visitor);
    }

    private void finishLoadingStructOrUnionMembers(ShapeId id, ObjectNode node, LoaderVisitor visitor) {
        this.applyShapeTraits(id, node, visitor);
        ObjectNode memberObject = node.getObjectMember(MEMBERS).orElse(Node.objectNode());
        for (Map.Entry<String, Node> entry : memberObject.getStringMap().entrySet()) {
            this.loadMember(visitor, id.withMember(entry.getKey()), entry.getValue().expectObjectNode());
        }
    }

    private Optional<ShapeId> loadOptionalTarget(LoaderVisitor visitor, ShapeId id, ObjectNode node, String member) {
        return node.getObjectMember(member).map(r -> this.loadReferenceBody(visitor, id, (Node)r));
    }

    private ShapeId loadReferenceBody(LoaderVisitor visitor, ShapeId id, Node reference) {
        ObjectNode referenceObject = reference.expectObjectNode();
        visitor.checkForAdditionalProperties(referenceObject, id, REFERENCE_PROPERTIES);
        return referenceObject.expectStringMember(TARGET).expectShapeId();
    }

    private List<ShapeId> loadOptionalTargetList(LoaderVisitor visitor, ShapeId id, ObjectNode node, String member) {
        return node.getArrayMember(member).map(array -> {
            ArrayList<ShapeId> ids = new ArrayList<ShapeId>(array.size());
            for (Node element : array.getElements()) {
                ids.add(this.loadReferenceBody(visitor, id, element));
            }
            return ids;
        }).orElseGet(Collections::emptyList);
    }

    static {
        TOP_LEVEL_PROPERTIES = ListUtils.of((Object[])new String[]{"smithy", SHAPES, METADATA});
        APPLY_PROPERTIES = ListUtils.of((Object[])new String[]{TYPE, TRAITS});
        SIMPLE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{TYPE, TRAITS});
        STRUCTURE_AND_UNION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{TYPE, MEMBERS, TRAITS});
        COLLECTION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{TYPE, "member", TRAITS});
        MAP_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{TYPE, "key", "value", TRAITS});
        MEMBER_PROPERTIES = SetUtils.of((Object[])new String[]{TARGET, TRAITS});
        REFERENCE_PROPERTIES = SetUtils.of((Object)TARGET);
        OPERATION_PROPERTY_NAMES = SetUtils.of((Object[])new String[]{TYPE, "input", "output", "errors", TRAITS});
        RESOURCE_PROPERTIES = SetUtils.of((Object[])new String[]{TYPE, "create", "read", "update", "delete", "list", "put", "identifiers", "resources", "operations", "collectionOperations", TRAITS});
        SERVICE_PROPERTIES = SetUtils.of((Object[])new String[]{TYPE, "version", "operations", "resources", TRAITS});
    }
}

