/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.LoaderVisitor;
import software.amazon.smithy.model.loader.ModelImportException;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.loader.SmithyModelLexer;
import software.amazon.smithy.model.loader.UseException;
import software.amazon.smithy.model.node.ArrayNode;
import software.amazon.smithy.model.node.BooleanNode;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.NullNode;
import software.amazon.smithy.model.node.NumberNode;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.BigDecimalShape;
import software.amazon.smithy.model.shapes.BigIntegerShape;
import software.amazon.smithy.model.shapes.BlobShape;
import software.amazon.smithy.model.shapes.BooleanShape;
import software.amazon.smithy.model.shapes.ByteShape;
import software.amazon.smithy.model.shapes.CollectionShape;
import software.amazon.smithy.model.shapes.DocumentShape;
import software.amazon.smithy.model.shapes.DoubleShape;
import software.amazon.smithy.model.shapes.FloatShape;
import software.amazon.smithy.model.shapes.IntegerShape;
import software.amazon.smithy.model.shapes.ListShape;
import software.amazon.smithy.model.shapes.LongShape;
import software.amazon.smithy.model.shapes.MapShape;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.ResourceShape;
import software.amazon.smithy.model.shapes.ServiceShape;
import software.amazon.smithy.model.shapes.SetShape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeIdSyntaxException;
import software.amazon.smithy.model.shapes.ShortShape;
import software.amazon.smithy.model.shapes.StringShape;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.TimestampShape;
import software.amazon.smithy.model.shapes.ToShapeId;
import software.amazon.smithy.model.shapes.UnionShape;
import software.amazon.smithy.model.traits.DocumentationTrait;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.model.validation.ValidationUtils;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.MapUtils;
import software.amazon.smithy.utils.Pair;
import software.amazon.smithy.utils.SetUtils;

final class IdlModelLoader {
    private static final String PUT_KEY = "put";
    private static final String CREATE_KEY = "create";
    private static final String READ_KEY = "read";
    private static final String UPDATE_KEY = "update";
    private static final String DELETE_KEY = "delete";
    private static final String LIST_KEY = "list";
    private static final String RESOURCES_KEY = "resources";
    private static final String OPERATIONS_KEY = "operations";
    private static final String COLLECTION_OPERATIONS_KEY = "collectionOperations";
    private static final String IDENTIFIERS_KEY = "identifiers";
    private static final String VERSION_KEY = "version";
    private static final String TYPE_KEY = "type";
    static final Collection<String> RESOURCE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "create", "read", "update", "delete", "list", "identifiers", "resources", "operations", "put", "collectionOperations"});
    static final List<String> SERVICE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "version", "operations", "resources"});
    private static final Set<String> STATEMENTS = SetUtils.of((Object[])new String[]{"namespace", "use", "service", "operation", "resource", "structure", "union", "list", "set", "map", "boolean", "string", "blob", "byte", "short", "integer", "long", "float", "document", "double", "bigInteger", "bigDecimal", "timestamp", "metadata", "apply"});
    private static final Set<String> SUPPORTS_TRAITS = SetUtils.of((Object[])new String[]{"bigDecimal", "bigInteger", "blob", "boolean", "byte", "document", "double", "float", "integer", "list", "long", "map", "operation", "resource", "service", "set", "short", "string", "structure", "timestamp", "union"});
    private static final Collection<String> MAP_KEYS = ListUtils.of((Object[])new String[]{"key", "value"});
    private static final Collection<String> OPERATION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"input", "output", "errors"});
    private final String filename;
    private final SmithyModelLexer lexer;
    private final LoaderVisitor visitor;
    private final List<TraitEntry> pendingTraits = new ArrayList<TraitEntry>();
    private final Map<String, ShapeId> useShapes = new HashMap<String, ShapeId>();
    private SmithyModelLexer.Token current;
    private DocComment pendingDocComment;
    private String definedVersion;
    private String namespace;
    private boolean definedMetadata;
    private boolean definedShapes;

    private IdlModelLoader(String filename, SmithyModelLexer lexer, LoaderVisitor visitor) {
        this.filename = filename;
        this.visitor = visitor;
        this.lexer = lexer;
        while (!this.eof()) {
            SmithyModelLexer.Token token = this.expect(SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.ANNOTATION, SmithyModelLexer.TokenType.CONTROL, SmithyModelLexer.TokenType.DOC);
            if (token.type == SmithyModelLexer.TokenType.UNQUOTED) {
                this.parseStatement(token);
                continue;
            }
            if (token.type == SmithyModelLexer.TokenType.ANNOTATION) {
                this.pendingTraits.add(this.parseTraitValue(token, TraitValueType.SHAPE));
                continue;
            }
            if (token.type == SmithyModelLexer.TokenType.CONTROL) {
                this.parseControlStatement(token);
                continue;
            }
            if (token.type != SmithyModelLexer.TokenType.DOC) continue;
            this.parseDocComment(token, false);
        }
    }

    public static void load(String path, Supplier<InputStream> contentSupplier, LoaderVisitor visitor) {
        try (SmithyModelLexer lexer = new SmithyModelLexer(path, contentSupplier.get());){
            new IdlModelLoader(path, lexer, visitor);
        }
        catch (IOException e) {
            throw new ModelImportException("Error loading " + path + ": " + e.getMessage(), e);
        }
    }

    private void parseStatement(SmithyModelLexer.Token token) {
        switch (token.lexeme) {
            case "namespace": {
                this.parseNamespace();
                break;
            }
            case "use": {
                this.parseUseStatement();
                break;
            }
            case "structure": {
                this.parseStructuredShape("structure", StructureShape.builder());
                break;
            }
            case "union": {
                this.parseStructuredShape("union", UnionShape.builder());
                break;
            }
            case "list": {
                this.parseCollection(LIST_KEY, ListShape.builder());
                break;
            }
            case "set": {
                this.parseCollection("set", SetShape.builder());
                break;
            }
            case "map": {
                this.parseMap();
                break;
            }
            case "boolean": {
                this.parseSimpleShape(BooleanShape.builder());
                break;
            }
            case "string": {
                this.parseSimpleShape(StringShape.builder());
                break;
            }
            case "blob": {
                this.parseSimpleShape(BlobShape.builder());
                break;
            }
            case "byte": {
                this.parseSimpleShape(ByteShape.builder());
                break;
            }
            case "short": {
                this.parseSimpleShape(ShortShape.builder());
                break;
            }
            case "integer": {
                this.parseSimpleShape(IntegerShape.builder());
                break;
            }
            case "long": {
                this.parseSimpleShape(LongShape.builder());
                break;
            }
            case "float": {
                this.parseSimpleShape(FloatShape.builder());
                break;
            }
            case "document": {
                this.parseSimpleShape(DocumentShape.builder());
                break;
            }
            case "double": {
                this.parseSimpleShape(DoubleShape.builder());
                break;
            }
            case "bigInteger": {
                this.parseSimpleShape(BigIntegerShape.builder());
                break;
            }
            case "bigDecimal": {
                this.parseSimpleShape(BigDecimalShape.builder());
                break;
            }
            case "timestamp": {
                this.parseSimpleShape(TimestampShape.builder());
                break;
            }
            case "service": {
                this.parseService();
                break;
            }
            case "operation": {
                this.parseOperation();
                break;
            }
            case "resource": {
                this.parseResource();
                break;
            }
            case "metadata": {
                this.parseMetadata();
                break;
            }
            case "apply": {
                this.parseApply();
                break;
            }
            default: {
                throw this.syntax(String.format("Expected one of %s", ValidationUtils.tickedList(STATEMENTS)));
            }
        }
    }

    private void parseNamespace() {
        if (this.namespace != null) {
            throw this.syntax("Only a single namespace can be declared per/file. The previous namespace was set to `" + this.namespace + "`.");
        }
        String parsedNamespace = this.expect((SmithyModelLexer.TokenType[])new SmithyModelLexer.TokenType[]{SmithyModelLexer.TokenType.UNQUOTED}).lexeme;
        if (!ShapeId.isValidNamespace(parsedNamespace)) {
            throw this.syntax(String.format("Invalid namespace name `%s`", parsedNamespace));
        }
        this.namespace = parsedNamespace;
    }

    private void parseUseStatement() {
        if (this.namespace == null) {
            throw this.syntax("Use statements must appear after a namespace is defined");
        }
        if (this.definedShapes) {
            throw this.syntax("A use statement must come before any shape definition");
        }
        try {
            SmithyModelLexer.Token namespaceToken = this.expect(SmithyModelLexer.TokenType.UNQUOTED);
            ShapeId target = ShapeId.from(namespaceToken.lexeme);
            ShapeId previous = this.useShapes.put(target.getName(), target);
            if (previous != null) {
                String message = String.format("Cannot use name `%s` because it conflicts with `%s`", target, previous);
                throw new UseException(message, namespaceToken.getSourceLocation());
            }
            this.expectNewline();
        }
        catch (ShapeIdSyntaxException e) {
            throw this.syntax(e.getMessage());
        }
    }

    private void parseControlStatement(SmithyModelLexer.Token token) {
        if (this.definedMetadata || this.namespace != null) {
            throw this.syntax("A control statement must come before any namespace, metadata, or shape");
        }
        String key = token.lexeme;
        Node value = this.parseNodeValue(this.next());
        if (key.equals(VERSION_KEY)) {
            this.onVersion(value);
        } else {
            this.visitor.onError(ValidationEvent.builder().eventId("Model").sourceLocation(value).severity(Severity.WARNING).message(String.format("Unknown control statement `%s` with value `%s", key, Node.printJson(value))).build());
        }
    }

    private void onVersion(Node value) {
        if (!value.isStringNode()) {
            value.expectStringNode("The $version control statement must have a string value, but found " + Node.printJson(value));
        }
        String parsedVersion = value.expectStringNode().getValue();
        if (this.definedVersion != null) {
            throw this.syntax("Cannot define multiple versions in the same file");
        }
        if (!this.visitor.isVersionSupported(parsedVersion)) {
            throw this.syntax("Unsupported Smithy version number: " + parsedVersion);
        }
        this.definedVersion = parsedVersion;
    }

    private void parseMetadata() {
        if (this.namespace != null) {
            throw this.syntax("Metadata statements must appear before a namespace statement");
        }
        this.definedMetadata = true;
        String key = this.expect((SmithyModelLexer.TokenType[])new SmithyModelLexer.TokenType[]{SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED}).lexeme;
        this.expect(SmithyModelLexer.TokenType.EQUAL);
        this.visitor.onMetadata(key, this.parseNode());
        this.expectNewline();
    }

    private void parseDocComment(SmithyModelLexer.Token token, boolean memberScope) {
        StringBuilder builder = new StringBuilder(token.getDocContents());
        while (this.test(SmithyModelLexer.TokenType.DOC)) {
            builder.append('\n').append(this.next().getDocContents());
        }
        this.pendingDocComment = new DocComment(builder.toString(), token.getSourceLocation());
        if (!this.peek().isPresent()) {
            throw this.syntax("Found a documentation comment that doesn't document anything");
        }
        SmithyModelLexer.Token next = this.peek().get();
        if (next.type != SmithyModelLexer.TokenType.ANNOTATION && (next.type != SmithyModelLexer.TokenType.UNQUOTED || !memberScope && !SUPPORTS_TRAITS.contains(next.lexeme))) {
            throw this.syntax("Documentation cannot be applied to `" + next.lexeme + "`");
        }
    }

    private TraitEntry parseTraitValue(SmithyModelLexer.Token token, TraitValueType type) {
        try {
            this.requireNamespaceOrThrow();
            ShapeId.fromOptionalNamespace(this.namespace, token.lexeme);
            TraitEntry result = this.parseTraitValueBody(token.lexeme);
            if (type == TraitValueType.APPLY) {
                return result;
            }
            if (!this.peek().isPresent()) {
                throw this.syntax("Found a trait doesn't apply to anything");
            }
            SmithyModelLexer.Token next = this.peek().get();
            if (next.type != SmithyModelLexer.TokenType.ANNOTATION && (next.type != SmithyModelLexer.TokenType.UNQUOTED || type != TraitValueType.MEMBER && !SUPPORTS_TRAITS.contains(next.lexeme))) {
                throw this.syntax("Traits cannot be applied to `" + next.lexeme + "`");
            }
            return result;
        }
        catch (ShapeIdSyntaxException e) {
            throw this.syntax("Invalid trait name syntax. Trait names must adhere to the same syntax as shape IDs.");
        }
    }

    private void requireNamespaceOrThrow() {
        if (this.namespace == null) {
            throw this.syntax("A namespace must be set before shapes or traits can be defined");
        }
    }

    private TraitEntry parseTraitValueBody(String traitName) {
        if (!this.test(SmithyModelLexer.TokenType.LPAREN)) {
            return new TraitEntry(traitName, new NullNode(this.currentLocation()), true);
        }
        this.expect(SmithyModelLexer.TokenType.LPAREN);
        SmithyModelLexer.Token next = this.expect(SmithyModelLexer.TokenType.RPAREN, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.LBRACKET, SmithyModelLexer.TokenType.NUMBER);
        if (next.type == SmithyModelLexer.TokenType.RPAREN) {
            return new TraitEntry(traitName, new ObjectNode(MapUtils.of(), next.getSourceLocation()), false);
        }
        if (this.test(SmithyModelLexer.TokenType.COLON)) {
            if (next.type == SmithyModelLexer.TokenType.QUOTED || next.type == SmithyModelLexer.TokenType.UNQUOTED) {
                return new TraitEntry(traitName, this.parseObjectNodeWithKey(this.currentLocation(), SmithyModelLexer.TokenType.RPAREN, next), false);
            }
            throw this.syntax("Expected a string to start a trait value object");
        }
        Node result = next.type == SmithyModelLexer.TokenType.LBRACKET ? this.parseArrayNode(next.getSourceLocation()) : (next.type == SmithyModelLexer.TokenType.NUMBER ? this.parseNumber(next) : this.parseNodeValue(next));
        this.expect(SmithyModelLexer.TokenType.RPAREN);
        return new TraitEntry(traitName, result, false);
    }

    private Node parseNode() {
        return this.parseNodeValue(this.expect(SmithyModelLexer.TokenType.LBRACE, SmithyModelLexer.TokenType.LBRACKET, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.NUMBER));
    }

    private Node parseNodeValue(SmithyModelLexer.Token token) {
        switch (token.type) {
            case LBRACE: {
                return this.parseObjectNode(token.getSourceLocation(), SmithyModelLexer.TokenType.RBRACE);
            }
            case LBRACKET: {
                return this.parseArrayNode(token.getSourceLocation());
            }
            case QUOTED: {
                return new StringNode(token.lexeme, token.getSourceLocation());
            }
            case NUMBER: {
                return this.parseNumber(token);
            }
            case UNQUOTED: {
                return this.parseUnquotedNode(token);
            }
        }
        throw new IllegalStateException("Parse node value not expected to be called with: " + token);
    }

    private Node parseUnquotedNode(SmithyModelLexer.Token token) {
        switch (token.lexeme) {
            case "true": {
                return new BooleanNode(true, token.getSourceLocation());
            }
            case "false": {
                return new BooleanNode(false, token.getSourceLocation());
            }
            case "null": {
                return new NullNode(token.getSourceLocation());
            }
        }
        Pair<StringNode, Consumer<String>> pair = StringNode.createLazyString(token.lexeme, token.getSourceLocation());
        Consumer consumer = (Consumer)pair.right;
        this.onShapeTarget(token.lexeme, token, id -> consumer.accept(id.toString()));
        return (Node)pair.left;
    }

    private void onShapeTarget(String target, FromSourceLocation sourceLocation, Consumer<ShapeId> resolver) {
        if (this.useShapes.containsKey(target)) {
            resolver.accept(this.useShapes.get(target));
            return;
        }
        try {
            ShapeId expectedId;
            ShapeId shapeId = expectedId = this.namespace == null ? ShapeId.from(target) : ShapeId.fromOptionalNamespace(this.namespace, target);
            if (this.isRealizedShapeId(expectedId, target)) {
                resolver.accept(expectedId);
            } else {
                this.visitor.addForwardReference(expectedId, resolver);
            }
        }
        catch (ShapeIdSyntaxException e) {
            throw new SourceException("Error resolving shape target; " + e.getMessage(), sourceLocation, e);
        }
    }

    private NumberNode parseNumber(SmithyModelLexer.Token token) {
        if (token.lexeme.contains("e") || token.lexeme.contains(".")) {
            return new NumberNode(Double.valueOf(token.lexeme), token.getSourceLocation());
        }
        return new NumberNode(Long.parseLong(token.lexeme), token.getSourceLocation());
    }

    private ObjectNode parseObjectNode(SourceLocation location, SmithyModelLexer.TokenType closing) {
        return this.parseObjectNodeWithKey(location, closing, this.expect(SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, closing));
    }

    private ObjectNode parseObjectNodeWithKey(SourceLocation sloc, SmithyModelLexer.TokenType closing, SmithyModelLexer.Token key) {
        LinkedHashMap<StringNode, Node> entries = new LinkedHashMap<StringNode, Node>();
        while (key.type != closing) {
            this.expect(SmithyModelLexer.TokenType.COLON);
            Node value = this.parseNode();
            entries.put(new StringNode(key.lexeme, key.getSourceLocation()), value);
            if (this.expect((SmithyModelLexer.TokenType[])new SmithyModelLexer.TokenType[]{closing, SmithyModelLexer.TokenType.COMMA}).type == closing) break;
            key = this.expect(closing, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED);
        }
        return new ObjectNode(entries, sloc);
    }

    private ArrayNode parseArrayNode(SourceLocation location) {
        ArrayList<Node> values = new ArrayList<Node>();
        SmithyModelLexer.Token next = this.expect(SmithyModelLexer.TokenType.LBRACE, SmithyModelLexer.TokenType.LBRACKET, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.NUMBER, SmithyModelLexer.TokenType.RBRACKET);
        while (next.type != SmithyModelLexer.TokenType.RBRACKET) {
            values.add(this.parseNodeValue(next));
            next = this.expect(SmithyModelLexer.TokenType.RBRACKET, SmithyModelLexer.TokenType.COMMA);
            if (next.type == SmithyModelLexer.TokenType.RBRACKET) break;
            next = this.expect(SmithyModelLexer.TokenType.RBRACKET, SmithyModelLexer.TokenType.LBRACE, SmithyModelLexer.TokenType.LBRACKET, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.NUMBER);
        }
        return new ArrayNode(values, location);
    }

    private void parseSimpleShape(AbstractShapeBuilder builder) {
        builder.source(this.currentLocation());
        builder.id(this.parseShapeName());
        this.visitor.onShape(builder);
        this.expectNewline();
    }

    private ShapeId parseShapeName() {
        this.requireNamespaceOrThrow();
        this.definedShapes = true;
        SmithyModelLexer.Token nameToken = this.expect(SmithyModelLexer.TokenType.UNQUOTED);
        String name = nameToken.lexeme;
        if (this.useShapes.containsKey(name)) {
            String msg = String.format("shape name `%s` conflicts with imported shape `%s`", name, this.useShapes.get(name));
            throw new UseException(msg, nameToken);
        }
        try {
            ShapeId id = ShapeId.fromRelative(this.namespace, name);
            for (TraitEntry traitEntry : this.pendingTraits) {
                this.onDeferredTrait(id, traitEntry.traitName, traitEntry.value, traitEntry.isAnnotation);
            }
            this.pendingTraits.clear();
            this.collectPendingDocString(id);
            return id;
        }
        catch (ShapeIdSyntaxException e) {
            throw new ModelSyntaxException("Invalid shape name: " + name, nameToken);
        }
    }

    private void onDeferredTrait(ShapeId target, String traitName, Node traitValue, boolean isAnnotation) {
        this.onShapeTarget(traitName, traitValue.getSourceLocation(), id -> {
            if (isAnnotation) {
                this.visitor.onAnnotationTrait(target, (ShapeId)id, traitValue.expectNullNode());
            } else {
                this.visitor.onTrait(target, (ShapeId)id, traitValue);
            }
        });
    }

    private boolean isRealizedShapeId(ShapeId expectedId, String target) {
        return Objects.equals(this.namespace, "smithy.api") || this.visitor.hasDefinedShape(expectedId) || target.contains("#");
    }

    private void collectPendingDocString(ShapeId id) {
        if (this.pendingDocComment != null) {
            StringNode value = new StringNode(this.pendingDocComment.content, this.pendingDocComment.sourceLocation);
            this.pendingDocComment = null;
            this.visitor.onTrait(id, new DocumentationTrait(((Object)value).toString(), value.getSourceLocation()));
        }
    }

    private void parseStructuredShape(String shapeType, AbstractShapeBuilder builder) {
        builder.source(this.currentLocation());
        ShapeId id = this.parseShapeName();
        this.visitor.onShape((AbstractShapeBuilder)builder.id(id));
        this.parseStructuredBody(shapeType, id);
    }

    private void parseStructuredBody(String shapeType, ShapeId parent) {
        this.parseStructuredContents(shapeType, parent, SetUtils.of());
        this.expectNewline();
    }

    private void parseStructuredContents(String shapeType, ShapeId parent, Collection<String> requiredMembers) {
        this.expect(SmithyModelLexer.TokenType.LBRACE);
        ArrayList<TraitEntry> memberTraits = new ArrayList<TraitEntry>();
        Set<Object> remainingMembers = requiredMembers.isEmpty() ? SetUtils.of() : new HashSet<String>(requiredMembers);
        SmithyModelLexer.Token token = this.expect(SmithyModelLexer.TokenType.ANNOTATION, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.RBRACE, SmithyModelLexer.TokenType.DOC);
        while (token.type != SmithyModelLexer.TokenType.RBRACE) {
            if (token.type == SmithyModelLexer.TokenType.ANNOTATION) {
                memberTraits.add(this.parseTraitValue(token, TraitValueType.MEMBER));
            } else if (token.type == SmithyModelLexer.TokenType.DOC) {
                this.parseDocComment(token, true);
            } else {
                String memberName = token.lexeme;
                if (!requiredMembers.isEmpty()) {
                    if (!requiredMembers.contains(memberName)) {
                        throw this.syntax(String.format("Invalid member `%s` found in %s shape `%s`. Expected one of the following members: [%s]", memberName, shapeType, parent, ValidationUtils.tickedList(requiredMembers)));
                    }
                    remainingMembers.remove(memberName);
                }
                ShapeId memberId = parent.withMember(memberName);
                this.expect(SmithyModelLexer.TokenType.COLON);
                this.parseMember(memberId);
                for (TraitEntry traitEntry : memberTraits) {
                    this.onDeferredTrait(memberId, traitEntry.traitName, traitEntry.value, traitEntry.isAnnotation);
                }
                memberTraits.clear();
                this.collectPendingDocString(memberId);
                if (this.expect((SmithyModelLexer.TokenType[])new SmithyModelLexer.TokenType[]{SmithyModelLexer.TokenType.COMMA, SmithyModelLexer.TokenType.RBRACE}).type == SmithyModelLexer.TokenType.RBRACE) break;
            }
            token = this.expect(SmithyModelLexer.TokenType.ANNOTATION, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.RBRACE, SmithyModelLexer.TokenType.DOC);
        }
        if (!remainingMembers.isEmpty()) {
            throw this.syntax(String.format("Missing required members of %s shape `%s`: [%s]", shapeType, parent, ValidationUtils.tickedList(remainingMembers)));
        }
    }

    private void parseMember(ShapeId memberId) {
        MemberShape.Builder memberBuilder = (MemberShape.Builder)((MemberShape.Builder)MemberShape.builder().id(memberId)).source(this.currentLocation());
        this.visitor.onShape(memberBuilder);
        SmithyModelLexer.Token targetToken = this.expect(SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.QUOTED);
        this.onShapeTarget(targetToken.lexeme, targetToken, memberBuilder::target);
    }

    private void parseCollection(String shapeType, CollectionShape.Builder builder) {
        builder.source(this.currentLocation());
        ShapeId id = this.parseShapeName();
        this.parseStructuredContents(shapeType, id, SetUtils.of((Object)"member"));
        this.visitor.onShape((AbstractShapeBuilder)builder.id(id));
        this.expectNewline();
    }

    private void parseMap() {
        SourceLocation sourceLocation = this.currentLocation();
        ShapeId id = this.parseShapeName();
        MapShape.Builder builder = (MapShape.Builder)((MapShape.Builder)MapShape.builder().id(id)).source(sourceLocation);
        this.parseStructuredContents("map", id, MAP_KEYS);
        this.visitor.onShape(builder);
        this.expectNewline();
    }

    private void parseApply() {
        this.requireNamespaceOrThrow();
        SmithyModelLexer.Token nextToken = this.expect(SmithyModelLexer.TokenType.UNQUOTED);
        String name = nextToken.lexeme;
        SmithyModelLexer.Token token = this.expect(SmithyModelLexer.TokenType.ANNOTATION);
        TraitEntry traitEntry = this.parseTraitValue(token, TraitValueType.APPLY);
        this.expectNewline();
        this.onShapeTarget(name, nextToken.getSourceLocation(), id -> this.onDeferredTrait((ShapeId)id, traitEntry.traitName, traitEntry.value, traitEntry.isAnnotation));
    }

    private boolean eof() {
        return !this.lexer.hasNext();
    }

    private SmithyModelLexer.Token current() {
        return Objects.requireNonNull(this.current, "Call to next must occur before accessing current");
    }

    private SourceLocation currentLocation() {
        return this.current.getSourceLocation();
    }

    private SmithyModelLexer.Token next() {
        if (!this.lexer.hasNext()) {
            throw this.syntax("Unexpected EOF", this.current != null ? this.current.span : 0);
        }
        this.current = this.lexer.next();
        return this.current;
    }

    private SmithyModelLexer.Token expect(SmithyModelLexer.TokenType ... tokens) {
        SmithyModelLexer.Token next = this.next();
        for (SmithyModelLexer.TokenType type : tokens) {
            if (type != next.type) continue;
            return next;
        }
        if (next.type != SmithyModelLexer.TokenType.ERROR) {
            throw this.syntax(tokens.length == 1 ? "Expected " + (Object)((Object)tokens[0]) : "Expected one of " + Arrays.toString((Object[])tokens));
        }
        if (next.lexeme.contains("'") || next.lexeme.contains("\"")) {
            throw this.syntax("Unexpected syntax. Perhaps an unclosed quote?");
        }
        if (next.errorMessage == null) {
            throw this.syntax("Unexpected syntax");
        }
        throw this.syntax(next.errorMessage);
    }

    private boolean test(SmithyModelLexer.TokenType type) {
        return !this.eof() && this.lexer.peek().type == type;
    }

    private Optional<SmithyModelLexer.Token> peek() {
        return Optional.ofNullable(this.lexer.peek());
    }

    private void expectNewline() {
        if (this.peek().isPresent() && this.peek().get().line == this.current().line) {
            this.next();
            throw this.syntax("Expected a new line before this token");
        }
    }

    private ModelSyntaxException syntax(String message) {
        return this.syntax(message, 0);
    }

    private ModelSyntaxException syntax(String message, int offset) {
        SmithyModelLexer.Token token = this.current();
        int line = token.line;
        int column = token.column + offset;
        String lexeme = token.lexeme;
        String formatted = String.format("Parse error at line %d, column %d near `%s`: %s", line, column, lexeme, message);
        return new ModelSyntaxException(formatted, new SourceLocation(this.filename, line, column));
    }

    private void parseService() {
        SourceLocation sourceLocation = this.currentLocation();
        ShapeId shapeId = this.parseShapeName();
        ServiceShape.Builder builder = (ServiceShape.Builder)((ServiceShape.Builder)new ServiceShape.Builder().id(shapeId)).source(sourceLocation);
        ObjectNode shapeNode = this.parseObjectNode(this.expect(SmithyModelLexer.TokenType.LBRACE).getSourceLocation(), SmithyModelLexer.TokenType.RBRACE);
        this.visitor.checkForAdditionalProperties(shapeNode, shapeId, SERVICE_PROPERTY_NAMES);
        builder.version(shapeNode.expectStringMember(VERSION_KEY).getValue());
        IdlModelLoader.optionalIdList(shapeNode, shapeId.getNamespace(), OPERATIONS_KEY).forEach(builder::addOperation);
        IdlModelLoader.optionalIdList(shapeNode, shapeId.getNamespace(), RESOURCES_KEY).forEach(builder::addResource);
        this.visitor.onShape(builder);
        this.expectNewline();
    }

    static Optional<ShapeId> optionalId(ObjectNode node, String namespace, String name) {
        return node.getStringMember(name).map(stringNode -> stringNode.expectShapeId(namespace));
    }

    static List<ShapeId> optionalIdList(ObjectNode node, String namespace, String name) {
        return node.getArrayMember(name).map(array -> array.getElements().stream().map(Node::expectStringNode).map(s -> s.expectShapeId(namespace)).collect(Collectors.toList())).orElseGet(Collections::emptyList);
    }

    private void parseResource() {
        SourceLocation sourceLocation = this.currentLocation();
        ShapeId shapeId = this.parseShapeName();
        ResourceShape.Builder builder = (ResourceShape.Builder)((ResourceShape.Builder)ResourceShape.builder().id(shapeId)).source(sourceLocation);
        this.visitor.onShape(builder);
        ObjectNode shapeNode = this.parseObjectNode(this.expect(SmithyModelLexer.TokenType.LBRACE).getSourceLocation(), SmithyModelLexer.TokenType.RBRACE);
        this.visitor.checkForAdditionalProperties(shapeNode, shapeId, RESOURCE_PROPERTY_NAMES);
        IdlModelLoader.optionalId(shapeNode, shapeId.getNamespace(), PUT_KEY).ifPresent(builder::put);
        IdlModelLoader.optionalId(shapeNode, shapeId.getNamespace(), CREATE_KEY).ifPresent(builder::create);
        IdlModelLoader.optionalId(shapeNode, shapeId.getNamespace(), READ_KEY).ifPresent(builder::read);
        IdlModelLoader.optionalId(shapeNode, shapeId.getNamespace(), UPDATE_KEY).ifPresent(builder::update);
        IdlModelLoader.optionalId(shapeNode, shapeId.getNamespace(), DELETE_KEY).ifPresent(builder::delete);
        IdlModelLoader.optionalId(shapeNode, shapeId.getNamespace(), LIST_KEY).ifPresent(builder::list);
        IdlModelLoader.optionalIdList(shapeNode, shapeId.getNamespace(), OPERATIONS_KEY).forEach(builder::addOperation);
        IdlModelLoader.optionalIdList(shapeNode, shapeId.getNamespace(), RESOURCES_KEY).forEach(builder::addResource);
        IdlModelLoader.optionalIdList(shapeNode, shapeId.getNamespace(), COLLECTION_OPERATIONS_KEY).forEach(builder::addCollectionOperation);
        shapeNode.getObjectMember(IDENTIFIERS_KEY).ifPresent(ids -> {
            for (Map.Entry<StringNode, Node> entry : ids.getMembers().entrySet()) {
                String name = entry.getKey().getValue();
                StringNode target = entry.getValue().expectStringNode();
                this.onShapeTarget(target.getValue(), target, id -> builder.addIdentifier(name, (ToShapeId)id));
            }
        });
        this.expectNewline();
    }

    private void parseOperation() {
        SourceLocation sourceLocation = this.currentLocation();
        ShapeId id = this.parseShapeName();
        OperationShape.Builder builder = (OperationShape.Builder)((OperationShape.Builder)OperationShape.builder().id(id)).source(sourceLocation);
        this.visitor.onShape(builder);
        SmithyModelLexer.Token opening = this.expect(SmithyModelLexer.TokenType.LBRACE);
        ObjectNode node = this.parseObjectNode(opening.getSourceLocation(), SmithyModelLexer.TokenType.RBRACE);
        this.visitor.checkForAdditionalProperties(node, id, OPERATION_PROPERTY_NAMES);
        node.getStringMember("input").ifPresent(input -> this.onShapeTarget(input.getValue(), (FromSourceLocation)input, builder::input));
        node.getStringMember("output").ifPresent(output -> this.onShapeTarget(output.getValue(), (FromSourceLocation)output, builder::output));
        node.getArrayMember("errors").ifPresent(errors -> {
            for (StringNode value : errors.getElementsAs(StringNode.class)) {
                this.onShapeTarget(value.getValue(), value, builder::addError);
            }
        });
        this.expectNewline();
    }

    private static final class TraitEntry {
        final String traitName;
        final Node value;
        final boolean isAnnotation;

        TraitEntry(String traitName, Node value, boolean isAnnotation) {
            this.traitName = traitName;
            this.value = value;
            this.isAnnotation = isAnnotation;
        }
    }

    private static final class DocComment {
        final SourceLocation sourceLocation;
        final String content;

        DocComment(String content, SourceLocation sourceLocation) {
            this.content = content;
            this.sourceLocation = sourceLocation;
        }
    }

    private static enum TraitValueType {
        SHAPE,
        MEMBER,
        APPLY;

    }
}

