/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.LoaderUtils;
import software.amazon.smithy.model.loader.LoaderVisitor;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.BigDecimalShape;
import software.amazon.smithy.model.shapes.BigIntegerShape;
import software.amazon.smithy.model.shapes.BlobShape;
import software.amazon.smithy.model.shapes.BooleanShape;
import software.amazon.smithy.model.shapes.ByteShape;
import software.amazon.smithy.model.shapes.CollectionShape;
import software.amazon.smithy.model.shapes.DocumentShape;
import software.amazon.smithy.model.shapes.DoubleShape;
import software.amazon.smithy.model.shapes.FloatShape;
import software.amazon.smithy.model.shapes.IntegerShape;
import software.amazon.smithy.model.shapes.ListShape;
import software.amazon.smithy.model.shapes.LongShape;
import software.amazon.smithy.model.shapes.MapShape;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.ResourceShape;
import software.amazon.smithy.model.shapes.ServiceShape;
import software.amazon.smithy.model.shapes.SetShape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeIdSyntaxException;
import software.amazon.smithy.model.shapes.ShortShape;
import software.amazon.smithy.model.shapes.StringShape;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.TimestampShape;
import software.amazon.smithy.model.shapes.UnionShape;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.SetUtils;

enum DeprecatedAstModelLoader {
    INSTANCE;

    private static final Logger LOGGER;
    private static final String SMITHY = "smithy";
    private static final String METADATA = "metadata";
    private static final String SHAPES = "shapes";
    private static final String TRAITS = "traits";
    private static final List<String> NAMESPACE_PROPERTIES;
    private static final List<String> COLLECTION_PROPERTY_NAMES;
    private static final List<String> MAP_PROPERTY_NAMES;
    private static final Set<String> MEMBER_PROPERTIES;
    private static final List<String> OPERATION_PROPERTY_NAMES;
    private static final List<String> SIMPLE_PROPERTY_NAMES;
    private static final List<String> STRUCTURE_PROPERTY_NAMES;
    private static final List<String> UNION_PROPERTY_NAMES;

    void load(ObjectNode model, StringNode version, LoaderVisitor visitor) {
        LOGGER.warning("Version 0.4.0 of the Smithy JSON format is deprecated and will be removed in a future version of Smithy. Please migrate to version 0.5.0 or higher (" + version.getSourceLocation() + ")");
        visitor.onOpenFile(model.getSourceLocation().getFilename());
        try {
            model.getMember(METADATA).ifPresent(value -> {
                ObjectNode metadata = value.expectObjectNode("`metadata` must be an object");
                metadata.getMembers().forEach((k, v) -> visitor.onMetadata(k.getValue(), (Node)v));
            });
        }
        catch (SourceException e) {
            visitor.onError(ValidationEvent.fromSourceException(e));
        }
        model.getMembers().forEach((key, value) -> {
            String keyValue = key.getValue();
            if (!keyValue.equals(SMITHY) && !keyValue.equals(METADATA)) {
                visitor.onNamespace(key.getValue(), (FromSourceLocation)key);
                this.loadNamespace(visitor, key.getValue(), value.expectObjectNode());
            }
        });
    }

    private void loadShape(ShapeId id, String type, ObjectNode value, LoaderVisitor visitor) {
        switch (type) {
            case "blob": {
                this.loadSimpleShape(id, value, BlobShape.builder(), visitor);
                break;
            }
            case "boolean": {
                this.loadSimpleShape(id, value, BooleanShape.builder(), visitor);
                break;
            }
            case "byte": {
                this.loadSimpleShape(id, value, ByteShape.builder(), visitor);
                break;
            }
            case "short": {
                this.loadSimpleShape(id, value, ShortShape.builder(), visitor);
                break;
            }
            case "integer": {
                this.loadSimpleShape(id, value, IntegerShape.builder(), visitor);
                break;
            }
            case "long": {
                this.loadSimpleShape(id, value, LongShape.builder(), visitor);
                break;
            }
            case "float": {
                this.loadSimpleShape(id, value, FloatShape.builder(), visitor);
                break;
            }
            case "double": {
                this.loadSimpleShape(id, value, DoubleShape.builder(), visitor);
                break;
            }
            case "document": {
                this.loadSimpleShape(id, value, DocumentShape.builder(), visitor);
                break;
            }
            case "bigDecimal": {
                this.loadSimpleShape(id, value, BigDecimalShape.builder(), visitor);
                break;
            }
            case "bigInteger": {
                this.loadSimpleShape(id, value, BigIntegerShape.builder(), visitor);
                break;
            }
            case "string": {
                this.loadSimpleShape(id, value, StringShape.builder(), visitor);
                break;
            }
            case "timestamp": {
                this.loadSimpleShape(id, value, TimestampShape.builder(), visitor);
                break;
            }
            case "list": {
                this.loadCollection(id, value, ListShape.builder(), visitor);
                break;
            }
            case "set": {
                this.loadCollection(id, value, SetShape.builder(), visitor);
                break;
            }
            case "map": {
                this.loadMap(id, value, visitor);
                break;
            }
            case "resource": {
                this.loadResource(id, value, visitor);
                break;
            }
            case "service": {
                this.loadService(id, value, visitor);
                break;
            }
            case "structure": {
                this.loadStructure(id, value, visitor);
                break;
            }
            case "union": {
                this.loadUnion(id, value, visitor);
                break;
            }
            case "operation": {
                this.loadOperation(id, value, visitor);
                break;
            }
            default: {
                throw new SourceException("Invalid shape `type`: " + type, value);
            }
        }
    }

    private void loadNamespace(LoaderVisitor visitor, String namespace, ObjectNode node) {
        ObjectNode members = node.expectObjectNode("Each namespace must be an object. Found {type}.");
        members.warnIfAdditionalProperties(NAMESPACE_PROPERTIES);
        node.getObjectMember(SHAPES).ifPresent(shapes -> this.loadShapes(visitor, namespace, (ObjectNode)shapes));
        node.getObjectMember(TRAITS).ifPresent(traits -> this.loadTraits(visitor, namespace, (ObjectNode)traits));
    }

    private void loadShapes(LoaderVisitor visitor, String namespace, ObjectNode members) {
        members.getMembers().forEach((k, v) -> {
            try {
                ShapeId shapeId = ShapeId.fromRelative(namespace, k.getValue());
                ObjectNode shape = v.expectObjectNode("Each value in the `shapes` property must be an object; found {type}.");
                this.loadShape(shapeId, shape, visitor);
            }
            catch (ShapeIdSyntaxException e) {
                visitor.onError(this.invalidShapeId(SHAPES, e.getMessage(), k.getSourceLocation()));
            }
            catch (SourceException e) {
                visitor.onError(ValidationEvent.fromSourceException(e));
            }
        });
    }

    private void loadTraits(LoaderVisitor visitor, String namespace, ObjectNode members) {
        members.getMembers().forEach((k, v) -> {
            try {
                ShapeId shapeId = ShapeId.fromRelative(namespace, k.getValue());
                ObjectNode traitValues = v.expectObjectNode("Each value in the inner object of `traits` must be an object; found {type}.");
                for (Map.Entry<StringNode, Node> traitNode : traitValues.getMembers().entrySet()) {
                    visitor.onTrait(shapeId, traitNode.getKey().getValue(), traitNode.getValue());
                }
            }
            catch (ShapeIdSyntaxException e) {
                visitor.onError(this.invalidShapeId(TRAITS, e.getMessage(), k.getSourceLocation()));
            }
        });
    }

    private ValidationEvent invalidShapeId(String descriptor, String message, SourceLocation sourceLocation) {
        return ValidationEvent.builder().eventId("Model").sourceLocation(sourceLocation).severity(Severity.ERROR).message("Each key in the `" + descriptor + "` object must be a valid relative shape ID; " + message).build();
    }

    private void loadShape(ShapeId id, ObjectNode value, LoaderVisitor visitor) throws SourceException {
        String type = value.expectMember("type", "Missing `type` in shape definition").expectStringNode("Shape `type` must be a string. Found {type}.").getValue();
        try {
            this.loadShape(id, type, value, visitor);
        }
        catch (SourceException e) {
            ValidationEvent event = ValidationEvent.fromSourceException(e).toBuilder().shapeId(id).build();
            visitor.onError(event);
        }
    }

    private void loadMember(LoaderVisitor visitor, ShapeId shapeId, Node node) {
        MemberShape.Builder builder = (MemberShape.Builder)((MemberShape.Builder)MemberShape.builder().source(node.getSourceLocation())).id(shapeId);
        ObjectNode targetNode = node.expectObjectNode("Expected member to be an object; found {type}");
        StringNode targetStringNode = targetNode.expectMember("target", "Missing required member property `target`.").expectStringNode("Expected `target` property of member to be a string; found {type}.");
        visitor.onShapeTarget(targetStringNode.getValue(), targetStringNode, builder::target);
        this.extractTraits(shapeId, targetNode, MEMBER_PROPERTIES, visitor);
        visitor.onShape(builder);
    }

    private void extractTraits(ShapeId shapeId, ObjectNode shapeNode, Collection<String> propertyNames, LoaderVisitor visitor) {
        for (Map.Entry<StringNode, Node> entry : shapeNode.getMembers().entrySet()) {
            String traitName = entry.getKey().getValue();
            if (propertyNames.contains(traitName)) continue;
            visitor.onTrait(shapeId, traitName, entry.getValue());
        }
    }

    private void loadCollection(ShapeId shapeId, ObjectNode shapeNode, CollectionShape.Builder builder, LoaderVisitor visitor) {
        this.extractTraits(shapeId, shapeNode, COLLECTION_PROPERTY_NAMES, visitor);
        this.loadMember(visitor, shapeId.withMember("member"), shapeNode.expectMember("member", "Shape is missing required property `member`."));
        visitor.onShape((AbstractShapeBuilder)((AbstractShapeBuilder)builder.id(shapeId)).source(shapeNode.getSourceLocation()));
    }

    private void loadMap(ShapeId shapeId, ObjectNode shapeNode, LoaderVisitor visitor) {
        this.loadMember(visitor, shapeId.withMember("key"), shapeNode.expectMember("key", "Missing required map property `key`."));
        this.loadMember(visitor, shapeId.withMember("value"), shapeNode.expectMember("value", "Missing required map property `value`."));
        this.extractTraits(shapeId, shapeNode, MAP_PROPERTY_NAMES, visitor);
        visitor.onShape((AbstractShapeBuilder)((MapShape.Builder)MapShape.builder().id(shapeId)).source(shapeNode.getSourceLocation()));
    }

    private void loadOperation(ShapeId operationShapeId, ObjectNode node, LoaderVisitor visitor) {
        String namespace = operationShapeId.getNamespace();
        this.extractTraits(operationShapeId, node, OPERATION_PROPERTY_NAMES, visitor);
        OperationShape.Builder builder = ((OperationShape.Builder)((OperationShape.Builder)OperationShape.builder().id(operationShapeId)).source(node.getSourceLocation())).addErrors(LoaderUtils.optionalIdList(node, namespace, "errors"));
        LoaderUtils.optionalId(node, namespace, "input").ifPresent(builder::input);
        LoaderUtils.optionalId(node, namespace, "output").ifPresent(builder::output);
        visitor.onShape(builder);
    }

    private void loadResource(ShapeId shapeId, ObjectNode shapeNode, LoaderVisitor visitor) {
        this.extractTraits(shapeId, shapeNode, LoaderUtils.RESOURCE_PROPERTY_NAMES, visitor);
        ResourceShape.Builder builder = (ResourceShape.Builder)((ResourceShape.Builder)ResourceShape.builder().id(shapeId)).source(shapeNode.getSourceLocation());
        LoaderUtils.loadResourceObject(builder, shapeId, shapeNode, visitor);
        visitor.onShape(builder);
    }

    private void loadService(ShapeId shapeId, ObjectNode shapeNode, LoaderVisitor visitor) {
        this.extractTraits(shapeId, shapeNode, LoaderUtils.SERVICE_PROPERTY_NAMES, visitor);
        ServiceShape.Builder builder = (ServiceShape.Builder)((ServiceShape.Builder)new ServiceShape.Builder().id(shapeId)).source(shapeNode.getSourceLocation());
        LoaderUtils.loadServiceObject(builder, shapeId, shapeNode);
        visitor.onShape(builder);
    }

    private void loadSimpleShape(ShapeId shapeId, ObjectNode shapeNode, AbstractShapeBuilder builder, LoaderVisitor visitor) {
        this.extractTraits(shapeId, shapeNode, SIMPLE_PROPERTY_NAMES, visitor);
        visitor.onShape((AbstractShapeBuilder)((AbstractShapeBuilder)builder.id(shapeId)).source(shapeNode.getSourceLocation()));
    }

    private void loadStructure(ShapeId shapeId, ObjectNode shapeNode, LoaderVisitor visitor) {
        ObjectNode memberObject = shapeNode.getMember("members").orElse(Node.objectNode()).expectObjectNode("Expected structure `members` to be an object; found {type}.");
        this.extractTraits(shapeId, shapeNode, STRUCTURE_PROPERTY_NAMES, visitor);
        visitor.onShape((AbstractShapeBuilder)((StructureShape.Builder)StructureShape.builder().id(shapeId)).source(shapeNode.getSourceLocation()));
        memberObject.getMembers().forEach((k, v) -> this.loadMember(visitor, shapeId.withMember(k.getValue()), (Node)v));
    }

    private void loadUnion(ShapeId shapeId, ObjectNode shapeNode, LoaderVisitor visitor) {
        ObjectNode membersNode = shapeNode.expectMember("members", "Missing required union property `members`.").expectObjectNode("Expected union `members` to be an `object`, got `{type}`.");
        this.extractTraits(shapeId, shapeNode, UNION_PROPERTY_NAMES, visitor);
        visitor.onShape((AbstractShapeBuilder)((UnionShape.Builder)UnionShape.builder().id(shapeId)).source(shapeNode.getSourceLocation()));
        membersNode.getMembers().forEach((k, v) -> this.loadMember(visitor, shapeId.withMember(k.getValue()), (Node)v));
    }

    static {
        LOGGER = Logger.getLogger(DeprecatedAstModelLoader.class.getName());
        NAMESPACE_PROPERTIES = ListUtils.of((Object[])new String[]{SHAPES, TRAITS});
        COLLECTION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "member"});
        MAP_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "key", "value"});
        MEMBER_PROPERTIES = SetUtils.of((Object)"target");
        OPERATION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "input", "output", "errors"});
        SIMPLE_PROPERTY_NAMES = ListUtils.of((Object)"type");
        STRUCTURE_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "members"});
        UNION_PROPERTY_NAMES = ListUtils.of((Object[])new String[]{"type", "members"});
    }
}

