/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.logging.Logger;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.loader.ValidatorDefinition;
import software.amazon.smithy.model.node.ArrayNode;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.selector.Selector;
import software.amazon.smithy.model.selector.SelectorSyntaxException;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.Suppression;
import software.amazon.smithy.model.validation.ValidatedResult;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.ListUtils;

final class ValidationLoader {
    private static final Logger LOGGER = Logger.getLogger(ValidationLoader.class.getName());
    private static final List<String> SEVERITIES = ListUtils.of((Object[])new String[]{"DANGER", "WARNING", "NOTE"});
    private static final List<String> SUPPRESSION_PROPERTIES = ListUtils.of((Object[])new String[]{"ids", "shapes", "reason"});
    private static final List<String> VALIDATOR_PROPERTIES = ListUtils.of((Object[])new String[]{"name", "id", "message", "severity", "namespaces", "selector", "configuration"});

    private ValidationLoader() {
    }

    static ValidatedResult<List<ValidatorDefinition>> loadValidators(Map<String, Node> metadata) {
        return ValidationLoader.loadMultiple(metadata, "validators", "smithy.validators", ValidationLoader::loadSingleValidator);
    }

    static ValidatedResult<List<Suppression>> loadSuppressions(Map<String, Node> metadata) {
        return ValidationLoader.loadMultiple(metadata, "suppressions", "smithy.suppressions", ValidationLoader::loadSingleSuppression);
    }

    private static <T> ValidatedResult<List<T>> loadMultiple(Map<String, Node> metadata, String newKey, String oldKey, Function<ObjectNode, T> f) {
        if (!metadata.containsKey(oldKey)) {
            return ValidationLoader.load(metadata, newKey, f);
        }
        LOGGER.warning(String.format("`%s` is deprecated. Use `%s` instead", oldKey, newKey));
        if (!metadata.containsKey(newKey)) {
            return ValidationLoader.load(metadata, oldKey, f);
        }
        ValidatedResult<List<T>> result1 = ValidationLoader.load(metadata, newKey, f);
        ValidatedResult<List<T>> result2 = ValidationLoader.load(metadata, oldKey, f);
        ArrayList merged = new ArrayList(result1.getResult().orElse(ListUtils.of()));
        merged.addAll(result2.getResult().orElse(ListUtils.of()));
        ArrayList<ValidationEvent> events = new ArrayList<ValidationEvent>(result1.getValidationEvents());
        events.addAll(result2.getValidationEvents());
        return new ValidatedResult<List<T>>(merged, events);
    }

    private static <T> ValidatedResult<List<T>> load(Map<String, Node> metadata, String key, Function<ObjectNode, T> f) {
        if (!metadata.containsKey(key)) {
            return ValidatedResult.empty();
        }
        ArrayList<ValidationEvent> events = new ArrayList<ValidationEvent>();
        ArrayList<T> result = new ArrayList<T>();
        Node node = metadata.get(key);
        try {
            ArrayNode values = node.expectArrayNode(key + " must be an array. Found {type}.");
            for (Node element : values.getElements()) {
                try {
                    ObjectNode definition = element.expectObjectNode("Each element of `" + key + "` must be an object. Found {type}.");
                    result.add(f.apply(definition));
                }
                catch (SourceException e) {
                    events.add(ValidationEvent.fromSourceException(e));
                }
            }
        }
        catch (SourceException e) {
            events.add(ValidationEvent.fromSourceException(e));
        }
        return new ValidatedResult<List<T>>(result, events);
    }

    private static ValidatorDefinition loadSingleValidator(ObjectNode node) {
        node.warnIfAdditionalProperties(VALIDATOR_PROPERTIES);
        String name = node.expectMember("name", "Validator is missing a required `name` property.").expectStringNode().getValue();
        ValidatorDefinition def = new ValidatorDefinition(name, node.getStringMember("id").map(StringNode::getValue).orElse(name));
        def.message = node.getStringMember("message").map(StringNode::getValue).orElse(null);
        def.severity = node.getStringMember("severity").map(value -> value.expectOneOf(SEVERITIES)).map(value -> Severity.fromString(value).get()).orElse(null);
        node.getMember("namespaces").ifPresent(value -> def.namespaces.addAll(Node.loadArrayOfString("namespaces", value)));
        def.configuration = node.getObjectMember("configuration").orElse(Node.objectNode());
        node.getStringMember("selector").ifPresent(selector -> {
            try {
                def.selector = Selector.parse(selector.getValue());
            }
            catch (SelectorSyntaxException e) {
                throw new SourceException(String.format("Invalid validator selector `%s`: %s", selector.getValue(), e.getMessage()), (FromSourceLocation)selector, e);
            }
        });
        return def;
    }

    private static Suppression loadSingleSuppression(ObjectNode node) {
        node.warnIfAdditionalProperties(SUPPRESSION_PROPERTIES);
        Suppression.Builder builder = Suppression.builder().sourceLocation(node.getSourceLocation());
        Node.loadArrayOfString("ids", node.expectMember("ids", "Missing required validator `ids` property.")).forEach(builder::addValidatorId);
        node.getMember("shapes").map(value -> Node.loadArrayOfString("shapes", value)).ifPresent(values -> values.forEach(builder::addShape));
        node.getStringMember("reason").map(StringNode::getValue).ifPresent(builder::reason);
        return builder.build();
    }
}

