/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.diff.testrunner;

import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ToShapeId;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.IoUtils;

public final class SmithyDiffTestCase {
    private static final Pattern EVENT_PATTERN = Pattern.compile("^\\[(?<severity>SUPPRESSED|NOTE|WARNING|DANGER|ERROR)] (?<shape>[^ ]+): ?(?<message>.*) \\| (?<id>[^)]+)", 32);
    private final Path path;
    private final String name;
    private final List<ValidationEvent> expectedEvents;

    public SmithyDiffTestCase(Path path, String name, List<ValidationEvent> expectedEvents) {
        this.path = Objects.requireNonNull(path);
        this.name = Objects.requireNonNull(name);
        this.expectedEvents = Collections.unmodifiableList(expectedEvents);
    }

    public static SmithyDiffTestCase from(Path path, String name) {
        List<ValidationEvent> expectedEvents = SmithyDiffTestCase.loadExpectedEvents(path, name);
        return new SmithyDiffTestCase(path, name, expectedEvents);
    }

    public Path getPath() {
        return this.path;
    }

    public String getName() {
        return this.name;
    }

    public List<ValidationEvent> getExpectedEvents() {
        return this.expectedEvents;
    }

    public Result createResult(List<ValidationEvent> actualEvents) {
        List<ValidationEvent> unmatchedEvents = this.expectedEvents.stream().filter(expectedEvent -> actualEvents.stream().noneMatch(actualEvent -> SmithyDiffTestCase.compareEvents(expectedEvent, actualEvent))).collect(Collectors.toList());
        List<ValidationEvent> extraEvents = actualEvents.stream().filter(actualEvent -> this.expectedEvents.stream().noneMatch(expectedEvent -> SmithyDiffTestCase.compareEvents(expectedEvent, actualEvent))).filter(event -> event.getSeverity() != Severity.SUPPRESSED).collect(Collectors.toList());
        return new Result(this.name, unmatchedEvents, extraEvents);
    }

    private static boolean compareEvents(ValidationEvent expected, ValidationEvent actual) {
        String normalizedActualMessage = SmithyDiffTestCase.normalizeMessage(actual.getMessage());
        if (actual.getSuppressionReason().isPresent()) {
            normalizedActualMessage = normalizedActualMessage + " (" + (String)actual.getSuppressionReason().get() + ")";
        }
        normalizedActualMessage = SmithyDiffTestCase.normalizeMessage(normalizedActualMessage);
        String comparedMessage = SmithyDiffTestCase.normalizeMessage(expected.getMessage());
        return expected.getSeverity() == actual.getSeverity() && actual.containsId(expected.getId()) && expected.getShapeId().equals(actual.getShapeId()) && normalizedActualMessage.startsWith(comparedMessage);
    }

    private static String normalizeMessage(String message) {
        return message.replace("\n", "\\n").replace("\r", "\\n");
    }

    private static List<ValidationEvent> loadExpectedEvents(Path path, String name) {
        String fileName = path.resolve(name + ".events").toString();
        String contents = IoUtils.readUtf8File((String)fileName);
        return Arrays.stream(contents.split("-----")).map(chunk -> chunk.trim()).filter(chunk -> !chunk.isEmpty()).map(chunk -> SmithyDiffTestCase.parseValidationEvent(chunk, fileName)).collect(Collectors.toList());
    }

    static ValidationEvent parseValidationEvent(String event, String fileName) {
        Matcher matcher = EVENT_PATTERN.matcher(event);
        if (!matcher.find()) {
            throw new IllegalArgumentException(String.format("Invalid validation event in file `%s`, the following event did not match the expected regular expression `%s`: %s", fileName, EVENT_PATTERN.pattern(), event));
        }
        SourceLocation location = new SourceLocation("/", 0, 0);
        ValidationEvent.Builder builder = ValidationEvent.builder().severity((Severity)Severity.fromString((String)matcher.group("severity")).get()).sourceLocation((FromSourceLocation)location).id(matcher.group("id")).message(matcher.group("message"));
        if (!matcher.group("shape").equals("-")) {
            builder.shapeId((ToShapeId)ShapeId.from((String)matcher.group("shape")));
        }
        return builder.build();
    }

    public static final class Result {
        private final String name;
        private final Collection<ValidationEvent> unmatchedEvents;
        private final Collection<ValidationEvent> extraEvents;

        Result(String name, Collection<ValidationEvent> unmatchedEvents, Collection<ValidationEvent> extraEvents) {
            this.name = name;
            this.unmatchedEvents = Collections.unmodifiableCollection(new TreeSet<ValidationEvent>(unmatchedEvents));
            this.extraEvents = Collections.unmodifiableCollection(new TreeSet<ValidationEvent>(extraEvents));
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("============================\nModel Diff Validation Result\n============================\n").append(this.name).append('\n');
            if (!this.unmatchedEvents.isEmpty()) {
                builder.append("\nDid not match the following events\n----------------------------------\n");
                for (ValidationEvent event : this.unmatchedEvents) {
                    builder.append(event.toString()).append("\n\n");
                }
            }
            if (!this.extraEvents.isEmpty()) {
                builder.append("\nEncountered unexpected events\n-----------------------------\n");
                for (ValidationEvent event : this.extraEvents) {
                    builder.append(event.toString()).append("\n\n");
                }
            }
            return builder.toString();
        }

        public String getName() {
            return this.name;
        }

        public Collection<ValidationEvent> getUnmatchedEvents() {
            return this.unmatchedEvents;
        }

        public Collection<ValidationEvent> getExtraEvents() {
            return this.extraEvents;
        }

        public boolean isInvalid() {
            return !this.unmatchedEvents.isEmpty() || !this.extraEvents.isEmpty();
        }

        public Result unwrap() {
            if (this.isInvalid()) {
                throw new Error(this);
            }
            return this;
        }
    }

    public static final class Error
    extends RuntimeException {
        public final Result result;

        Error(Result result) {
            super(result.toString());
            this.result = result;
        }
    }
}

