/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.cli.commands;

import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.function.Consumer;
import software.amazon.smithy.cli.ArgumentReceiver;
import software.amazon.smithy.cli.Arguments;
import software.amazon.smithy.cli.CliError;
import software.amazon.smithy.cli.ColorBuffer;
import software.amazon.smithy.cli.ColorTheme;
import software.amazon.smithy.cli.Command;
import software.amazon.smithy.cli.HelpPrinter;
import software.amazon.smithy.cli.StandardOptions;
import software.amazon.smithy.cli.Style;
import software.amazon.smithy.cli.commands.CliCache;
import software.amazon.smithy.cli.commands.HelpActionWrapper;
import software.amazon.smithy.cli.commands.ProgressStyle;
import software.amazon.smithy.cli.commands.ProgressTracker;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.utils.IoUtils;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.StringUtils;

final class InitCommand
implements Command {
    private static final int LINE_LENGTH = 100;
    private static final String COLUMN_SEPARATOR = "  ";
    private static final String SMITHY_TEMPLATE_JSON = "smithy-templates.json";
    private static final String DEFAULT_REPOSITORY_URL = "https://github.com/smithy-lang/smithy-examples.git";
    private static final String DEFAULT_TEMPLATE_NAME = "quickstart-cli";
    private static final String DOCUMENTATION = "documentation";
    private static final String NAME = "name";
    private static final String TEMPLATES = "templates";
    private static final String PATH = "path";
    private static final String INCLUDED = "include";
    private final String parentCommandName;

    InitCommand(String parentCommandName) {
        this.parentCommandName = parentCommandName;
    }

    @Override
    public String getName() {
        return "init";
    }

    @Override
    public String getSummary() {
        return "Initialize a smithy project using a template";
    }

    @Override
    public int execute(Arguments arguments, Command.Env env) {
        arguments.addReceiver(new Options());
        HelpActionWrapper action = HelpActionWrapper.fromCommand(this, this.parentCommandName, c -> {
            ColorBuffer buffer = ColorBuffer.of(c, new StringBuilder());
            buffer.println("Examples:", new Style[0]);
            buffer.println("   smithy init --list", ColorTheme.LITERAL);
            buffer.println("   smithy init -o /tmp/quickstart-gradle -t quickstart-gradle", ColorTheme.LITERAL);
            return buffer.toString();
        }, this::run);
        return action.apply(arguments, env);
    }

    private int run(Arguments arguments, Command.Env env) {
        Options options = arguments.getReceiver(Options.class);
        StandardOptions standardOptions = arguments.getReceiver(StandardOptions.class);
        boolean isLocalRepo = this.isLocalRepo(options.repositoryUrl);
        Path templateRepoDirPath = this.getTemplateRepoDirPath(options.repositoryUrl, isLocalRepo);
        if (!Files.exists(templateRepoDirPath, new LinkOption[0])) {
            try (ProgressTracker t = new ProgressTracker(env, ProgressStyle.dots("cloning template repo", "template repo cloned"), standardOptions.quiet());){
                Path templateCachePath = CliCache.getTemplateCache().get();
                String relativeTemplateDir = templateCachePath.relativize(templateRepoDirPath).toString();
                InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "clone", "--depth", "1", "--single-branch", options.repositoryUrl, relativeTemplateDir}), templateCachePath);
            }
        }
        this.validateTemplateDir(templateRepoDirPath, options.repositoryUrl);
        if (!isLocalRepo) {
            InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "fetch", "--depth", "1"}), templateRepoDirPath);
            String response = InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "rev-list", "origin..HEAD"}), templateRepoDirPath);
            if (!StringUtils.isEmpty((CharSequence)response)) {
                try (ProgressTracker t = new ProgressTracker(env, ProgressStyle.dots("updating template cache", "template repo updated"), standardOptions.quiet());){
                    InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "reset", "--hard", "origin/main"}), templateRepoDirPath);
                    InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "clean", "-dfx"}), templateRepoDirPath);
                }
            }
        }
        ObjectNode smithyTemplatesNode = InitCommand.readJsonFileAsNode(templateRepoDirPath.resolve(SMITHY_TEMPLATE_JSON)).expectObjectNode();
        if (options.listTemplates.booleanValue()) {
            this.listTemplates(smithyTemplatesNode, env);
        } else {
            this.cloneTemplate(templateRepoDirPath, smithyTemplatesNode, options, standardOptions, env);
        }
        return 0;
    }

    private void validateTemplateDir(Path templateRepoDirPath, String templateUrl) {
        if (!Files.isDirectory(templateRepoDirPath, new LinkOption[0])) {
            throw new CliError("Template repository " + templateRepoDirPath + " is not a directory");
        }
        Path templateJsonPath = templateRepoDirPath.resolve(SMITHY_TEMPLATE_JSON);
        if (!Files.exists(templateJsonPath, new LinkOption[0]) && Files.isRegularFile(templateJsonPath, new LinkOption[0])) {
            throw new CliError("Template repository " + templateUrl + " does not contain a valid `smithy-templates.json`.");
        }
    }

    private Path getTemplateRepoDirPath(String repoPath, boolean isLocalRepo) {
        if (isLocalRepo) {
            return Paths.get(repoPath, new String[0]);
        }
        return CliCache.getTemplateCache().get().resolve(this.getCacheDirFromUrl(repoPath));
    }

    private String getCacheDirFromUrl(String repositoryUrl) {
        return repositoryUrl.replace(".git", "").replace(":", "_").replace("/", "_").replace(".", "_");
    }

    private void listTemplates(ObjectNode smithyTemplatesNode, Command.Env env) {
        try (ColorBuffer buffer = ColorBuffer.of(env.colors(), env.stdout());){
            buffer.println(this.getTemplateList(smithyTemplatesNode, env), new Style[0]);
        }
    }

    private boolean isLocalRepo(String repoPath) {
        try {
            Path localPath = Paths.get(repoPath, new String[0]);
            return Files.exists(localPath, new LinkOption[0]);
        }
        catch (InvalidPathException exc) {
            return false;
        }
    }

    private String getTemplateList(ObjectNode smithyTemplatesNode, Command.Env env) {
        int maxTemplateLength = 0;
        TreeMap<String, String> templates = new TreeMap<String, String>();
        for (Map.Entry entry : InitCommand.getTemplatesNode(smithyTemplatesNode).getMembers().entrySet()) {
            String template = ((StringNode)entry.getKey()).getValue();
            String documentation = ((Node)entry.getValue()).expectObjectNode().expectMember(DOCUMENTATION, String.format("Missing expected member `%s` from `%s` object", DOCUMENTATION, template)).expectStringNode().getValue();
            templates.put(template, documentation);
            maxTemplateLength = Math.max(maxTemplateLength, template.length());
        }
        int maxDocLength = 100 - maxTemplateLength - COLUMN_SEPARATOR.length();
        ColorBuffer builder = ColorBuffer.of(env.colors(), new StringBuilder());
        InitCommand.writeTemplateBorder(builder, maxTemplateLength, maxDocLength);
        builder.print(InitCommand.pad(NAME.toUpperCase(Locale.US), maxTemplateLength), ColorTheme.NOTE).print(COLUMN_SEPARATOR, new Style[0]).print(DOCUMENTATION.toUpperCase(Locale.US), ColorTheme.NOTE).println();
        InitCommand.writeTemplateBorder(builder, maxTemplateLength, maxDocLength);
        int offset = maxTemplateLength + COLUMN_SEPARATOR.length();
        for (Map.Entry entry : templates.entrySet()) {
            String template = (String)entry.getKey();
            String doc = (String)entry.getValue();
            builder.print(InitCommand.pad(template, maxTemplateLength), ColorTheme.TEMPLATE_TITLE).print(COLUMN_SEPARATOR, new Style[0]).print(InitCommand.wrapDocumentation(doc, maxDocLength, offset), ColorTheme.MUTED).println();
        }
        return builder.toString();
    }

    private static void writeTemplateBorder(ColorBuffer writer, int maxNameLength, int maxDocLength) {
        writer.print(InitCommand.pad("", maxNameLength).replace(" ", "\u2500"), ColorTheme.TEMPLATE_LIST_BORDER).print(COLUMN_SEPARATOR, new Style[0]).print(InitCommand.pad("", maxDocLength).replace(" ", "\u2500"), ColorTheme.TEMPLATE_LIST_BORDER).println();
    }

    private static String wrapDocumentation(String doc, int maxLength, int offset) {
        return StringUtils.wrap((String)doc, (int)maxLength, (String)(System.lineSeparator() + InitCommand.pad("", offset)), (boolean)false);
    }

    private void cloneTemplate(Path templateRepoDirPath, ObjectNode smithyTemplatesNode, Options options, StandardOptions standardOptions, Command.Env env) {
        Path dest;
        String template = options.template;
        String directory = options.directory;
        if (template == null || template.isEmpty()) {
            throw new IllegalArgumentException("Please specify a template name using `--template` or `-t`");
        }
        if (directory == null) {
            directory = template;
        }
        if (Files.exists(dest = Paths.get(directory, new String[0]), new LinkOption[0])) {
            throw new CliError("Output directory `" + directory + "` already exists.");
        }
        ObjectNode templatesNode = InitCommand.getTemplatesNode(smithyTemplatesNode);
        if (!templatesNode.containsMember(template)) {
            throw new IllegalArgumentException(String.format("Invalid template `%s`. `%s` provides the following templates:%n%n%s", template, InitCommand.getTemplatesName(smithyTemplatesNode), this.getTemplateList(smithyTemplatesNode, env)));
        }
        ObjectNode templateNode = templatesNode.expectObjectMember(template).expectObjectNode();
        String templatePath = InitCommand.getTemplatePath(templateNode, template);
        List<String> includedFiles = InitCommand.getIncludedFiles(templateNode);
        Path stagingPath = InitCommand.createStagingRepo(templateRepoDirPath);
        InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "sparse-checkout", "set", "--no-cone", templatePath}), stagingPath);
        for (String includedFile : includedFiles) {
            InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "sparse-checkout", "add", "--no-cone", includedFile}), stagingPath);
        }
        InitCommand.exec(ListUtils.of((Object)"git", (Object)"checkout"), stagingPath);
        if (!Files.exists(stagingPath.resolve(templatePath), new LinkOption[0])) {
            throw new CliError(String.format("Template path `%s` for template \"%s\" is invalid.", templatePath, template));
        }
        IoUtils.copyDir((Path)Paths.get(stagingPath.toString(), templatePath), (Path)dest);
        InitCommand.copyIncludedFiles(stagingPath.toString(), dest.toString(), includedFiles, template, env);
        if (!standardOptions.quiet()) {
            try (ColorBuffer buffer = ColorBuffer.of(env.colors(), env.stdout());){
                buffer.println(String.format("Smithy project created in directory: %s", directory), ColorTheme.SUCCESS);
            }
        }
    }

    private static Path createStagingRepo(Path repoPath) {
        Path temp;
        try {
            temp = Files.createTempDirectory("temp", new FileAttribute[0]);
        }
        catch (IOException exc) {
            throw new CliError("Unable to create staging directory for template.");
        }
        InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "clone", "--no-checkout", "--depth", "1", "--sparse", "file://" + repoPath.toString(), temp.toString()}), Paths.get(".", new String[0]));
        return temp;
    }

    private static ObjectNode getSmithyTemplatesNode(Path jsonFilePath) {
        return InitCommand.readJsonFileAsNode(Paths.get(jsonFilePath.toString(), SMITHY_TEMPLATE_JSON)).expectObjectNode();
    }

    private static ObjectNode getTemplatesNode(ObjectNode smithyTemplatesNode) {
        return smithyTemplatesNode.expectMember(TEMPLATES, String.format("Missing expected member `%s` from %s", TEMPLATES, SMITHY_TEMPLATE_JSON)).expectObjectNode();
    }

    private static String getTemplatesName(ObjectNode smithyTemplatesNode) {
        return smithyTemplatesNode.expectMember(NAME, String.format("Missing expected member `%s` from %s", NAME, SMITHY_TEMPLATE_JSON)).expectStringNode().getValue();
    }

    private static String getTemplatePath(ObjectNode templateNode, String templateName) {
        return templateNode.expectMember(PATH, String.format("Missing expected member `%s` from `%s` object", PATH, templateName)).expectStringNode().getValue();
    }

    private static List<String> getIncludedFiles(ObjectNode templateNode) {
        ArrayList<String> includedPaths = new ArrayList<String>();
        templateNode.getArrayMember(INCLUDED, StringNode::getValue, includedPaths::addAll);
        return includedPaths;
    }

    private static void copyIncludedFiles(String temp, String dest, List<String> includedFiles, String templateName, Command.Env env) {
        for (String included : includedFiles) {
            Path includedPath = Paths.get(temp, included);
            if (!Files.exists(includedPath, new LinkOption[0])) {
                throw new CliError(String.format("File or directory `%s` is marked for inclusion in template \"%s\", but was not found", included, templateName));
            }
            Path target = Paths.get(dest, Objects.requireNonNull(includedPath.getFileName()).toString());
            if (Files.isDirectory(includedPath, new LinkOption[0])) {
                IoUtils.copyDir((Path)includedPath, (Path)target);
                continue;
            }
            if (!Files.isRegularFile(includedPath, new LinkOption[0])) continue;
            try {
                Files.copy(includedPath, target, new CopyOption[0]);
            }
            catch (IOException e) {
                throw new CliError("Unable to copy included file: " + includedPath + "to destination directory: " + target);
            }
        }
    }

    private static String exec(List<String> args, Path directory) {
        StringBuilder output = new StringBuilder();
        int code = IoUtils.runCommand(args, (Path)directory, (Appendable)output, Collections.emptyMap());
        if (code != 0) {
            String errorPrefix = "Unable to run `" + String.join((CharSequence)" ", args) + "`";
            throw new CliError(errorPrefix + ": " + output);
        }
        return output.toString();
    }

    private static Node readJsonFileAsNode(Path jsonFilePath) {
        return Node.parse((String)IoUtils.readUtf8File((Path)jsonFilePath));
    }

    private static String pad(String s, int n) {
        return String.format("%-" + n + "s", s);
    }

    private static final class Options
    implements ArgumentReceiver {
        private String template = "quickstart-cli";
        private String directory;
        private Boolean listTemplates = false;
        private String repositoryUrl = "https://github.com/smithy-lang/smithy-examples.git";

        private Options() {
        }

        @Override
        public boolean testOption(String name) {
            switch (name) {
                case "--list": 
                case "-l": {
                    this.listTemplates = true;
                    return true;
                }
            }
            return false;
        }

        @Override
        public Consumer<String> testParameter(String name) {
            switch (name) {
                case "--template": 
                case "-t": {
                    return value -> {
                        this.template = value;
                    };
                }
                case "--url": 
                case "-u": {
                    return value -> {
                        this.repositoryUrl = value;
                    };
                }
                case "--output": 
                case "-o": {
                    return value -> {
                        this.directory = value;
                    };
                }
            }
            return null;
        }

        @Override
        public void registerHelp(HelpPrinter printer) {
            printer.param("--template", "-t", InitCommand.DEFAULT_TEMPLATE_NAME, "Specify the template to be used in the Smithy project");
            printer.param("--url", null, InitCommand.DEFAULT_REPOSITORY_URL, "Smithy templates repository url");
            printer.param("--output", "-o", "new-smithy-project", "Smithy project directory");
            printer.param("--list", "-l", null, "List available templates");
        }
    }
}

