/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.cli.commands;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.function.Consumer;
import software.amazon.smithy.cli.ArgumentReceiver;
import software.amazon.smithy.cli.Arguments;
import software.amazon.smithy.cli.CliError;
import software.amazon.smithy.cli.ColorBuffer;
import software.amazon.smithy.cli.ColorTheme;
import software.amazon.smithy.cli.Command;
import software.amazon.smithy.cli.HelpPrinter;
import software.amazon.smithy.cli.Style;
import software.amazon.smithy.cli.commands.HelpActionWrapper;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.utils.IoUtils;
import software.amazon.smithy.utils.ListUtils;

final class InitCommand
implements Command {
    private static final String SMITHY_TEMPLATE_JSON = "smithy-templates.json";
    private static final String DEFAULT_REPOSITORY_URL = "https://github.com/smithy-lang/smithy-examples.git";
    private static final String DOCUMENTATION = "documentation";
    private static final String NAME = "name";
    private static final String TEMPLATES = "templates";
    private static final String PATH = "path";
    private static final String INCLUDED = "include";
    private final String parentCommandName;

    InitCommand(String parentCommandName) {
        this.parentCommandName = parentCommandName;
    }

    @Override
    public String getName() {
        return "init";
    }

    @Override
    public String getSummary() {
        return "Initialize a smithy project using a template";
    }

    @Override
    public int execute(Arguments arguments, Command.Env env) {
        arguments.addReceiver(new Options());
        HelpActionWrapper action = HelpActionWrapper.fromCommand(this, this.parentCommandName, c -> {
            ColorBuffer buffer = ColorBuffer.of(c, new StringBuilder());
            buffer.println("Examples:", new Style[0]);
            buffer.println("   smithy init --list", ColorTheme.LITERAL);
            buffer.println("   smithy init -o /tmp/quickstart-gradle -t quickstart-gradle", ColorTheme.LITERAL);
            return buffer.toString();
        }, this::run);
        return action.apply(arguments, env);
    }

    private int run(Arguments arguments, Command.Env env) {
        Options options = arguments.getReceiver(Options.class);
        try {
            Path root = Paths.get(".", new String[0]);
            Path temp = Files.createTempDirectory("temp", new FileAttribute[0]);
            InitCommand.loadSmithyTemplateJsonFile(options.repositoryUrl, root, temp);
            ObjectNode smithyTemplatesNode = InitCommand.getSmithyTemplatesNode(temp);
            if (options.listTemplates.booleanValue()) {
                this.listTemplates(smithyTemplatesNode, env);
            } else {
                this.cloneTemplate(temp, smithyTemplatesNode, options.template, options.directory, env);
            }
        }
        catch (IOException | InterruptedException | URISyntaxException e) {
            throw new RuntimeException(e);
        }
        return 0;
    }

    private void listTemplates(ObjectNode smithyTemplatesNode, Command.Env env) throws IOException {
        try (ColorBuffer buffer = ColorBuffer.of(env.colors(), env.stderr());){
            buffer.println(this.getTemplateList(smithyTemplatesNode, env), new Style[0]);
        }
    }

    private String getTemplateList(ObjectNode smithyTemplatesNode, Command.Env env) {
        int maxTemplateLength = 0;
        int maxDocumentationLength = 0;
        TreeMap<String, String> templates = new TreeMap<String, String>();
        for (Map.Entry entry : InitCommand.getTemplatesNode(smithyTemplatesNode).getMembers().entrySet()) {
            String template = ((StringNode)entry.getKey()).getValue();
            String documentation = ((Node)entry.getValue()).expectObjectNode().expectMember(DOCUMENTATION, String.format("Missing expected member `%s` from `%s` object", DOCUMENTATION, template)).expectStringNode().getValue();
            templates.put(template, documentation);
            maxTemplateLength = Math.max(maxTemplateLength, template.length());
            maxDocumentationLength = Math.max(maxDocumentationLength, documentation.length());
        }
        String space = "   ";
        ColorBuffer builder = ColorBuffer.of(env.colors(), new StringBuilder()).print(InitCommand.pad(NAME.toUpperCase(Locale.US), maxTemplateLength), ColorTheme.LITERAL).print("   ", new Style[0]).print(DOCUMENTATION.toUpperCase(Locale.US), ColorTheme.LITERAL).println().print(InitCommand.pad("", maxTemplateLength).replace(' ', '-'), ColorTheme.MUTED).print("   ", new Style[0]).print(InitCommand.pad("", maxDocumentationLength).replace(' ', '-'), ColorTheme.MUTED).println();
        for (Map.Entry entry : templates.entrySet()) {
            String template = (String)entry.getKey();
            String doc = (String)entry.getValue();
            builder.print(InitCommand.pad(template, maxTemplateLength), new Style[0]).print("   ", new Style[0]).print(InitCommand.pad(doc, maxDocumentationLength), new Style[0]).println();
        }
        return builder.toString();
    }

    private void cloneTemplate(Path temp, ObjectNode smithyTemplatesNode, String template, String directory, Command.Env env) throws IOException, InterruptedException, URISyntaxException {
        if (template == null || template.isEmpty()) {
            throw new IllegalArgumentException("Please specify a template using `--template` or `-t`");
        }
        ObjectNode templatesNode = InitCommand.getTemplatesNode(smithyTemplatesNode);
        if (!templatesNode.containsMember(template)) {
            throw new IllegalArgumentException(String.format("Invalid template `%s`. `%s` provides the following templates:%n%n%s", template, InitCommand.getTemplatesName(smithyTemplatesNode), this.getTemplateList(smithyTemplatesNode, env)));
        }
        ObjectNode templateNode = templatesNode.expectObjectMember(template).expectObjectNode();
        String templatePath = InitCommand.getTemplatePath(templateNode, template);
        List<String> includedFiles = InitCommand.getIncludedFiles(templateNode);
        InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "sparse-checkout", "set", "--no-cone", templatePath}), temp);
        for (String includedFile : includedFiles) {
            InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "sparse-checkout", "add", "--no-cone", includedFile}), temp);
        }
        InitCommand.exec(ListUtils.of((Object)"git", (Object)"checkout"), temp);
        if (directory == null) {
            directory = template;
        }
        Path dest = Paths.get(directory, new String[0]);
        IoUtils.copyDir((Path)Paths.get(temp.toString(), templatePath), (Path)dest);
        InitCommand.copyIncludedFiles(temp.toString(), dest.toString(), includedFiles, template, env);
        try (ColorBuffer buffer = ColorBuffer.of(env.colors(), env.stderr());){
            buffer.println(String.format("Smithy project created in directory: %s", directory), ColorTheme.SUCCESS);
        }
    }

    private static void loadSmithyTemplateJsonFile(String repositoryUrl, Path root, Path temp) {
        InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "clone", "--filter=blob:none", "--no-checkout", "--depth", "1", "--sparse", repositoryUrl, temp.toString()}), root);
        InitCommand.exec(ListUtils.of((Object[])new String[]{"git", "sparse-checkout", "set", "--no-cone", SMITHY_TEMPLATE_JSON}), temp);
        InitCommand.exec(ListUtils.of((Object)"git", (Object)"checkout"), temp);
    }

    private static ObjectNode getSmithyTemplatesNode(Path jsonFilePath) {
        return InitCommand.readJsonFileAsNode(Paths.get(jsonFilePath.toString(), SMITHY_TEMPLATE_JSON)).expectObjectNode();
    }

    private static ObjectNode getTemplatesNode(ObjectNode smithyTemplatesNode) {
        return smithyTemplatesNode.expectMember(TEMPLATES, String.format("Missing expected member `%s` from %s", TEMPLATES, SMITHY_TEMPLATE_JSON)).expectObjectNode();
    }

    private static String getTemplatesName(ObjectNode smithyTemplatesNode) {
        return smithyTemplatesNode.expectMember(NAME, String.format("Missing expected member `%s` from %s", NAME, SMITHY_TEMPLATE_JSON)).expectStringNode().getValue();
    }

    private static String getTemplatePath(ObjectNode templateNode, String templateName) {
        return templateNode.expectMember(PATH, String.format("Missing expected member `%s` from `%s` object", PATH, templateName)).expectStringNode().getValue();
    }

    private static List<String> getIncludedFiles(ObjectNode templateNode) {
        ArrayList<String> includedPaths = new ArrayList<String>();
        templateNode.getArrayMember(INCLUDED, StringNode::getValue, includedPaths::addAll);
        return includedPaths;
    }

    private static void copyIncludedFiles(String temp, String dest, List<String> includedFiles, String templateName, Command.Env env) throws IOException {
        for (String included : includedFiles) {
            Path includedPath = Paths.get(temp, included);
            if (!Files.exists(includedPath, new LinkOption[0])) {
                try (ColorBuffer buffer = ColorBuffer.of(env.colors(), env.stderr());){
                    buffer.println(String.format("File or directory %s is marked for inclusion in template %s but was not found", included, templateName), ColorTheme.WARNING);
                }
            }
            Path target = Paths.get(dest, Objects.requireNonNull(includedPath.getFileName()).toString());
            if (Files.isDirectory(includedPath, new LinkOption[0])) {
                IoUtils.copyDir((Path)includedPath, (Path)target);
                continue;
            }
            if (!Files.isRegularFile(includedPath, new LinkOption[0])) continue;
            Files.copy(includedPath, target, new CopyOption[0]);
        }
    }

    private static String exec(List<String> args, Path directory) {
        StringBuilder output = new StringBuilder();
        int code = IoUtils.runCommand(args, (Path)directory, (Appendable)output, Collections.emptyMap());
        if (code != 0) {
            String errorPrefix = "Unable to run `" + String.join((CharSequence)" ", args) + "`";
            throw new CliError(errorPrefix + ": " + output);
        }
        return output.toString();
    }

    private static Node readJsonFileAsNode(Path jsonFilePath) {
        return Node.parse((String)IoUtils.readUtf8File((Path)jsonFilePath));
    }

    private static String pad(String s, int n) {
        return String.format("%-" + n + "s", s);
    }

    private static final class Options
    implements ArgumentReceiver {
        private String template;
        private String directory;
        private Boolean listTemplates = false;
        private String repositoryUrl = "https://github.com/smithy-lang/smithy-examples.git";

        private Options() {
        }

        @Override
        public boolean testOption(String name) {
            switch (name) {
                case "--list": 
                case "-l": {
                    this.listTemplates = true;
                    return true;
                }
            }
            return false;
        }

        @Override
        public Consumer<String> testParameter(String name) {
            switch (name) {
                case "--template": 
                case "-t": {
                    return value -> {
                        this.template = value;
                    };
                }
                case "--url": 
                case "-u": {
                    return value -> {
                        this.repositoryUrl = value;
                    };
                }
                case "--output": 
                case "-o": {
                    return value -> {
                        this.directory = value;
                    };
                }
            }
            return null;
        }

        @Override
        public void registerHelp(HelpPrinter printer) {
            printer.param("--template", "-t", "quickstart-cli", "Specify the template to be used in the Smithy project");
            printer.param("--url", null, InitCommand.DEFAULT_REPOSITORY_URL, "Smithy templates repository url");
            printer.param("--output", "-o", "new-smithy-project", "Smithy project directory");
            printer.param("--list", "-l", null, "List available templates");
        }
    }
}

