/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.cli;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.smithy.build.SmithyBuildException;
import software.amazon.smithy.utils.FunctionalUtils;
import software.amazon.smithy.utils.SetUtils;

public final class BuildParameterBuilder {
    private static final Logger LOGGER = Logger.getLogger(BuildParameterBuilder.class.getName());
    private static final String SMITHY_TAG_PROPERTY = "Smithy-Tags";
    private static final String SOURCE = "source";
    private static final String PATH_SEPARATOR = "path.separator";
    private String projectionSource = "source";
    private Set<String> projectionSourceTags = new LinkedHashSet<String>();
    private Set<String> buildClasspath = new LinkedHashSet<String>();
    private Set<String> libClasspath = new LinkedHashSet<String>();
    private Set<String> sources = new LinkedHashSet<String>();
    private ClassPathTagMatcher tagMatcher;
    private Set<String> configs = new LinkedHashSet<String>();
    private String output;
    private String projection;
    private String plugin;
    private boolean discover;
    private boolean allowUnknownTraits;
    private List<String> extraArgs = new ArrayList<String>();

    public BuildParameterBuilder projectionSource(String projectionSource) {
        this.projectionSource = projectionSource == null || projectionSource.isEmpty() ? SOURCE : projectionSource;
        return this;
    }

    public BuildParameterBuilder sources(Collection<String> sources) {
        if (sources != null) {
            this.sources.addAll(sources);
        }
        return this;
    }

    public BuildParameterBuilder addSourcesIfExists(Collection<String> sources) {
        if (sources != null) {
            for (String source : sources) {
                if (!source.isEmpty() && Files.exists(Paths.get(source, new String[0]), new LinkOption[0])) {
                    this.sources.add(source);
                    continue;
                }
                LOGGER.info("Skipping source that does not exist: " + source);
            }
        }
        return this;
    }

    public BuildParameterBuilder buildClasspath(String buildClasspath) {
        this.buildClasspath.addAll(BuildParameterBuilder.splitAndFilterString(System.getProperty(PATH_SEPARATOR), buildClasspath));
        return this;
    }

    private static Set<String> splitAndFilterString(String delimiter, String value) {
        if (value == null) {
            return SetUtils.of();
        }
        return Stream.of(value.split(Pattern.quote(delimiter))).map(String::trim).filter(FunctionalUtils.not(String::isEmpty)).collect(Collectors.toCollection(LinkedHashSet::new));
    }

    public BuildParameterBuilder libClasspath(String libClasspath) {
        this.libClasspath.addAll(BuildParameterBuilder.splitAndFilterString(System.getProperty(PATH_SEPARATOR), libClasspath));
        return this;
    }

    public BuildParameterBuilder projectionSourceTags(String projectionSourceTags) {
        return this.projectionSourceTags(BuildParameterBuilder.splitAndFilterString(",", projectionSourceTags));
    }

    public BuildParameterBuilder projectionSourceTags(Collection<String> projectionSourceTags) {
        if (projectionSourceTags != null) {
            this.projectionSourceTags.addAll(projectionSourceTags);
        }
        return this;
    }

    public BuildParameterBuilder tagMatcher(ClassPathTagMatcher tagMatcher) {
        this.tagMatcher = Objects.requireNonNull(tagMatcher);
        return this;
    }

    public BuildParameterBuilder addConfig(String pathToConfig) {
        if (pathToConfig != null && !pathToConfig.isEmpty()) {
            this.configs.add(pathToConfig);
        }
        return this;
    }

    public BuildParameterBuilder addConfigIfExists(String pathToConfig) {
        if (pathToConfig == null || pathToConfig.isEmpty()) {
            return this;
        }
        if (!Files.exists(Paths.get(pathToConfig, new String[0]), new LinkOption[0])) {
            LOGGER.info("Not setting --config to " + pathToConfig + " because it does not exist");
            return this;
        }
        return this.addConfig(pathToConfig);
    }

    public BuildParameterBuilder output(String output) {
        this.output = output;
        return this;
    }

    public BuildParameterBuilder projection(String projection) {
        this.projection = projection;
        return this;
    }

    public BuildParameterBuilder plugin(String plugin) {
        this.plugin = plugin;
        return this;
    }

    public BuildParameterBuilder discover(boolean discover) {
        this.discover = discover;
        return this;
    }

    public BuildParameterBuilder allowUnknownTraits(boolean allowUnknownTraits) {
        this.allowUnknownTraits = allowUnknownTraits;
        return this;
    }

    public BuildParameterBuilder addExtraArgs(String ... args) {
        Collections.addAll(this.extraArgs, Objects.requireNonNull(args));
        return this;
    }

    public Result build() {
        if (this.projectionSource.equals(SOURCE)) {
            return this.configureSourceProjection();
        }
        if (this.tagMatcher == null) {
            this.tagMatcher = new JarFileClassPathTagMatcher();
        }
        return this.configureProjection();
    }

    private Result configureSourceProjection() {
        LOGGER.info("Configuring SmithyBuild classpaths for the `source` projection");
        if (!this.projectionSourceTags.isEmpty()) {
            throw new SmithyBuildException("Projection source tags cannot be set when building a source projection.");
        }
        LinkedHashSet<String> computedDiscovery = new LinkedHashSet<String>(this.libClasspath);
        computedDiscovery.removeAll(this.sources);
        if (!this.discover) {
            computedDiscovery.clear();
        }
        LinkedHashSet<String> combined = new LinkedHashSet<String>(this.libClasspath);
        combined.addAll(this.buildClasspath);
        String discoveryClasspath = String.join((CharSequence)System.getProperty(PATH_SEPARATOR), computedDiscovery);
        String classpath = String.join((CharSequence)System.getProperty(PATH_SEPARATOR), combined);
        return new Result(this, discoveryClasspath, classpath, this.sources);
    }

    private Result configureProjection() {
        if (this.projectionSourceTags.isEmpty()) {
            LOGGER.warning("No projection source tags were set for the projection `" + this.projection + "`, so the projection will not have any sources in it other than files found in the sources of the package being built.");
            String buildCp = String.join((CharSequence)System.getProperty(PATH_SEPARATOR), this.buildClasspath);
            return new Result(this, buildCp, buildCp, this.sources);
        }
        LOGGER.fine("Configuring Smithy classpaths for projection `" + this.projection + "`");
        LinkedHashSet<String> computedSources = new LinkedHashSet<String>(this.sources);
        Set<String> tagSourceJars = this.tagMatcher.findJarsWithMatchingTags(this.buildClasspath, this.projectionSourceTags);
        computedSources.addAll(tagSourceJars);
        LOGGER.info("Found the following JARs that matched the Smithy projection tags query: " + tagSourceJars);
        LinkedHashSet<String> computedDiscovery = new LinkedHashSet<String>(this.buildClasspath);
        computedDiscovery.removeAll(computedSources);
        String discoveryClasspath = String.join((CharSequence)System.getProperty(PATH_SEPARATOR), computedDiscovery);
        String classpath = String.join((CharSequence)System.getProperty(PATH_SEPARATOR), this.buildClasspath);
        return new Result(this, discoveryClasspath, classpath, computedSources);
    }

    public static final class JarFileClassPathTagMatcher
    implements ClassPathTagMatcher {
        @Override
        public Set<String> findJarsWithMatchingTags(Set<String> classpath, Set<String> tagsToFind) {
            LinkedHashSet<String> tagSourceJars = new LinkedHashSet<String>();
            block8: for (String jar : classpath) {
                if (!Files.exists(Paths.get(jar, new String[0]), new LinkOption[0])) {
                    LOGGER.severe("Classpath entry not found: " + jar);
                    continue;
                }
                try (JarFile jarFile = new JarFile(jar);){
                    Manifest manifest = jarFile.getManifest();
                    Attributes.Name name = new Attributes.Name(BuildParameterBuilder.SMITHY_TAG_PROPERTY);
                    if (manifest == null || !manifest.getMainAttributes().containsKey(name)) continue;
                    Set<String> jarTags = this.loadTags((String)manifest.getMainAttributes().get(name));
                    LOGGER.info("Found Smithy-Tags in JAR dependency `" + jar + "`: " + jarTags);
                    for (String needle : tagsToFind) {
                        if (!jarTags.contains(needle)) continue;
                        tagSourceJars.add(jar);
                        continue block8;
                    }
                }
                catch (IOException e) {
                    throw new SmithyBuildException("Error reading manifest from JAR in build dependencies: " + e.getMessage(), (Throwable)e);
                }
            }
            return tagSourceJars;
        }

        private Set<String> loadTags(String sourceTagString) {
            return BuildParameterBuilder.splitAndFilterString(",", sourceTagString);
        }
    }

    public static interface ClassPathTagMatcher {
        public Set<String> findJarsWithMatchingTags(Set<String> var1, Set<String> var2);
    }

    public static final class Result {
        public final List<String> args;
        public final Set<String> sources;
        public final String classpath;
        public final String discoveryClasspath;

        private Result(BuildParameterBuilder builder, String discoveryClasspath, String classpath, Set<String> sources) {
            this.classpath = classpath;
            this.sources = new LinkedHashSet<String>(sources);
            this.args = new ArrayList<String>();
            this.args.add("build");
            this.args.addAll(builder.extraArgs);
            if (!builder.discover) {
                this.discoveryClasspath = "";
            } else {
                this.discoveryClasspath = discoveryClasspath;
                if (!discoveryClasspath.isEmpty()) {
                    this.args.add("--discover-classpath");
                    this.args.add(discoveryClasspath);
                } else {
                    this.args.add("--discover");
                }
            }
            if (builder.allowUnknownTraits) {
                this.args.add("--allow-unknown-traits");
            }
            builder.configs.forEach(config -> {
                this.args.add("--config");
                this.args.add((String)config);
            });
            if (builder.output != null) {
                this.args.add("--output");
                this.args.add(builder.output);
            }
            if (builder.projection != null) {
                this.args.add("--projection");
                this.args.add(builder.projection);
            }
            if (builder.plugin != null) {
                this.args.add("--plugin");
                this.args.add(builder.plugin);
            }
            this.args.addAll(sources);
        }
    }
}

