/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.kinesis.leases.dynamodb;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import lombok.NonNull;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.cloudwatch.model.StandardUnit;
import software.amazon.kinesis.annotations.KinesisClientInternalApi;
import software.amazon.kinesis.leases.Lease;
import software.amazon.kinesis.leases.LeaseRefresher;
import software.amazon.kinesis.leases.LeaseRenewer;
import software.amazon.kinesis.leases.exceptions.DependencyException;
import software.amazon.kinesis.leases.exceptions.InvalidStateException;
import software.amazon.kinesis.leases.exceptions.ProvisionedThroughputException;
import software.amazon.kinesis.metrics.MetricsFactory;
import software.amazon.kinesis.metrics.MetricsLevel;
import software.amazon.kinesis.metrics.MetricsScope;
import software.amazon.kinesis.metrics.MetricsUtil;

@KinesisClientInternalApi
public class DynamoDBLeaseRenewer
implements LeaseRenewer {
    private static final Logger log = LoggerFactory.getLogger(DynamoDBLeaseRenewer.class);
    private static final int RENEWAL_RETRIES = 2;
    private static final String RENEW_ALL_LEASES_DIMENSION = "RenewAllLeases";
    private final LeaseRefresher leaseRefresher;
    private final String workerIdentifier;
    private final long leaseDurationNanos;
    private final ExecutorService executorService;
    private final MetricsFactory metricsFactory;
    private final ConcurrentNavigableMap<String, Lease> ownedLeases = new ConcurrentSkipListMap<String, Lease>();

    public DynamoDBLeaseRenewer(LeaseRefresher leaseRefresher, String workerIdentifier, long leaseDurationMillis, ExecutorService executorService, MetricsFactory metricsFactory) {
        this.leaseRefresher = leaseRefresher;
        this.workerIdentifier = workerIdentifier;
        this.leaseDurationNanos = TimeUnit.MILLISECONDS.toNanos(leaseDurationMillis);
        this.executorService = executorService;
        this.metricsFactory = metricsFactory;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void renewLeases() throws DependencyException, InvalidStateException {
        log.debug("Worker {} holding {} leases: {}", new Object[]{this.workerIdentifier, this.ownedLeases.size(), this.ownedLeases});
        MetricsScope scope = MetricsUtil.createMetricsWithOperation(this.metricsFactory, RENEW_ALL_LEASES_DIMENSION);
        long startTime = System.currentTimeMillis();
        boolean success = false;
        try {
            int lostLeases = 0;
            ArrayList<Future<Boolean>> renewLeaseTasks = new ArrayList<Future<Boolean>>();
            for (Lease lease : this.ownedLeases.descendingMap().values()) {
                renewLeaseTasks.add(this.executorService.submit(new RenewLeaseTask(lease)));
            }
            int leasesInUnknownState = 0;
            ExecutionException lastException = null;
            for (Future future : renewLeaseTasks) {
                try {
                    if (((Boolean)future.get()).booleanValue()) continue;
                    ++lostLeases;
                }
                catch (InterruptedException e) {
                    log.info("Interrupted while waiting for a lease to renew.");
                    ++leasesInUnknownState;
                    Thread.currentThread().interrupt();
                }
                catch (ExecutionException e) {
                    log.error("Encountered an exception while renewing a lease.", e.getCause());
                    ++leasesInUnknownState;
                    lastException = e;
                }
            }
            scope.addData("LostLeases", lostLeases, StandardUnit.COUNT, MetricsLevel.SUMMARY);
            scope.addData("CurrentLeases", this.ownedLeases.size(), StandardUnit.COUNT, MetricsLevel.SUMMARY);
            if (leasesInUnknownState > 0) {
                throw new DependencyException(String.format("Encountered an exception while renewing leases. The number of leases which might not have been renewed is %d", leasesInUnknownState), lastException);
            }
            success = true;
        }
        finally {
            MetricsUtil.addWorkerIdentifier(scope, this.workerIdentifier);
            MetricsUtil.addSuccessAndLatency(scope, success, startTime, MetricsLevel.SUMMARY);
            MetricsUtil.endScope(scope);
        }
    }

    private boolean renewLease(Lease lease) throws DependencyException, InvalidStateException {
        return this.renewLease(lease, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean renewLease(Lease lease, boolean renewEvenIfExpired) throws DependencyException, InvalidStateException {
        String leaseKey = lease.leaseKey();
        MetricsScope scope = MetricsUtil.createMetricsWithOperation(this.metricsFactory, RENEW_ALL_LEASES_DIMENSION);
        boolean success = false;
        boolean renewedLease = false;
        long startTime = System.currentTimeMillis();
        try {
            for (int i = 1; i <= 2; ++i) {
                try {
                    Lease lease2 = lease;
                    synchronized (lease2) {
                        boolean isLeaseExpired = lease.isExpired(this.leaseDurationNanos, System.nanoTime());
                        if (renewEvenIfExpired || !isLeaseExpired) {
                            renewedLease = this.leaseRefresher.renewLease(lease);
                        }
                        if (renewedLease) {
                            lease.lastCounterIncrementNanos(System.nanoTime());
                        }
                    }
                    if (renewedLease) {
                        if (log.isDebugEnabled()) {
                            log.debug("Worker {} successfully renewed lease with key {}", (Object)this.workerIdentifier, (Object)leaseKey);
                        }
                    } else {
                        log.info("Worker {} lost lease with key {}", (Object)this.workerIdentifier, (Object)leaseKey);
                        this.ownedLeases.remove(leaseKey);
                    }
                    success = true;
                    break;
                }
                catch (ProvisionedThroughputException e) {
                    log.info("Worker {} could not renew lease with key {} on try {} out of {} due to capacity", new Object[]{this.workerIdentifier, leaseKey, i, 2});
                    continue;
                }
            }
        }
        finally {
            MetricsUtil.addWorkerIdentifier(scope, this.workerIdentifier);
            MetricsUtil.addSuccessAndLatency(scope, "RenewLease", success, startTime, MetricsLevel.DETAILED);
            MetricsUtil.endScope(scope);
        }
        return renewedLease;
    }

    @Override
    public Map<String, Lease> getCurrentlyHeldLeases() {
        HashMap<String, Lease> result = new HashMap<String, Lease>();
        long now = System.nanoTime();
        for (String leaseKey : this.ownedLeases.keySet()) {
            Lease copy = this.getCopyOfHeldLease(leaseKey, now);
            if (copy == null) continue;
            result.put(copy.leaseKey(), copy);
        }
        return result;
    }

    @Override
    public Lease getCurrentlyHeldLease(String leaseKey) {
        return this.getCopyOfHeldLease(leaseKey, System.nanoTime());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Lease getCopyOfHeldLease(String leaseKey, long now) {
        Lease authoritativeLease = (Lease)this.ownedLeases.get(leaseKey);
        if (authoritativeLease == null) {
            return null;
        }
        Lease copy = null;
        Lease lease = authoritativeLease;
        synchronized (lease) {
            copy = authoritativeLease.copy();
        }
        if (copy.isExpired(this.leaseDurationNanos, now)) {
            log.info("getCurrentlyHeldLease not returning lease with key {} because it is expired", (Object)copy.leaseKey());
            return null;
        }
        return copy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean updateLease(Lease lease, UUID concurrencyToken, @NonNull String operation, String shardId) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        if (operation == null) {
            throw new NullPointerException("operation");
        }
        this.verifyNotNull(lease, "lease cannot be null");
        this.verifyNotNull(lease.leaseKey(), "leaseKey cannot be null");
        this.verifyNotNull(concurrencyToken, "concurrencyToken cannot be null");
        String leaseKey = lease.leaseKey();
        Lease authoritativeLease = (Lease)this.ownedLeases.get(leaseKey);
        if (authoritativeLease == null) {
            log.info("Worker {} could not update lease with key {} because it does not hold it", (Object)this.workerIdentifier, (Object)leaseKey);
            return false;
        }
        if (!authoritativeLease.concurrencyToken().equals(concurrencyToken)) {
            log.info("Worker {} refusing to update lease with key {} because concurrency tokens don't match", (Object)this.workerIdentifier, (Object)leaseKey);
            return false;
        }
        MetricsScope scope = MetricsUtil.createMetricsWithOperation(this.metricsFactory, operation);
        if (StringUtils.isNotEmpty((String)shardId)) {
            MetricsUtil.addShardId(scope, shardId);
        }
        long startTime = System.currentTimeMillis();
        boolean success = false;
        try {
            Lease lease2 = authoritativeLease;
            synchronized (lease2) {
                authoritativeLease.update(lease);
                boolean updatedLease = this.leaseRefresher.updateLease(authoritativeLease);
                if (updatedLease) {
                    authoritativeLease.lastCounterIncrementNanos(System.nanoTime());
                } else {
                    log.info("Worker {} lost lease with key {} - discovered during update", (Object)this.workerIdentifier, (Object)leaseKey);
                    this.ownedLeases.remove(leaseKey, authoritativeLease);
                }
                success = true;
                boolean bl = updatedLease;
                return bl;
            }
        }
        finally {
            MetricsUtil.addSuccessAndLatency(scope, "UpdateLease", success, startTime, MetricsLevel.DETAILED);
            MetricsUtil.endScope(scope);
        }
    }

    @Override
    public void addLeasesToRenew(Collection<Lease> newLeases) {
        this.verifyNotNull(newLeases, "newLeases cannot be null");
        for (Lease lease : newLeases) {
            if (lease.lastCounterIncrementNanos() == null) {
                log.info("addLeasesToRenew ignoring lease with key {} because it does not have lastRenewalNanos set", (Object)lease.leaseKey());
                continue;
            }
            Lease authoritativeLease = lease.copy();
            authoritativeLease.concurrencyToken(UUID.randomUUID());
            this.ownedLeases.put(authoritativeLease.leaseKey(), authoritativeLease);
        }
    }

    @Override
    public void clearCurrentlyHeldLeases() {
        this.ownedLeases.clear();
    }

    @Override
    public void dropLease(Lease lease) {
        this.ownedLeases.remove(lease.leaseKey());
    }

    @Override
    public void initialize() throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        List<Lease> leases = this.leaseRefresher.listLeases();
        LinkedList<Lease> myLeases = new LinkedList<Lease>();
        boolean renewEvenIfExpired = true;
        for (Lease lease : leases) {
            if (this.workerIdentifier.equals(lease.leaseOwner())) {
                log.info(" Worker {} found lease {}", (Object)this.workerIdentifier, (Object)lease);
                if (!this.renewLease(lease, renewEvenIfExpired)) continue;
                myLeases.add(lease);
                continue;
            }
            log.debug("Worker {} ignoring lease {} ", (Object)this.workerIdentifier, (Object)lease);
        }
        this.addLeasesToRenew(myLeases);
    }

    private void verifyNotNull(Object object, String message) {
        if (object == null) {
            throw new IllegalArgumentException(message);
        }
    }

    private class RenewLeaseTask
    implements Callable<Boolean> {
        private final Lease lease;

        @Override
        public Boolean call() throws Exception {
            return DynamoDBLeaseRenewer.this.renewLease(this.lease);
        }

        public RenewLeaseTask(Lease lease) {
            this.lease = lease;
        }
    }
}

