/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.documentdb.jdbc.persist;

import com.google.common.collect.Streams;
import com.mongodb.MongoClientSettings;
import com.mongodb.MongoException;
import com.mongodb.client.ClientSession;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.Updates;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import com.mongodb.connection.ClusterSettings;
import java.sql.SQLException;
import java.time.Instant;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import lombok.NonNull;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.documentdb.jdbc.DocumentDbConnectionProperties;
import software.amazon.documentdb.jdbc.common.utilities.SqlError;
import software.amazon.documentdb.jdbc.common.utilities.SqlState;
import software.amazon.documentdb.jdbc.metadata.DocumentDbSchema;
import software.amazon.documentdb.jdbc.metadata.DocumentDbSchemaColumn;
import software.amazon.documentdb.jdbc.metadata.DocumentDbSchemaTable;
import software.amazon.documentdb.jdbc.persist.DocumentDbSchemaReader;
import software.amazon.documentdb.jdbc.persist.DocumentDbSchemaSecurityException;

public class DocumentDbSchemaWriter
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(DocumentDbSchemaWriter.class);
    static final int MONGO_AUTHORIZATION_FAILURE = 13;
    private static final int MONGO_ALREADY_EXISTS = 48;
    private final DocumentDbConnectionProperties properties;
    private final MongoClient client;
    private final boolean closeClient;

    public DocumentDbSchemaWriter(@NonNull DocumentDbConnectionProperties properties, MongoClient client) {
        if (properties == null) {
            throw new NullPointerException("properties is marked non-null but is null");
        }
        this.properties = properties;
        this.client = client != null ? client : MongoClients.create((MongoClientSettings)properties.buildMongoClientSettings());
        this.closeClient = client == null;
    }

    public void write(@NonNull DocumentDbSchema schema, @NonNull Collection<DocumentDbSchemaTable> tablesSchema) throws SQLException, DocumentDbSchemaSecurityException {
        if (schema == null) {
            throw new NullPointerException("schema is marked non-null but is null");
        }
        if (tablesSchema == null) {
            throw new NullPointerException("tablesSchema is marked non-null but is null");
        }
        MongoDatabase database = DocumentDbSchemaWriter.getDatabase(this.client, this.properties.getDatabase());
        MongoCollection schemasCollection = database.getCollection("_sqlSchemas", DocumentDbSchema.class);
        MongoCollection tableSchemasCollection = database.getCollection("_sqlTableSchemas");
        boolean supportsMultiDocTransactions = DocumentDbSchemaWriter.supportsMultiDocTransactions(this.client, database);
        this.ensureSchemaCollections(database);
        DocumentDbSchemaWriter.runTransactedSession(this.client, supportsMultiDocTransactions, session -> this.upsertSchemaHandleSecurityException((ClientSession)session, (MongoCollection<DocumentDbSchema>)schemasCollection, (MongoCollection<Document>)tableSchemasCollection, schema, tablesSchema));
    }

    public void update(@NonNull DocumentDbSchema schema, @NonNull Collection<DocumentDbSchemaTable> tableSchemas) {
        if (schema == null) {
            throw new NullPointerException("schema is marked non-null but is null");
        }
        if (tableSchemas == null) {
            throw new NullPointerException("tableSchemas is marked non-null but is null");
        }
        String schemaName = schema.getSchemaName();
        MongoDatabase database = DocumentDbSchemaWriter.getDatabase(this.client, this.properties.getDatabase());
        DocumentDbSchema latestSchema = DocumentDbSchemaReader.getSchema(schemaName, -2, database);
        int schemaVersion = this.getSchemaVersion(schema, latestSchema) + 1;
        Set tableReferences = tableSchemas.stream().map(DocumentDbSchemaTable::getId).collect(Collectors.toSet());
        MongoCollection tableSchemasCollection = database.getCollection("_sqlTableSchemas");
        boolean supportsMultiDocTransactions = DocumentDbSchemaWriter.supportsMultiDocTransactions(this.client, database);
        DocumentDbSchemaWriter.runTransactedSession(this.client, supportsMultiDocTransactions, session -> this.upsertSchemaHandleSecurityException((ClientSession)session, (MongoCollection<Document>)tableSchemasCollection, database, schemaName, schemaVersion, schema, tableSchemas, tableReferences));
    }

    public void remove(@NonNull String schemaName) {
        if (schemaName == null) {
            throw new NullPointerException("schemaName is marked non-null but is null");
        }
        this.remove(schemaName, 0);
    }

    public void remove(@NonNull String schemaName, int schemaVersion) {
        if (schemaName == null) {
            throw new NullPointerException("schemaName is marked non-null but is null");
        }
        MongoDatabase database = DocumentDbSchemaWriter.getDatabase(this.client, this.properties.getDatabase());
        MongoCollection schemasCollection = database.getCollection("_sqlSchemas", DocumentDbSchema.class);
        MongoCollection tableSchemasCollection = database.getCollection("_sqlTableSchemas");
        boolean supportsMultiDocTransactions = DocumentDbSchemaWriter.supportsMultiDocTransactions(this.client, database);
        DocumentDbSchemaWriter.runTransactedSession(this.client, supportsMultiDocTransactions, session -> this.deleteSchema((ClientSession)session, (MongoCollection<DocumentDbSchema>)schemasCollection, (MongoCollection<Document>)tableSchemasCollection, schemaName, schemaVersion));
    }

    private static void runTransactedSession(MongoClient client, boolean supportsMultiDocTransactions, Consumer<ClientSession> process) {
        try (ClientSession session = supportsMultiDocTransactions ? client.startSession() : null;){
            DocumentDbSchemaWriter.maybeStartTransaction(session);
            process.accept(session);
            DocumentDbSchemaWriter.maybeCommitTransaction(session);
        }
    }

    private void upsertSchemaHandleSecurityException(ClientSession session, MongoCollection<Document> tableSchemasCollection, MongoDatabase database, String schemaName, int schemaVersion, DocumentDbSchema schema, Collection<DocumentDbSchemaTable> tableSchemas, Set<String> tableReferences) {
        MongoCollection schemaCollection = database.getCollection("_sqlSchemas", DocumentDbSchema.class);
        try {
            this.upsertSchema(session, (MongoCollection<DocumentDbSchema>)schemaCollection, tableSchemasCollection, schemaName, schemaVersion, schema, tableSchemas, tableReferences);
        }
        catch (MongoException e) {
            if (DocumentDbSchemaWriter.isAuthorizationFailure(e)) {
                throw new DocumentDbSchemaSecurityException(e.getMessage(), e);
            }
            throw e;
        }
    }

    private void deleteSchema(ClientSession session, MongoCollection<DocumentDbSchema> schemasCollection, MongoCollection<Document> tableSchemasCollection, String schemaName, int schemaVersion) {
        Bson schemaFilter = DocumentDbSchemaWriter.getSchemaFilter(schemaName, schemaVersion);
        for (DocumentDbSchema schema : schemasCollection.find(schemaFilter)) {
            DocumentDbSchemaWriter.deleteTableSchemas(session, tableSchemasCollection, schema.getTableReferences());
            long numDeleted = DocumentDbSchemaWriter.deleteDatabaseSchema(session, schemasCollection, schemaName, schema.getSchemaVersion());
            if (numDeleted >= 1L) continue;
            throw SqlError.createSQLException(LOGGER, SqlState.DATA_EXCEPTION, SqlError.DELETE_SCHEMA_FAILED, schemaName);
        }
    }

    private void upsertSchemaHandleSecurityException(ClientSession session, MongoCollection<DocumentDbSchema> schemasCollection, MongoCollection<Document> tableSchemasCollection, DocumentDbSchema schema, Collection<DocumentDbSchemaTable> tablesSchema) {
        try {
            this.upsertSchema(session, schemasCollection, tableSchemasCollection, schema, tablesSchema);
        }
        catch (MongoException e) {
            if (DocumentDbSchemaWriter.isAuthorizationFailure(e)) {
                throw new DocumentDbSchemaSecurityException(e.getMessage(), e);
            }
            throw e;
        }
    }

    private void upsertSchema(ClientSession session, MongoCollection<DocumentDbSchema> schemasCollection, MongoCollection<Document> tableSchemasCollection, DocumentDbSchema schema, Collection<DocumentDbSchemaTable> tablesSchema) throws SQLException {
        for (DocumentDbSchemaTable tableSchema : tablesSchema) {
            DocumentDbSchemaWriter.upsertTableSchema(session, tableSchemasCollection, tableSchema, schema.getSchemaName());
        }
        DocumentDbSchemaWriter.upsertDatabaseSchema(session, schemasCollection, schema);
    }

    private void upsertSchema(ClientSession session, MongoCollection<DocumentDbSchema> schemaCollection, MongoCollection<Document> tableSchemasCollection, String schemaName, int schemaVersion, DocumentDbSchema schema, Collection<DocumentDbSchemaTable> tableSchemas, Set<String> tableReferences) throws SQLException {
        this.upsertNewSchema(session, schemaCollection, tableSchemasCollection, schemaName, schemaVersion, schema, tableSchemas, tableReferences);
    }

    private void ensureSchemaCollections(MongoDatabase database) throws DocumentDbSchemaSecurityException {
        this.createCollectionIfNotExists(database, "_sqlSchemas");
        this.createCollectionIfNotExists(database, "_sqlTableSchemas");
    }

    private void upsertNewSchema(ClientSession session, MongoCollection<DocumentDbSchema> schemaCollection, MongoCollection<Document> tableSchemasCollection, String schemaName, int schemaVersion, DocumentDbSchema schema, Collection<DocumentDbSchemaTable> tableSchemas, Set<String> tableReferences) throws SQLException {
        for (DocumentDbSchemaTable tableSchema : tableSchemas) {
            DocumentDbSchemaWriter.upsertTableSchema(session, tableSchemasCollection, tableSchema, schemaName);
        }
        DocumentDbSchema newSchema = new DocumentDbSchema(schema.getSchemaName(), schemaVersion, schema.getSqlName(), new Date(Instant.now().toEpochMilli()), tableReferences);
        DocumentDbSchemaWriter.upsertDatabaseSchema(session, schemaCollection, newSchema);
    }

    private int getSchemaVersion(DocumentDbSchema schema, DocumentDbSchema latestSchema) {
        return latestSchema != null ? Math.max(latestSchema.getSchemaVersion(), schema.getSchemaVersion()) : schema.getSchemaVersion();
    }

    static MongoDatabase getDatabase(MongoClient client, String databaseName) {
        return client.getDatabase(databaseName).withCodecRegistry(DocumentDbSchemaReader.POJO_CODEC_REGISTRY);
    }

    private static boolean supportsMultiDocTransactions(MongoClient client, MongoDatabase database) {
        ClusterSettings settings = client.getClusterDescription().getClusterSettings();
        Document buildInfo = database.runCommand((Bson)Document.parse((String)"{ \"buildInfo\": 1 }"));
        List version = buildInfo.getList((Object)"versionArray", Integer.class);
        boolean supportsMultiDocTransactions = settings.getRequiredReplicaSetName() != null && version != null && !version.isEmpty() && (Integer)version.get(0) >= 4;
        return supportsMultiDocTransactions;
    }

    private static void maybeAbortTransaction(ClientSession session) {
        if (session != null) {
            session.abortTransaction();
        }
    }

    private static void maybeCommitTransaction(ClientSession session) {
        if (session != null) {
            session.commitTransaction();
        }
    }

    private static void maybeStartTransaction(ClientSession session) {
        if (session != null) {
            session.startTransaction();
        }
    }

    private static void upsertDatabaseSchema(@Nullable ClientSession session, @NonNull MongoCollection<DocumentDbSchema> schemasCollection, @NonNull DocumentDbSchema schema) throws SQLException {
        UpdateResult result;
        if (schemasCollection == null) {
            throw new NullPointerException("schemasCollection is marked non-null but is null");
        }
        if (schema == null) {
            throw new NullPointerException("schema is marked non-null but is null");
        }
        Bson schemaFilter = DocumentDbSchemaWriter.getSchemaFilter(schema.getSchemaName(), schema.getSchemaVersion());
        Bson schemaUpdate = DocumentDbSchemaWriter.getSchemaUpdate(schema);
        UpdateOptions upsertOption = new UpdateOptions().upsert(true);
        UpdateResult updateResult = result = session != null ? schemasCollection.updateOne(session, schemaFilter, schemaUpdate, upsertOption) : schemasCollection.updateOne(schemaFilter, schemaUpdate, upsertOption);
        if (!result.wasAcknowledged()) {
            throw SqlError.createSQLException(LOGGER, SqlState.DATA_EXCEPTION, SqlError.UPSERT_SCHEMA_FAILED, schema.getSchemaName());
        }
    }

    private static void upsertTableSchema(@Nullable ClientSession session, @NonNull MongoCollection<Document> tableSchemasCollection, @NonNull DocumentDbSchemaTable tableSchema, @NonNull String schemaName) throws SQLException {
        UpdateResult result;
        if (tableSchemasCollection == null) {
            throw new NullPointerException("tableSchemasCollection is marked non-null but is null");
        }
        if (tableSchema == null) {
            throw new NullPointerException("tableSchema is marked non-null but is null");
        }
        if (schemaName == null) {
            throw new NullPointerException("schemaName is marked non-null but is null");
        }
        Bson tableSchemaFilter = DocumentDbSchemaWriter.getTableSchemaFilter(tableSchema.getId());
        Bson tableSchemaUpdate = DocumentDbSchemaWriter.getTableSchemaUpdate(tableSchema);
        UpdateOptions upsertOption = new UpdateOptions().upsert(true);
        UpdateResult updateResult = result = session != null ? tableSchemasCollection.updateOne(session, tableSchemaFilter, tableSchemaUpdate, upsertOption) : tableSchemasCollection.updateOne(tableSchemaFilter, tableSchemaUpdate, upsertOption);
        if (!result.wasAcknowledged()) {
            throw SqlError.createSQLException(LOGGER, SqlState.DATA_EXCEPTION, SqlError.UPSERT_SCHEMA_FAILED, schemaName);
        }
    }

    private static long deleteDatabaseSchema(ClientSession session, MongoCollection<DocumentDbSchema> schemasCollection, String schemaName, int schemaVersion) throws SQLException {
        DeleteResult result;
        Bson schemaFilter = DocumentDbSchemaWriter.getSchemaFilter(schemaName, schemaVersion);
        DeleteResult deleteResult = result = session != null ? schemasCollection.deleteOne(session, schemaFilter) : schemasCollection.deleteOne(schemaFilter);
        if (!result.wasAcknowledged()) {
            throw SqlError.createSQLException(LOGGER, SqlState.DATA_EXCEPTION, SqlError.DELETE_SCHEMA_FAILED, schemaName);
        }
        return result.getDeletedCount();
    }

    private static void deleteTableSchemas(ClientSession session, MongoCollection<Document> tableSchemasCollection, Collection<String> tableReferences) throws SQLException {
        List tableReferencesFilter = tableReferences.stream().map(DocumentDbSchemaWriter::getTableSchemaFilter).collect(Collectors.toList());
        if (!tableReferencesFilter.isEmpty()) {
            DeleteResult result;
            Bson allTableReferencesFilter = Filters.or(tableReferencesFilter);
            DeleteResult deleteResult = result = session != null ? tableSchemasCollection.deleteMany(session, allTableReferencesFilter) : tableSchemasCollection.deleteMany(allTableReferencesFilter);
            if (!result.wasAcknowledged()) {
                throw SqlError.createSQLException(LOGGER, SqlState.DATA_EXCEPTION, SqlError.DELETE_TABLE_SCHEMA_FAILED, new Object[0]);
            }
            if (result.getDeletedCount() != (long)tableReferencesFilter.size()) {
                LOGGER.warn(SqlError.lookup(SqlError.DELETE_TABLE_SCHEMA_INCONSISTENT, tableReferencesFilter.size(), result.getDeletedCount()));
            }
        }
    }

    private static Bson getTableSchemaUpdate(DocumentDbSchemaTable schemaTable) {
        return Updates.combine((Bson[])new Bson[]{Updates.set((String)"sqlName", (Object)schemaTable.getSqlName()), Updates.set((String)"collectionName", (Object)schemaTable.getCollectionName()), Updates.set((String)"modifyDate", (Object)schemaTable.getModifyDate()), Updates.set((String)"columns", schemaTable.getColumnMap().values().stream().map(c -> new DocumentDbSchemaColumn(c.getFieldPath(), c.getSqlName(), c.getSqlType(), c.getDbType(), c.isIndex(), c.isPrimaryKey(), c.getForeignKeyTableName(), c.getForeignKeyColumnName())).collect(Collectors.toList())), Updates.setOnInsert((String)"uuid", (Object)schemaTable.getUuid())});
    }

    static Bson getTableSchemaFilter(String tableId) {
        return Filters.eq((String)"_id", (Object)tableId);
    }

    private static Bson getSchemaUpdate(DocumentDbSchema schema) {
        return Updates.combine((Bson[])new Bson[]{Updates.set((String)"sqlName", (Object)schema.getSqlName()), Updates.set((String)"modifyDate", (Object)schema.getModifyDate()), Updates.set((String)"tables", schema.getTableReferences()), Updates.setOnInsert((String)"schemaName", (Object)schema.getSchemaName()), Updates.setOnInsert((String)"schemaVersion", (Object)schema.getSchemaVersion())});
    }

    static Bson getSchemaFilter(String schemaName, int schemaVersion) {
        return schemaVersion > 0 ? Filters.and((Bson[])new Bson[]{Filters.eq((String)"schemaName", (Object)schemaName), Filters.eq((String)"schemaVersion", (Object)schemaVersion)}) : Filters.eq((String)"schemaName", (Object)schemaName);
    }

    private void createCollectionIfNotExists(MongoDatabase database, String collectionName) throws DocumentDbSchemaSecurityException {
        if (Streams.stream((Iterable)database.listCollectionNames()).anyMatch(c -> c.equals(collectionName))) {
            return;
        }
        try {
            database.createCollection(collectionName);
        }
        catch (MongoException e) {
            if (e.getCode() == 48) {
                LOGGER.info(String.format("Schema collection '%s' already exists.", collectionName));
            }
            if (DocumentDbSchemaWriter.isAuthorizationFailure(e)) {
                throw new DocumentDbSchemaSecurityException(e.getMessage(), e);
            }
            throw e;
        }
    }

    static boolean isAuthorizationFailure(MongoException e) {
        return e.getCode() == 13 || "authorization failure".equalsIgnoreCase(e.getMessage());
    }

    @Override
    public void close() {
        if (this.closeClient && this.client != null) {
            this.client.close();
        }
    }
}

