/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Additional information that's used by a managed rule group. Many managed rule groups don't require this.
 * </p>
 * <p>
 * The rule groups used for intelligent threat mitigation require additional configuration:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Use the <code>AWSManagedRulesACFPRuleSet</code> configuration object to configure the account creation fraud
 * prevention managed rule group. The configuration includes the registration and sign-up pages of your application and
 * the locations in the account creation request payload of data, such as the user email and phone number fields.
 * </p>
 * </li>
 * <li>
 * <p>
 * Use the <code>AWSManagedRulesAntiDDoSRuleSet</code> configuration object to configure the anti-DDoS managed rule
 * group. The configuration includes the sensitivity levels to use in the rules that typically block and challenge
 * requests that might be participating in DDoS attacks and the specification to use to indicate whether a request can
 * handle a silent browser challenge.
 * </p>
 * </li>
 * <li>
 * <p>
 * Use the <code>AWSManagedRulesATPRuleSet</code> configuration object to configure the account takeover prevention
 * managed rule group. The configuration includes the sign-in page of your application and the locations in the login
 * request payload of data such as the username and password.
 * </p>
 * </li>
 * <li>
 * <p>
 * Use the <code>AWSManagedRulesBotControlRuleSet</code> configuration object to configure the protection level that you
 * want the Bot Control rule group to use.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For example specifications, see the examples section of <a>CreateWebACL</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ManagedRuleGroupConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<ManagedRuleGroupConfig.Builder, ManagedRuleGroupConfig> {
    private static final SdkField<String> LOGIN_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LoginPath").getter(getter(ManagedRuleGroupConfig::loginPath)).setter(setter(Builder::loginPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoginPath").build()).build();

    private static final SdkField<String> PAYLOAD_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PayloadType").getter(getter(ManagedRuleGroupConfig::payloadTypeAsString))
            .setter(setter(Builder::payloadType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PayloadType").build()).build();

    private static final SdkField<UsernameField> USERNAME_FIELD_FIELD = SdkField
            .<UsernameField> builder(MarshallingType.SDK_POJO).memberName("UsernameField")
            .getter(getter(ManagedRuleGroupConfig::usernameField)).setter(setter(Builder::usernameField))
            .constructor(UsernameField::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsernameField").build()).build();

    private static final SdkField<PasswordField> PASSWORD_FIELD_FIELD = SdkField
            .<PasswordField> builder(MarshallingType.SDK_POJO).memberName("PasswordField")
            .getter(getter(ManagedRuleGroupConfig::passwordField)).setter(setter(Builder::passwordField))
            .constructor(PasswordField::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PasswordField").build()).build();

    private static final SdkField<AWSManagedRulesBotControlRuleSet> AWS_MANAGED_RULES_BOT_CONTROL_RULE_SET_FIELD = SdkField
            .<AWSManagedRulesBotControlRuleSet> builder(MarshallingType.SDK_POJO)
            .memberName("AWSManagedRulesBotControlRuleSet")
            .getter(getter(ManagedRuleGroupConfig::awsManagedRulesBotControlRuleSet))
            .setter(setter(Builder::awsManagedRulesBotControlRuleSet))
            .constructor(AWSManagedRulesBotControlRuleSet::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AWSManagedRulesBotControlRuleSet")
                    .build()).build();

    private static final SdkField<AWSManagedRulesATPRuleSet> AWS_MANAGED_RULES_ATP_RULE_SET_FIELD = SdkField
            .<AWSManagedRulesATPRuleSet> builder(MarshallingType.SDK_POJO).memberName("AWSManagedRulesATPRuleSet")
            .getter(getter(ManagedRuleGroupConfig::awsManagedRulesATPRuleSet)).setter(setter(Builder::awsManagedRulesATPRuleSet))
            .constructor(AWSManagedRulesATPRuleSet::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AWSManagedRulesATPRuleSet").build())
            .build();

    private static final SdkField<AWSManagedRulesACFPRuleSet> AWS_MANAGED_RULES_ACFP_RULE_SET_FIELD = SdkField
            .<AWSManagedRulesACFPRuleSet> builder(MarshallingType.SDK_POJO)
            .memberName("AWSManagedRulesACFPRuleSet")
            .getter(getter(ManagedRuleGroupConfig::awsManagedRulesACFPRuleSet))
            .setter(setter(Builder::awsManagedRulesACFPRuleSet))
            .constructor(AWSManagedRulesACFPRuleSet::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AWSManagedRulesACFPRuleSet").build())
            .build();

    private static final SdkField<AWSManagedRulesAntiDDoSRuleSet> AWS_MANAGED_RULES_ANTI_D_DO_S_RULE_SET_FIELD = SdkField
            .<AWSManagedRulesAntiDDoSRuleSet> builder(MarshallingType.SDK_POJO)
            .memberName("AWSManagedRulesAntiDDoSRuleSet")
            .getter(getter(ManagedRuleGroupConfig::awsManagedRulesAntiDDoSRuleSet))
            .setter(setter(Builder::awsManagedRulesAntiDDoSRuleSet))
            .constructor(AWSManagedRulesAntiDDoSRuleSet::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AWSManagedRulesAntiDDoSRuleSet")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOGIN_PATH_FIELD,
            PAYLOAD_TYPE_FIELD, USERNAME_FIELD_FIELD, PASSWORD_FIELD_FIELD, AWS_MANAGED_RULES_BOT_CONTROL_RULE_SET_FIELD,
            AWS_MANAGED_RULES_ATP_RULE_SET_FIELD, AWS_MANAGED_RULES_ACFP_RULE_SET_FIELD,
            AWS_MANAGED_RULES_ANTI_D_DO_S_RULE_SET_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String loginPath;

    private final String payloadType;

    private final UsernameField usernameField;

    private final PasswordField passwordField;

    private final AWSManagedRulesBotControlRuleSet awsManagedRulesBotControlRuleSet;

    private final AWSManagedRulesATPRuleSet awsManagedRulesATPRuleSet;

    private final AWSManagedRulesACFPRuleSet awsManagedRulesACFPRuleSet;

    private final AWSManagedRulesAntiDDoSRuleSet awsManagedRulesAntiDDoSRuleSet;

    private ManagedRuleGroupConfig(BuilderImpl builder) {
        this.loginPath = builder.loginPath;
        this.payloadType = builder.payloadType;
        this.usernameField = builder.usernameField;
        this.passwordField = builder.passwordField;
        this.awsManagedRulesBotControlRuleSet = builder.awsManagedRulesBotControlRuleSet;
        this.awsManagedRulesATPRuleSet = builder.awsManagedRulesATPRuleSet;
        this.awsManagedRulesACFPRuleSet = builder.awsManagedRulesACFPRuleSet;
        this.awsManagedRulesAntiDDoSRuleSet = builder.awsManagedRulesAntiDDoSRuleSet;
    }

    /**
     * <note>
     * <p>
     * Instead of this setting, provide your configuration under <code>AWSManagedRulesATPRuleSet</code>.
     * </p>
     * </note>
     * 
     * @return <p>
     *         Instead of this setting, provide your configuration under <code>AWSManagedRulesATPRuleSet</code>.
     *         </p>
     * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet LoginPath
     */
    @Deprecated
    public final String loginPath() {
        return loginPath;
    }

    /**
     * <note>
     * <p>
     * Instead of this setting, provide your configuration under the request inspection configuration for
     * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #payloadType} will
     * return {@link PayloadType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #payloadTypeAsString}.
     * </p>
     * 
     * @return <p>
     *         Instead of this setting, provide your configuration under the request inspection configuration for
     *         <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
     *         </p>
     * @see PayloadType
     * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection PayloadType
     */
    @Deprecated
    public final PayloadType payloadType() {
        return PayloadType.fromValue(payloadType);
    }

    /**
     * <note>
     * <p>
     * Instead of this setting, provide your configuration under the request inspection configuration for
     * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #payloadType} will
     * return {@link PayloadType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #payloadTypeAsString}.
     * </p>
     * 
     * @return <p>
     *         Instead of this setting, provide your configuration under the request inspection configuration for
     *         <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
     *         </p>
     * @see PayloadType
     * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection PayloadType
     */
    @Deprecated
    public final String payloadTypeAsString() {
        return payloadType;
    }

    /**
     * <note>
     * <p>
     * Instead of this setting, provide your configuration under the request inspection configuration for
     * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
     * </p>
     * </note>
     * 
     * @return <p>
     *         Instead of this setting, provide your configuration under the request inspection configuration for
     *         <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
     *         </p>
     * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection UsernameField
     */
    @Deprecated
    public final UsernameField usernameField() {
        return usernameField;
    }

    /**
     * <note>
     * <p>
     * Instead of this setting, provide your configuration under the request inspection configuration for
     * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
     * </p>
     * </note>
     * 
     * @return <p>
     *         Instead of this setting, provide your configuration under the request inspection configuration for
     *         <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
     *         </p>
     * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection PasswordField
     */
    @Deprecated
    public final PasswordField passwordField() {
        return passwordField;
    }

    /**
     * <p>
     * Additional configuration for using the Bot Control managed rule group. Use this to specify the inspection level
     * that you want to use. For information about using the Bot Control managed rule group, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-bot.html">WAF Bot Control
     * rule group</a> and <a href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-bot-control.html">WAF Bot
     * Control</a> in the <i>WAF Developer Guide</i>.
     * </p>
     * 
     * @return Additional configuration for using the Bot Control managed rule group. Use this to specify the inspection
     *         level that you want to use. For information about using the Bot Control managed rule group, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-bot.html">WAF Bot
     *         Control rule group</a> and <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-bot-control.html">WAF Bot Control</a> in
     *         the <i>WAF Developer Guide</i>.
     */
    public final AWSManagedRulesBotControlRuleSet awsManagedRulesBotControlRuleSet() {
        return awsManagedRulesBotControlRuleSet;
    }

    /**
     * <p>
     * Additional configuration for using the account takeover prevention (ATP) managed rule group,
     * <code>AWSManagedRulesATPRuleSet</code>. Use this to provide login request information to the rule group. For web
     * ACLs that protect CloudFront distributions, use this to also provide the information about how your distribution
     * responds to login requests.
     * </p>
     * <p>
     * This configuration replaces the individual configuration fields in <code>ManagedRuleGroupConfig</code> and
     * provides additional feature configuration.
     * </p>
     * <p>
     * For information about using the ATP managed rule group, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-atp.html">WAF Fraud Control
     * account takeover prevention (ATP) rule group</a> and <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-atp.html">WAF Fraud Control account takeover
     * prevention (ATP)</a> in the <i>WAF Developer Guide</i>.
     * </p>
     * 
     * @return Additional configuration for using the account takeover prevention (ATP) managed rule group,
     *         <code>AWSManagedRulesATPRuleSet</code>. Use this to provide login request information to the rule group.
     *         For web ACLs that protect CloudFront distributions, use this to also provide the information about how
     *         your distribution responds to login requests. </p>
     *         <p>
     *         This configuration replaces the individual configuration fields in <code>ManagedRuleGroupConfig</code>
     *         and provides additional feature configuration.
     *         </p>
     *         <p>
     *         For information about using the ATP managed rule group, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-atp.html">WAF Fraud
     *         Control account takeover prevention (ATP) rule group</a> and <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-atp.html">WAF Fraud Control account
     *         takeover prevention (ATP)</a> in the <i>WAF Developer Guide</i>.
     */
    public final AWSManagedRulesATPRuleSet awsManagedRulesATPRuleSet() {
        return awsManagedRulesATPRuleSet;
    }

    /**
     * <p>
     * Additional configuration for using the account creation fraud prevention (ACFP) managed rule group,
     * <code>AWSManagedRulesACFPRuleSet</code>. Use this to provide account creation request information to the rule
     * group. For web ACLs that protect CloudFront distributions, use this to also provide the information about how
     * your distribution responds to account creation requests.
     * </p>
     * <p>
     * For information about using the ACFP managed rule group, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-acfp.html">WAF Fraud Control
     * account creation fraud prevention (ACFP) rule group</a> and <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-acfp.html">WAF Fraud Control account creation
     * fraud prevention (ACFP)</a> in the <i>WAF Developer Guide</i>.
     * </p>
     * 
     * @return Additional configuration for using the account creation fraud prevention (ACFP) managed rule group,
     *         <code>AWSManagedRulesACFPRuleSet</code>. Use this to provide account creation request information to the
     *         rule group. For web ACLs that protect CloudFront distributions, use this to also provide the information
     *         about how your distribution responds to account creation requests. </p>
     *         <p>
     *         For information about using the ACFP managed rule group, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-acfp.html">WAF Fraud
     *         Control account creation fraud prevention (ACFP) rule group</a> and <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-acfp.html">WAF Fraud Control account
     *         creation fraud prevention (ACFP)</a> in the <i>WAF Developer Guide</i>.
     */
    public final AWSManagedRulesACFPRuleSet awsManagedRulesACFPRuleSet() {
        return awsManagedRulesACFPRuleSet;
    }

    /**
     * <p>
     * Additional configuration for using the anti-DDoS managed rule group, <code>AWSManagedRulesAntiDDoSRuleSet</code>.
     * Use this to configure anti-DDoS behavior for the rule group.
     * </p>
     * <p>
     * For information about using the anti-DDoS managed rule group, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-anti-ddos.html">WAF Anti-DDoS
     * rule group</a> and <a href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-anti-ddos.html">Distributed
     * Denial of Service (DDoS) prevention</a> in the <i>WAF Developer Guide</i>.
     * </p>
     * 
     * @return Additional configuration for using the anti-DDoS managed rule group,
     *         <code>AWSManagedRulesAntiDDoSRuleSet</code>. Use this to configure anti-DDoS behavior for the rule group.
     *         </p>
     *         <p>
     *         For information about using the anti-DDoS managed rule group, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-anti-ddos.html">WAF
     *         Anti-DDoS rule group</a> and <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-anti-ddos.html">Distributed Denial of
     *         Service (DDoS) prevention</a> in the <i>WAF Developer Guide</i>.
     */
    public final AWSManagedRulesAntiDDoSRuleSet awsManagedRulesAntiDDoSRuleSet() {
        return awsManagedRulesAntiDDoSRuleSet;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(loginPath());
        hashCode = 31 * hashCode + Objects.hashCode(payloadTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(usernameField());
        hashCode = 31 * hashCode + Objects.hashCode(passwordField());
        hashCode = 31 * hashCode + Objects.hashCode(awsManagedRulesBotControlRuleSet());
        hashCode = 31 * hashCode + Objects.hashCode(awsManagedRulesATPRuleSet());
        hashCode = 31 * hashCode + Objects.hashCode(awsManagedRulesACFPRuleSet());
        hashCode = 31 * hashCode + Objects.hashCode(awsManagedRulesAntiDDoSRuleSet());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ManagedRuleGroupConfig)) {
            return false;
        }
        ManagedRuleGroupConfig other = (ManagedRuleGroupConfig) obj;
        return Objects.equals(loginPath(), other.loginPath())
                && Objects.equals(payloadTypeAsString(), other.payloadTypeAsString())
                && Objects.equals(usernameField(), other.usernameField())
                && Objects.equals(passwordField(), other.passwordField())
                && Objects.equals(awsManagedRulesBotControlRuleSet(), other.awsManagedRulesBotControlRuleSet())
                && Objects.equals(awsManagedRulesATPRuleSet(), other.awsManagedRulesATPRuleSet())
                && Objects.equals(awsManagedRulesACFPRuleSet(), other.awsManagedRulesACFPRuleSet())
                && Objects.equals(awsManagedRulesAntiDDoSRuleSet(), other.awsManagedRulesAntiDDoSRuleSet());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ManagedRuleGroupConfig").add("LoginPath", loginPath()).add("PayloadType", payloadTypeAsString())
                .add("UsernameField", usernameField()).add("PasswordField", passwordField())
                .add("AWSManagedRulesBotControlRuleSet", awsManagedRulesBotControlRuleSet())
                .add("AWSManagedRulesATPRuleSet", awsManagedRulesATPRuleSet())
                .add("AWSManagedRulesACFPRuleSet", awsManagedRulesACFPRuleSet())
                .add("AWSManagedRulesAntiDDoSRuleSet", awsManagedRulesAntiDDoSRuleSet()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LoginPath":
            return Optional.ofNullable(clazz.cast(loginPath()));
        case "PayloadType":
            return Optional.ofNullable(clazz.cast(payloadTypeAsString()));
        case "UsernameField":
            return Optional.ofNullable(clazz.cast(usernameField()));
        case "PasswordField":
            return Optional.ofNullable(clazz.cast(passwordField()));
        case "AWSManagedRulesBotControlRuleSet":
            return Optional.ofNullable(clazz.cast(awsManagedRulesBotControlRuleSet()));
        case "AWSManagedRulesATPRuleSet":
            return Optional.ofNullable(clazz.cast(awsManagedRulesATPRuleSet()));
        case "AWSManagedRulesACFPRuleSet":
            return Optional.ofNullable(clazz.cast(awsManagedRulesACFPRuleSet()));
        case "AWSManagedRulesAntiDDoSRuleSet":
            return Optional.ofNullable(clazz.cast(awsManagedRulesAntiDDoSRuleSet()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("LoginPath", LOGIN_PATH_FIELD);
        map.put("PayloadType", PAYLOAD_TYPE_FIELD);
        map.put("UsernameField", USERNAME_FIELD_FIELD);
        map.put("PasswordField", PASSWORD_FIELD_FIELD);
        map.put("AWSManagedRulesBotControlRuleSet", AWS_MANAGED_RULES_BOT_CONTROL_RULE_SET_FIELD);
        map.put("AWSManagedRulesATPRuleSet", AWS_MANAGED_RULES_ATP_RULE_SET_FIELD);
        map.put("AWSManagedRulesACFPRuleSet", AWS_MANAGED_RULES_ACFP_RULE_SET_FIELD);
        map.put("AWSManagedRulesAntiDDoSRuleSet", AWS_MANAGED_RULES_ANTI_D_DO_S_RULE_SET_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ManagedRuleGroupConfig, T> g) {
        return obj -> g.apply((ManagedRuleGroupConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ManagedRuleGroupConfig> {
        /**
         * <note>
         * <p>
         * Instead of this setting, provide your configuration under <code>AWSManagedRulesATPRuleSet</code>.
         * </p>
         * </note>
         * 
         * @param loginPath
         *        <p>
         *        Instead of this setting, provide your configuration under <code>AWSManagedRulesATPRuleSet</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet LoginPath
         */
        @Deprecated
        Builder loginPath(String loginPath);

        /**
         * <note>
         * <p>
         * Instead of this setting, provide your configuration under the request inspection configuration for
         * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         * </p>
         * </note>
         * 
         * @param payloadType
         *        <p>
         *        Instead of this setting, provide your configuration under the request inspection configuration for
         *        <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         *        </p>
         * @see PayloadType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PayloadType
         * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection PayloadType
         */
        @Deprecated
        Builder payloadType(String payloadType);

        /**
         * <note>
         * <p>
         * Instead of this setting, provide your configuration under the request inspection configuration for
         * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         * </p>
         * </note>
         * 
         * @param payloadType
         *        <p>
         *        Instead of this setting, provide your configuration under the request inspection configuration for
         *        <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         *        </p>
         * @see PayloadType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PayloadType
         * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection PayloadType
         */
        @Deprecated
        Builder payloadType(PayloadType payloadType);

        /**
         * <note>
         * <p>
         * Instead of this setting, provide your configuration under the request inspection configuration for
         * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         * </p>
         * </note>
         * 
         * @param usernameField
         *        <p>
         *        Instead of this setting, provide your configuration under the request inspection configuration for
         *        <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection UsernameField
         */
        @Deprecated
        Builder usernameField(UsernameField usernameField);

        /**
         * <note>
         * <p>
         * Instead of this setting, provide your configuration under the request inspection configuration for
         * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link UsernameField.Builder} avoiding
         * the need to create one manually via {@link UsernameField#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UsernameField.Builder#build()} is called immediately and its
         * result is passed to {@link #usernameField(UsernameField)}.
         * 
         * @param usernameField
         *        a consumer that will call methods on {@link UsernameField.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #usernameField(UsernameField)
         * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection UsernameField
         */
        @Deprecated
        default Builder usernameField(Consumer<UsernameField.Builder> usernameField) {
            return usernameField(UsernameField.builder().applyMutation(usernameField).build());
        }

        /**
         * <note>
         * <p>
         * Instead of this setting, provide your configuration under the request inspection configuration for
         * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         * </p>
         * </note>
         * 
         * @param passwordField
         *        <p>
         *        Instead of this setting, provide your configuration under the request inspection configuration for
         *        <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection PasswordField
         */
        @Deprecated
        Builder passwordField(PasswordField passwordField);

        /**
         * <note>
         * <p>
         * Instead of this setting, provide your configuration under the request inspection configuration for
         * <code>AWSManagedRulesATPRuleSet</code> or <code>AWSManagedRulesACFPRuleSet</code>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link PasswordField.Builder} avoiding
         * the need to create one manually via {@link PasswordField#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PasswordField.Builder#build()} is called immediately and its
         * result is passed to {@link #passwordField(PasswordField)}.
         * 
         * @param passwordField
         *        a consumer that will call methods on {@link PasswordField.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #passwordField(PasswordField)
         * @deprecated Deprecated. Use AWSManagedRulesATPRuleSet RequestInspection PasswordField
         */
        @Deprecated
        default Builder passwordField(Consumer<PasswordField.Builder> passwordField) {
            return passwordField(PasswordField.builder().applyMutation(passwordField).build());
        }

        /**
         * <p>
         * Additional configuration for using the Bot Control managed rule group. Use this to specify the inspection
         * level that you want to use. For information about using the Bot Control managed rule group, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-bot.html">WAF Bot Control
         * rule group</a> and <a href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-bot-control.html">WAF
         * Bot Control</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * 
         * @param awsManagedRulesBotControlRuleSet
         *        Additional configuration for using the Bot Control managed rule group. Use this to specify the
         *        inspection level that you want to use. For information about using the Bot Control managed rule group,
         *        see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-bot.html">WAF Bot
         *        Control rule group</a> and <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-bot-control.html">WAF Bot Control</a>
         *        in the <i>WAF Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsManagedRulesBotControlRuleSet(AWSManagedRulesBotControlRuleSet awsManagedRulesBotControlRuleSet);

        /**
         * <p>
         * Additional configuration for using the Bot Control managed rule group. Use this to specify the inspection
         * level that you want to use. For information about using the Bot Control managed rule group, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-bot.html">WAF Bot Control
         * rule group</a> and <a href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-bot-control.html">WAF
         * Bot Control</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link AWSManagedRulesBotControlRuleSet.Builder}
         * avoiding the need to create one manually via {@link AWSManagedRulesBotControlRuleSet#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AWSManagedRulesBotControlRuleSet.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #awsManagedRulesBotControlRuleSet(AWSManagedRulesBotControlRuleSet)}.
         * 
         * @param awsManagedRulesBotControlRuleSet
         *        a consumer that will call methods on {@link AWSManagedRulesBotControlRuleSet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #awsManagedRulesBotControlRuleSet(AWSManagedRulesBotControlRuleSet)
         */
        default Builder awsManagedRulesBotControlRuleSet(
                Consumer<AWSManagedRulesBotControlRuleSet.Builder> awsManagedRulesBotControlRuleSet) {
            return awsManagedRulesBotControlRuleSet(AWSManagedRulesBotControlRuleSet.builder()
                    .applyMutation(awsManagedRulesBotControlRuleSet).build());
        }

        /**
         * <p>
         * Additional configuration for using the account takeover prevention (ATP) managed rule group,
         * <code>AWSManagedRulesATPRuleSet</code>. Use this to provide login request information to the rule group. For
         * web ACLs that protect CloudFront distributions, use this to also provide the information about how your
         * distribution responds to login requests.
         * </p>
         * <p>
         * This configuration replaces the individual configuration fields in <code>ManagedRuleGroupConfig</code> and
         * provides additional feature configuration.
         * </p>
         * <p>
         * For information about using the ATP managed rule group, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-atp.html">WAF Fraud
         * Control account takeover prevention (ATP) rule group</a> and <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-atp.html">WAF Fraud Control account takeover
         * prevention (ATP)</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * 
         * @param awsManagedRulesATPRuleSet
         *        Additional configuration for using the account takeover prevention (ATP) managed rule group,
         *        <code>AWSManagedRulesATPRuleSet</code>. Use this to provide login request information to the rule
         *        group. For web ACLs that protect CloudFront distributions, use this to also provide the information
         *        about how your distribution responds to login requests. </p>
         *        <p>
         *        This configuration replaces the individual configuration fields in <code>ManagedRuleGroupConfig</code>
         *        and provides additional feature configuration.
         *        </p>
         *        <p>
         *        For information about using the ATP managed rule group, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-atp.html">WAF
         *        Fraud Control account takeover prevention (ATP) rule group</a> and <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-atp.html">WAF Fraud Control account
         *        takeover prevention (ATP)</a> in the <i>WAF Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsManagedRulesATPRuleSet(AWSManagedRulesATPRuleSet awsManagedRulesATPRuleSet);

        /**
         * <p>
         * Additional configuration for using the account takeover prevention (ATP) managed rule group,
         * <code>AWSManagedRulesATPRuleSet</code>. Use this to provide login request information to the rule group. For
         * web ACLs that protect CloudFront distributions, use this to also provide the information about how your
         * distribution responds to login requests.
         * </p>
         * <p>
         * This configuration replaces the individual configuration fields in <code>ManagedRuleGroupConfig</code> and
         * provides additional feature configuration.
         * </p>
         * <p>
         * For information about using the ATP managed rule group, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-atp.html">WAF Fraud
         * Control account takeover prevention (ATP) rule group</a> and <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-atp.html">WAF Fraud Control account takeover
         * prevention (ATP)</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link AWSManagedRulesATPRuleSet.Builder}
         * avoiding the need to create one manually via {@link AWSManagedRulesATPRuleSet#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AWSManagedRulesATPRuleSet.Builder#build()} is called immediately
         * and its result is passed to {@link #awsManagedRulesATPRuleSet(AWSManagedRulesATPRuleSet)}.
         * 
         * @param awsManagedRulesATPRuleSet
         *        a consumer that will call methods on {@link AWSManagedRulesATPRuleSet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #awsManagedRulesATPRuleSet(AWSManagedRulesATPRuleSet)
         */
        default Builder awsManagedRulesATPRuleSet(Consumer<AWSManagedRulesATPRuleSet.Builder> awsManagedRulesATPRuleSet) {
            return awsManagedRulesATPRuleSet(AWSManagedRulesATPRuleSet.builder().applyMutation(awsManagedRulesATPRuleSet).build());
        }

        /**
         * <p>
         * Additional configuration for using the account creation fraud prevention (ACFP) managed rule group,
         * <code>AWSManagedRulesACFPRuleSet</code>. Use this to provide account creation request information to the rule
         * group. For web ACLs that protect CloudFront distributions, use this to also provide the information about how
         * your distribution responds to account creation requests.
         * </p>
         * <p>
         * For information about using the ACFP managed rule group, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-acfp.html">WAF Fraud
         * Control account creation fraud prevention (ACFP) rule group</a> and <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-acfp.html">WAF Fraud Control account creation
         * fraud prevention (ACFP)</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * 
         * @param awsManagedRulesACFPRuleSet
         *        Additional configuration for using the account creation fraud prevention (ACFP) managed rule group,
         *        <code>AWSManagedRulesACFPRuleSet</code>. Use this to provide account creation request information to
         *        the rule group. For web ACLs that protect CloudFront distributions, use this to also provide the
         *        information about how your distribution responds to account creation requests. </p>
         *        <p>
         *        For information about using the ACFP managed rule group, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-acfp.html">WAF
         *        Fraud Control account creation fraud prevention (ACFP) rule group</a> and <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-acfp.html">WAF Fraud Control account
         *        creation fraud prevention (ACFP)</a> in the <i>WAF Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsManagedRulesACFPRuleSet(AWSManagedRulesACFPRuleSet awsManagedRulesACFPRuleSet);

        /**
         * <p>
         * Additional configuration for using the account creation fraud prevention (ACFP) managed rule group,
         * <code>AWSManagedRulesACFPRuleSet</code>. Use this to provide account creation request information to the rule
         * group. For web ACLs that protect CloudFront distributions, use this to also provide the information about how
         * your distribution responds to account creation requests.
         * </p>
         * <p>
         * For information about using the ACFP managed rule group, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-acfp.html">WAF Fraud
         * Control account creation fraud prevention (ACFP) rule group</a> and <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-acfp.html">WAF Fraud Control account creation
         * fraud prevention (ACFP)</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link AWSManagedRulesACFPRuleSet.Builder}
         * avoiding the need to create one manually via {@link AWSManagedRulesACFPRuleSet#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AWSManagedRulesACFPRuleSet.Builder#build()} is called immediately
         * and its result is passed to {@link #awsManagedRulesACFPRuleSet(AWSManagedRulesACFPRuleSet)}.
         * 
         * @param awsManagedRulesACFPRuleSet
         *        a consumer that will call methods on {@link AWSManagedRulesACFPRuleSet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #awsManagedRulesACFPRuleSet(AWSManagedRulesACFPRuleSet)
         */
        default Builder awsManagedRulesACFPRuleSet(Consumer<AWSManagedRulesACFPRuleSet.Builder> awsManagedRulesACFPRuleSet) {
            return awsManagedRulesACFPRuleSet(AWSManagedRulesACFPRuleSet.builder().applyMutation(awsManagedRulesACFPRuleSet)
                    .build());
        }

        /**
         * <p>
         * Additional configuration for using the anti-DDoS managed rule group,
         * <code>AWSManagedRulesAntiDDoSRuleSet</code>. Use this to configure anti-DDoS behavior for the rule group.
         * </p>
         * <p>
         * For information about using the anti-DDoS managed rule group, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-anti-ddos.html">WAF
         * Anti-DDoS rule group</a> and <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-anti-ddos.html">Distributed Denial of Service
         * (DDoS) prevention</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * 
         * @param awsManagedRulesAntiDDoSRuleSet
         *        Additional configuration for using the anti-DDoS managed rule group,
         *        <code>AWSManagedRulesAntiDDoSRuleSet</code>. Use this to configure anti-DDoS behavior for the rule
         *        group. </p>
         *        <p>
         *        For information about using the anti-DDoS managed rule group, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-anti-ddos.html"
         *        >WAF Anti-DDoS rule group</a> and <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-anti-ddos.html">Distributed Denial of
         *        Service (DDoS) prevention</a> in the <i>WAF Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsManagedRulesAntiDDoSRuleSet(AWSManagedRulesAntiDDoSRuleSet awsManagedRulesAntiDDoSRuleSet);

        /**
         * <p>
         * Additional configuration for using the anti-DDoS managed rule group,
         * <code>AWSManagedRulesAntiDDoSRuleSet</code>. Use this to configure anti-DDoS behavior for the rule group.
         * </p>
         * <p>
         * For information about using the anti-DDoS managed rule group, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-anti-ddos.html">WAF
         * Anti-DDoS rule group</a> and <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-anti-ddos.html">Distributed Denial of Service
         * (DDoS) prevention</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link AWSManagedRulesAntiDDoSRuleSet.Builder}
         * avoiding the need to create one manually via {@link AWSManagedRulesAntiDDoSRuleSet#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AWSManagedRulesAntiDDoSRuleSet.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #awsManagedRulesAntiDDoSRuleSet(AWSManagedRulesAntiDDoSRuleSet)}.
         * 
         * @param awsManagedRulesAntiDDoSRuleSet
         *        a consumer that will call methods on {@link AWSManagedRulesAntiDDoSRuleSet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #awsManagedRulesAntiDDoSRuleSet(AWSManagedRulesAntiDDoSRuleSet)
         */
        default Builder awsManagedRulesAntiDDoSRuleSet(
                Consumer<AWSManagedRulesAntiDDoSRuleSet.Builder> awsManagedRulesAntiDDoSRuleSet) {
            return awsManagedRulesAntiDDoSRuleSet(AWSManagedRulesAntiDDoSRuleSet.builder()
                    .applyMutation(awsManagedRulesAntiDDoSRuleSet).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String loginPath;

        private String payloadType;

        private UsernameField usernameField;

        private PasswordField passwordField;

        private AWSManagedRulesBotControlRuleSet awsManagedRulesBotControlRuleSet;

        private AWSManagedRulesATPRuleSet awsManagedRulesATPRuleSet;

        private AWSManagedRulesACFPRuleSet awsManagedRulesACFPRuleSet;

        private AWSManagedRulesAntiDDoSRuleSet awsManagedRulesAntiDDoSRuleSet;

        private BuilderImpl() {
        }

        private BuilderImpl(ManagedRuleGroupConfig model) {
            loginPath(model.loginPath);
            payloadType(model.payloadType);
            usernameField(model.usernameField);
            passwordField(model.passwordField);
            awsManagedRulesBotControlRuleSet(model.awsManagedRulesBotControlRuleSet);
            awsManagedRulesATPRuleSet(model.awsManagedRulesATPRuleSet);
            awsManagedRulesACFPRuleSet(model.awsManagedRulesACFPRuleSet);
            awsManagedRulesAntiDDoSRuleSet(model.awsManagedRulesAntiDDoSRuleSet);
        }

        @Deprecated
        public final String getLoginPath() {
            return loginPath;
        }

        @Deprecated
        public final void setLoginPath(String loginPath) {
            this.loginPath = loginPath;
        }

        @Override
        @Deprecated
        public final Builder loginPath(String loginPath) {
            this.loginPath = loginPath;
            return this;
        }

        @Deprecated
        public final String getPayloadType() {
            return payloadType;
        }

        @Deprecated
        public final void setPayloadType(String payloadType) {
            this.payloadType = payloadType;
        }

        @Override
        @Deprecated
        public final Builder payloadType(String payloadType) {
            this.payloadType = payloadType;
            return this;
        }

        @Override
        @Deprecated
        public final Builder payloadType(PayloadType payloadType) {
            this.payloadType(payloadType == null ? null : payloadType.toString());
            return this;
        }

        @Deprecated
        public final UsernameField.Builder getUsernameField() {
            return usernameField != null ? usernameField.toBuilder() : null;
        }

        @Deprecated
        public final void setUsernameField(UsernameField.BuilderImpl usernameField) {
            this.usernameField = usernameField != null ? usernameField.build() : null;
        }

        @Override
        @Deprecated
        public final Builder usernameField(UsernameField usernameField) {
            this.usernameField = usernameField;
            return this;
        }

        @Deprecated
        public final PasswordField.Builder getPasswordField() {
            return passwordField != null ? passwordField.toBuilder() : null;
        }

        @Deprecated
        public final void setPasswordField(PasswordField.BuilderImpl passwordField) {
            this.passwordField = passwordField != null ? passwordField.build() : null;
        }

        @Override
        @Deprecated
        public final Builder passwordField(PasswordField passwordField) {
            this.passwordField = passwordField;
            return this;
        }

        public final AWSManagedRulesBotControlRuleSet.Builder getAwsManagedRulesBotControlRuleSet() {
            return awsManagedRulesBotControlRuleSet != null ? awsManagedRulesBotControlRuleSet.toBuilder() : null;
        }

        public final void setAwsManagedRulesBotControlRuleSet(
                AWSManagedRulesBotControlRuleSet.BuilderImpl awsManagedRulesBotControlRuleSet) {
            this.awsManagedRulesBotControlRuleSet = awsManagedRulesBotControlRuleSet != null ? awsManagedRulesBotControlRuleSet
                    .build() : null;
        }

        @Override
        public final Builder awsManagedRulesBotControlRuleSet(AWSManagedRulesBotControlRuleSet awsManagedRulesBotControlRuleSet) {
            this.awsManagedRulesBotControlRuleSet = awsManagedRulesBotControlRuleSet;
            return this;
        }

        public final AWSManagedRulesATPRuleSet.Builder getAwsManagedRulesATPRuleSet() {
            return awsManagedRulesATPRuleSet != null ? awsManagedRulesATPRuleSet.toBuilder() : null;
        }

        public final void setAwsManagedRulesATPRuleSet(AWSManagedRulesATPRuleSet.BuilderImpl awsManagedRulesATPRuleSet) {
            this.awsManagedRulesATPRuleSet = awsManagedRulesATPRuleSet != null ? awsManagedRulesATPRuleSet.build() : null;
        }

        @Override
        public final Builder awsManagedRulesATPRuleSet(AWSManagedRulesATPRuleSet awsManagedRulesATPRuleSet) {
            this.awsManagedRulesATPRuleSet = awsManagedRulesATPRuleSet;
            return this;
        }

        public final AWSManagedRulesACFPRuleSet.Builder getAwsManagedRulesACFPRuleSet() {
            return awsManagedRulesACFPRuleSet != null ? awsManagedRulesACFPRuleSet.toBuilder() : null;
        }

        public final void setAwsManagedRulesACFPRuleSet(AWSManagedRulesACFPRuleSet.BuilderImpl awsManagedRulesACFPRuleSet) {
            this.awsManagedRulesACFPRuleSet = awsManagedRulesACFPRuleSet != null ? awsManagedRulesACFPRuleSet.build() : null;
        }

        @Override
        public final Builder awsManagedRulesACFPRuleSet(AWSManagedRulesACFPRuleSet awsManagedRulesACFPRuleSet) {
            this.awsManagedRulesACFPRuleSet = awsManagedRulesACFPRuleSet;
            return this;
        }

        public final AWSManagedRulesAntiDDoSRuleSet.Builder getAwsManagedRulesAntiDDoSRuleSet() {
            return awsManagedRulesAntiDDoSRuleSet != null ? awsManagedRulesAntiDDoSRuleSet.toBuilder() : null;
        }

        public final void setAwsManagedRulesAntiDDoSRuleSet(
                AWSManagedRulesAntiDDoSRuleSet.BuilderImpl awsManagedRulesAntiDDoSRuleSet) {
            this.awsManagedRulesAntiDDoSRuleSet = awsManagedRulesAntiDDoSRuleSet != null ? awsManagedRulesAntiDDoSRuleSet.build()
                    : null;
        }

        @Override
        public final Builder awsManagedRulesAntiDDoSRuleSet(AWSManagedRulesAntiDDoSRuleSet awsManagedRulesAntiDDoSRuleSet) {
            this.awsManagedRulesAntiDDoSRuleSet = awsManagedRulesAntiDDoSRuleSet;
            return this;
        }

        @Override
        public ManagedRuleGroupConfig build() {
            return new ManagedRuleGroupConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
