/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Inspect the body of the web request as JSON. The body immediately follows the request headers.
 * </p>
 * <p>
 * This is used to indicate the web request component to inspect, in the <a>FieldToMatch</a> specification.
 * </p>
 * <p>
 * Use the specifications in this object to indicate which parts of the JSON body to inspect using the rule's inspection
 * criteria. WAF inspects only the parts of the JSON that result from the matches that you indicate.
 * </p>
 * <p>
 * Example JSON: <code>"JsonBody": { "MatchPattern": { "All": {} }, "MatchScope": "ALL" }</code>
 * </p>
 * <p>
 * For additional information about this request component option, see <a href=
 * "https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-fields-list.html#waf-rule-statement-request-component-json-body"
 * >JSON body</a> in the <i>WAF Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JsonBody implements SdkPojo, Serializable, ToCopyableBuilder<JsonBody.Builder, JsonBody> {
    private static final SdkField<JsonMatchPattern> MATCH_PATTERN_FIELD = SdkField
            .<JsonMatchPattern> builder(MarshallingType.SDK_POJO).memberName("MatchPattern")
            .getter(getter(JsonBody::matchPattern)).setter(setter(Builder::matchPattern)).constructor(JsonMatchPattern::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchPattern").build()).build();

    private static final SdkField<String> MATCH_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MatchScope").getter(getter(JsonBody::matchScopeAsString)).setter(setter(Builder::matchScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchScope").build()).build();

    private static final SdkField<String> INVALID_FALLBACK_BEHAVIOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InvalidFallbackBehavior").getter(getter(JsonBody::invalidFallbackBehaviorAsString))
            .setter(setter(Builder::invalidFallbackBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InvalidFallbackBehavior").build())
            .build();

    private static final SdkField<String> OVERSIZE_HANDLING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OversizeHandling").getter(getter(JsonBody::oversizeHandlingAsString))
            .setter(setter(Builder::oversizeHandling))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OversizeHandling").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MATCH_PATTERN_FIELD,
            MATCH_SCOPE_FIELD, INVALID_FALLBACK_BEHAVIOR_FIELD, OVERSIZE_HANDLING_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final JsonMatchPattern matchPattern;

    private final String matchScope;

    private final String invalidFallbackBehavior;

    private final String oversizeHandling;

    private JsonBody(BuilderImpl builder) {
        this.matchPattern = builder.matchPattern;
        this.matchScope = builder.matchScope;
        this.invalidFallbackBehavior = builder.invalidFallbackBehavior;
        this.oversizeHandling = builder.oversizeHandling;
    }

    /**
     * <p>
     * The patterns to look for in the JSON body. WAF inspects the results of these pattern matches against the rule
     * inspection criteria.
     * </p>
     * 
     * @return The patterns to look for in the JSON body. WAF inspects the results of these pattern matches against the
     *         rule inspection criteria.
     */
    public final JsonMatchPattern matchPattern() {
        return matchPattern;
    }

    /**
     * <p>
     * The parts of the JSON to match against using the <code>MatchPattern</code>. If you specify <code>ALL</code>, WAF
     * matches against keys and values.
     * </p>
     * <p>
     * <code>All</code> does not require a match to be found in the keys and a match to be found in the values. It
     * requires a match to be found in the keys or the values or both. To require a match in the keys and in the values,
     * use a logical <code>AND</code> statement to combine two match rules, one that inspects the keys and another that
     * inspects the values.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #matchScope} will
     * return {@link JsonMatchScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #matchScopeAsString}.
     * </p>
     * 
     * @return The parts of the JSON to match against using the <code>MatchPattern</code>. If you specify
     *         <code>ALL</code>, WAF matches against keys and values. </p>
     *         <p>
     *         <code>All</code> does not require a match to be found in the keys and a match to be found in the values.
     *         It requires a match to be found in the keys or the values or both. To require a match in the keys and in
     *         the values, use a logical <code>AND</code> statement to combine two match rules, one that inspects the
     *         keys and another that inspects the values.
     * @see JsonMatchScope
     */
    public final JsonMatchScope matchScope() {
        return JsonMatchScope.fromValue(matchScope);
    }

    /**
     * <p>
     * The parts of the JSON to match against using the <code>MatchPattern</code>. If you specify <code>ALL</code>, WAF
     * matches against keys and values.
     * </p>
     * <p>
     * <code>All</code> does not require a match to be found in the keys and a match to be found in the values. It
     * requires a match to be found in the keys or the values or both. To require a match in the keys and in the values,
     * use a logical <code>AND</code> statement to combine two match rules, one that inspects the keys and another that
     * inspects the values.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #matchScope} will
     * return {@link JsonMatchScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #matchScopeAsString}.
     * </p>
     * 
     * @return The parts of the JSON to match against using the <code>MatchPattern</code>. If you specify
     *         <code>ALL</code>, WAF matches against keys and values. </p>
     *         <p>
     *         <code>All</code> does not require a match to be found in the keys and a match to be found in the values.
     *         It requires a match to be found in the keys or the values or both. To require a match in the keys and in
     *         the values, use a logical <code>AND</code> statement to combine two match rules, one that inspects the
     *         keys and another that inspects the values.
     * @see JsonMatchScope
     */
    public final String matchScopeAsString() {
        return matchScope;
    }

    /**
     * <p>
     * What WAF should do if it fails to completely parse the JSON body. The options are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EVALUATE_AS_STRING</code> - Inspect the body as plain text. WAF applies the text transformations and
     * inspection criteria that you defined for the JSON inspection to the body text string.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to the
     * request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't provide this setting, WAF parses and evaluates the content only up to the first parsing failure that
     * it encounters.
     * </p>
     * <note>
     * <p>
     * WAF parsing doesn't fully validate the input JSON string, so parsing can succeed even for invalid JSON. When
     * parsing succeeds, WAF doesn't apply the fallback behavior. For more information, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-fields-list.html#waf-rule-statement-request-component-json-body"
     * >JSON body</a> in the <i>WAF Developer Guide</i>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #invalidFallbackBehavior} will return {@link BodyParsingFallbackBehavior#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #invalidFallbackBehaviorAsString}.
     * </p>
     * 
     * @return What WAF should do if it fails to completely parse the JSON body. The options are the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EVALUATE_AS_STRING</code> - Inspect the body as plain text. WAF applies the text transformations
     *         and inspection criteria that you defined for the JSON inspection to the body text string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to
     *         the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't provide this setting, WAF parses and evaluates the content only up to the first parsing
     *         failure that it encounters.
     *         </p>
     *         <note>
     *         <p>
     *         WAF parsing doesn't fully validate the input JSON string, so parsing can succeed even for invalid JSON.
     *         When parsing succeeds, WAF doesn't apply the fallback behavior. For more information, see <a href=
     *         "https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-fields-list.html#waf-rule-statement-request-component-json-body"
     *         >JSON body</a> in the <i>WAF Developer Guide</i>.
     *         </p>
     * @see BodyParsingFallbackBehavior
     */
    public final BodyParsingFallbackBehavior invalidFallbackBehavior() {
        return BodyParsingFallbackBehavior.fromValue(invalidFallbackBehavior);
    }

    /**
     * <p>
     * What WAF should do if it fails to completely parse the JSON body. The options are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EVALUATE_AS_STRING</code> - Inspect the body as plain text. WAF applies the text transformations and
     * inspection criteria that you defined for the JSON inspection to the body text string.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to the
     * request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't provide this setting, WAF parses and evaluates the content only up to the first parsing failure that
     * it encounters.
     * </p>
     * <note>
     * <p>
     * WAF parsing doesn't fully validate the input JSON string, so parsing can succeed even for invalid JSON. When
     * parsing succeeds, WAF doesn't apply the fallback behavior. For more information, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-fields-list.html#waf-rule-statement-request-component-json-body"
     * >JSON body</a> in the <i>WAF Developer Guide</i>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #invalidFallbackBehavior} will return {@link BodyParsingFallbackBehavior#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #invalidFallbackBehaviorAsString}.
     * </p>
     * 
     * @return What WAF should do if it fails to completely parse the JSON body. The options are the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EVALUATE_AS_STRING</code> - Inspect the body as plain text. WAF applies the text transformations
     *         and inspection criteria that you defined for the JSON inspection to the body text string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to
     *         the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't provide this setting, WAF parses and evaluates the content only up to the first parsing
     *         failure that it encounters.
     *         </p>
     *         <note>
     *         <p>
     *         WAF parsing doesn't fully validate the input JSON string, so parsing can succeed even for invalid JSON.
     *         When parsing succeeds, WAF doesn't apply the fallback behavior. For more information, see <a href=
     *         "https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-fields-list.html#waf-rule-statement-request-component-json-body"
     *         >JSON body</a> in the <i>WAF Developer Guide</i>.
     *         </p>
     * @see BodyParsingFallbackBehavior
     */
    public final String invalidFallbackBehaviorAsString() {
        return invalidFallbackBehavior;
    }

    /**
     * <p>
     * What WAF should do if the body is larger than WAF can inspect.
     * </p>
     * <p>
     * WAF does not support inspecting the entire contents of the web request body if the body exceeds the limit for the
     * resource type. When a web request body is larger than the limit, the underlying host service only forwards the
     * contents that are within the limit to WAF for inspection.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
     * </p>
     * </li>
     * <li>
     * <p>
     * For CloudFront, API Gateway, Amazon Cognito, App Runner, and Verified Access, the default limit is 16 KB (16,384
     * bytes), and you can increase the limit for each resource type in the web ACL <code>AssociationConfig</code>, for
     * additional processing fees.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amplify, use the CloudFront limit.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The options for oversize handling are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CONTINUE</code> - Inspect the available body contents normally, according to the rule inspection criteria.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to the
     * request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can combine the <code>MATCH</code> or <code>NO_MATCH</code> settings for oversize handling with your rule and
     * web ACL action settings, so that you block any request whose body is over the limit.
     * </p>
     * <p>
     * Default: <code>CONTINUE</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #oversizeHandling}
     * will return {@link OversizeHandling#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #oversizeHandlingAsString}.
     * </p>
     * 
     * @return What WAF should do if the body is larger than WAF can inspect. </p>
     *         <p>
     *         WAF does not support inspecting the entire contents of the web request body if the body exceeds the limit
     *         for the resource type. When a web request body is larger than the limit, the underlying host service only
     *         forwards the contents that are within the limit to WAF for inspection.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For CloudFront, API Gateway, Amazon Cognito, App Runner, and Verified Access, the default limit is 16 KB
     *         (16,384 bytes), and you can increase the limit for each resource type in the web ACL
     *         <code>AssociationConfig</code>, for additional processing fees.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amplify, use the CloudFront limit.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The options for oversize handling are the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONTINUE</code> - Inspect the available body contents normally, according to the rule inspection
     *         criteria.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to
     *         the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can combine the <code>MATCH</code> or <code>NO_MATCH</code> settings for oversize handling with your
     *         rule and web ACL action settings, so that you block any request whose body is over the limit.
     *         </p>
     *         <p>
     *         Default: <code>CONTINUE</code>
     * @see OversizeHandling
     */
    public final OversizeHandling oversizeHandling() {
        return OversizeHandling.fromValue(oversizeHandling);
    }

    /**
     * <p>
     * What WAF should do if the body is larger than WAF can inspect.
     * </p>
     * <p>
     * WAF does not support inspecting the entire contents of the web request body if the body exceeds the limit for the
     * resource type. When a web request body is larger than the limit, the underlying host service only forwards the
     * contents that are within the limit to WAF for inspection.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
     * </p>
     * </li>
     * <li>
     * <p>
     * For CloudFront, API Gateway, Amazon Cognito, App Runner, and Verified Access, the default limit is 16 KB (16,384
     * bytes), and you can increase the limit for each resource type in the web ACL <code>AssociationConfig</code>, for
     * additional processing fees.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amplify, use the CloudFront limit.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The options for oversize handling are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CONTINUE</code> - Inspect the available body contents normally, according to the rule inspection criteria.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to the
     * request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can combine the <code>MATCH</code> or <code>NO_MATCH</code> settings for oversize handling with your rule and
     * web ACL action settings, so that you block any request whose body is over the limit.
     * </p>
     * <p>
     * Default: <code>CONTINUE</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #oversizeHandling}
     * will return {@link OversizeHandling#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #oversizeHandlingAsString}.
     * </p>
     * 
     * @return What WAF should do if the body is larger than WAF can inspect. </p>
     *         <p>
     *         WAF does not support inspecting the entire contents of the web request body if the body exceeds the limit
     *         for the resource type. When a web request body is larger than the limit, the underlying host service only
     *         forwards the contents that are within the limit to WAF for inspection.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For CloudFront, API Gateway, Amazon Cognito, App Runner, and Verified Access, the default limit is 16 KB
     *         (16,384 bytes), and you can increase the limit for each resource type in the web ACL
     *         <code>AssociationConfig</code>, for additional processing fees.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amplify, use the CloudFront limit.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The options for oversize handling are the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONTINUE</code> - Inspect the available body contents normally, according to the rule inspection
     *         criteria.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to
     *         the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can combine the <code>MATCH</code> or <code>NO_MATCH</code> settings for oversize handling with your
     *         rule and web ACL action settings, so that you block any request whose body is over the limit.
     *         </p>
     *         <p>
     *         Default: <code>CONTINUE</code>
     * @see OversizeHandling
     */
    public final String oversizeHandlingAsString() {
        return oversizeHandling;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(matchPattern());
        hashCode = 31 * hashCode + Objects.hashCode(matchScopeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(invalidFallbackBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(oversizeHandlingAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JsonBody)) {
            return false;
        }
        JsonBody other = (JsonBody) obj;
        return Objects.equals(matchPattern(), other.matchPattern())
                && Objects.equals(matchScopeAsString(), other.matchScopeAsString())
                && Objects.equals(invalidFallbackBehaviorAsString(), other.invalidFallbackBehaviorAsString())
                && Objects.equals(oversizeHandlingAsString(), other.oversizeHandlingAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JsonBody").add("MatchPattern", matchPattern()).add("MatchScope", matchScopeAsString())
                .add("InvalidFallbackBehavior", invalidFallbackBehaviorAsString())
                .add("OversizeHandling", oversizeHandlingAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MatchPattern":
            return Optional.ofNullable(clazz.cast(matchPattern()));
        case "MatchScope":
            return Optional.ofNullable(clazz.cast(matchScopeAsString()));
        case "InvalidFallbackBehavior":
            return Optional.ofNullable(clazz.cast(invalidFallbackBehaviorAsString()));
        case "OversizeHandling":
            return Optional.ofNullable(clazz.cast(oversizeHandlingAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MatchPattern", MATCH_PATTERN_FIELD);
        map.put("MatchScope", MATCH_SCOPE_FIELD);
        map.put("InvalidFallbackBehavior", INVALID_FALLBACK_BEHAVIOR_FIELD);
        map.put("OversizeHandling", OVERSIZE_HANDLING_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<JsonBody, T> g) {
        return obj -> g.apply((JsonBody) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JsonBody> {
        /**
         * <p>
         * The patterns to look for in the JSON body. WAF inspects the results of these pattern matches against the rule
         * inspection criteria.
         * </p>
         * 
         * @param matchPattern
         *        The patterns to look for in the JSON body. WAF inspects the results of these pattern matches against
         *        the rule inspection criteria.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchPattern(JsonMatchPattern matchPattern);

        /**
         * <p>
         * The patterns to look for in the JSON body. WAF inspects the results of these pattern matches against the rule
         * inspection criteria.
         * </p>
         * This is a convenience method that creates an instance of the {@link JsonMatchPattern.Builder} avoiding the
         * need to create one manually via {@link JsonMatchPattern#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JsonMatchPattern.Builder#build()} is called immediately and its
         * result is passed to {@link #matchPattern(JsonMatchPattern)}.
         * 
         * @param matchPattern
         *        a consumer that will call methods on {@link JsonMatchPattern.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #matchPattern(JsonMatchPattern)
         */
        default Builder matchPattern(Consumer<JsonMatchPattern.Builder> matchPattern) {
            return matchPattern(JsonMatchPattern.builder().applyMutation(matchPattern).build());
        }

        /**
         * <p>
         * The parts of the JSON to match against using the <code>MatchPattern</code>. If you specify <code>ALL</code>,
         * WAF matches against keys and values.
         * </p>
         * <p>
         * <code>All</code> does not require a match to be found in the keys and a match to be found in the values. It
         * requires a match to be found in the keys or the values or both. To require a match in the keys and in the
         * values, use a logical <code>AND</code> statement to combine two match rules, one that inspects the keys and
         * another that inspects the values.
         * </p>
         * 
         * @param matchScope
         *        The parts of the JSON to match against using the <code>MatchPattern</code>. If you specify
         *        <code>ALL</code>, WAF matches against keys and values. </p>
         *        <p>
         *        <code>All</code> does not require a match to be found in the keys and a match to be found in the
         *        values. It requires a match to be found in the keys or the values or both. To require a match in the
         *        keys and in the values, use a logical <code>AND</code> statement to combine two match rules, one that
         *        inspects the keys and another that inspects the values.
         * @see JsonMatchScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JsonMatchScope
         */
        Builder matchScope(String matchScope);

        /**
         * <p>
         * The parts of the JSON to match against using the <code>MatchPattern</code>. If you specify <code>ALL</code>,
         * WAF matches against keys and values.
         * </p>
         * <p>
         * <code>All</code> does not require a match to be found in the keys and a match to be found in the values. It
         * requires a match to be found in the keys or the values or both. To require a match in the keys and in the
         * values, use a logical <code>AND</code> statement to combine two match rules, one that inspects the keys and
         * another that inspects the values.
         * </p>
         * 
         * @param matchScope
         *        The parts of the JSON to match against using the <code>MatchPattern</code>. If you specify
         *        <code>ALL</code>, WAF matches against keys and values. </p>
         *        <p>
         *        <code>All</code> does not require a match to be found in the keys and a match to be found in the
         *        values. It requires a match to be found in the keys or the values or both. To require a match in the
         *        keys and in the values, use a logical <code>AND</code> statement to combine two match rules, one that
         *        inspects the keys and another that inspects the values.
         * @see JsonMatchScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JsonMatchScope
         */
        Builder matchScope(JsonMatchScope matchScope);

        /**
         * <p>
         * What WAF should do if it fails to completely parse the JSON body. The options are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EVALUATE_AS_STRING</code> - Inspect the body as plain text. WAF applies the text transformations and
         * inspection criteria that you defined for the JSON inspection to the body text string.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to the
         * request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't provide this setting, WAF parses and evaluates the content only up to the first parsing failure
         * that it encounters.
         * </p>
         * <note>
         * <p>
         * WAF parsing doesn't fully validate the input JSON string, so parsing can succeed even for invalid JSON. When
         * parsing succeeds, WAF doesn't apply the fallback behavior. For more information, see <a href=
         * "https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-fields-list.html#waf-rule-statement-request-component-json-body"
         * >JSON body</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param invalidFallbackBehavior
         *        What WAF should do if it fails to completely parse the JSON body. The options are the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EVALUATE_AS_STRING</code> - Inspect the body as plain text. WAF applies the text transformations
         *        and inspection criteria that you defined for the JSON inspection to the body text string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action
         *        to the request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't provide this setting, WAF parses and evaluates the content only up to the first parsing
         *        failure that it encounters.
         *        </p>
         *        <note>
         *        <p>
         *        WAF parsing doesn't fully validate the input JSON string, so parsing can succeed even for invalid
         *        JSON. When parsing succeeds, WAF doesn't apply the fallback behavior. For more information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-fields-list.html#waf-rule-statement-request-component-json-body"
         *        >JSON body</a> in the <i>WAF Developer Guide</i>.
         *        </p>
         * @see BodyParsingFallbackBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BodyParsingFallbackBehavior
         */
        Builder invalidFallbackBehavior(String invalidFallbackBehavior);

        /**
         * <p>
         * What WAF should do if it fails to completely parse the JSON body. The options are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EVALUATE_AS_STRING</code> - Inspect the body as plain text. WAF applies the text transformations and
         * inspection criteria that you defined for the JSON inspection to the body text string.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to the
         * request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't provide this setting, WAF parses and evaluates the content only up to the first parsing failure
         * that it encounters.
         * </p>
         * <note>
         * <p>
         * WAF parsing doesn't fully validate the input JSON string, so parsing can succeed even for invalid JSON. When
         * parsing succeeds, WAF doesn't apply the fallback behavior. For more information, see <a href=
         * "https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-fields-list.html#waf-rule-statement-request-component-json-body"
         * >JSON body</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param invalidFallbackBehavior
         *        What WAF should do if it fails to completely parse the JSON body. The options are the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EVALUATE_AS_STRING</code> - Inspect the body as plain text. WAF applies the text transformations
         *        and inspection criteria that you defined for the JSON inspection to the body text string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action
         *        to the request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't provide this setting, WAF parses and evaluates the content only up to the first parsing
         *        failure that it encounters.
         *        </p>
         *        <note>
         *        <p>
         *        WAF parsing doesn't fully validate the input JSON string, so parsing can succeed even for invalid
         *        JSON. When parsing succeeds, WAF doesn't apply the fallback behavior. For more information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-fields-list.html#waf-rule-statement-request-component-json-body"
         *        >JSON body</a> in the <i>WAF Developer Guide</i>.
         *        </p>
         * @see BodyParsingFallbackBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BodyParsingFallbackBehavior
         */
        Builder invalidFallbackBehavior(BodyParsingFallbackBehavior invalidFallbackBehavior);

        /**
         * <p>
         * What WAF should do if the body is larger than WAF can inspect.
         * </p>
         * <p>
         * WAF does not support inspecting the entire contents of the web request body if the body exceeds the limit for
         * the resource type. When a web request body is larger than the limit, the underlying host service only
         * forwards the contents that are within the limit to WAF for inspection.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
         * </p>
         * </li>
         * <li>
         * <p>
         * For CloudFront, API Gateway, Amazon Cognito, App Runner, and Verified Access, the default limit is 16 KB
         * (16,384 bytes), and you can increase the limit for each resource type in the web ACL
         * <code>AssociationConfig</code>, for additional processing fees.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amplify, use the CloudFront limit.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The options for oversize handling are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CONTINUE</code> - Inspect the available body contents normally, according to the rule inspection
         * criteria.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to the
         * request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can combine the <code>MATCH</code> or <code>NO_MATCH</code> settings for oversize handling with your rule
         * and web ACL action settings, so that you block any request whose body is over the limit.
         * </p>
         * <p>
         * Default: <code>CONTINUE</code>
         * </p>
         * 
         * @param oversizeHandling
         *        What WAF should do if the body is larger than WAF can inspect. </p>
         *        <p>
         *        WAF does not support inspecting the entire contents of the web request body if the body exceeds the
         *        limit for the resource type. When a web request body is larger than the limit, the underlying host
         *        service only forwards the contents that are within the limit to WAF for inspection.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For CloudFront, API Gateway, Amazon Cognito, App Runner, and Verified Access, the default limit is 16
         *        KB (16,384 bytes), and you can increase the limit for each resource type in the web ACL
         *        <code>AssociationConfig</code>, for additional processing fees.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amplify, use the CloudFront limit.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The options for oversize handling are the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONTINUE</code> - Inspect the available body contents normally, according to the rule inspection
         *        criteria.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action
         *        to the request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can combine the <code>MATCH</code> or <code>NO_MATCH</code> settings for oversize handling with
         *        your rule and web ACL action settings, so that you block any request whose body is over the limit.
         *        </p>
         *        <p>
         *        Default: <code>CONTINUE</code>
         * @see OversizeHandling
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OversizeHandling
         */
        Builder oversizeHandling(String oversizeHandling);

        /**
         * <p>
         * What WAF should do if the body is larger than WAF can inspect.
         * </p>
         * <p>
         * WAF does not support inspecting the entire contents of the web request body if the body exceeds the limit for
         * the resource type. When a web request body is larger than the limit, the underlying host service only
         * forwards the contents that are within the limit to WAF for inspection.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
         * </p>
         * </li>
         * <li>
         * <p>
         * For CloudFront, API Gateway, Amazon Cognito, App Runner, and Verified Access, the default limit is 16 KB
         * (16,384 bytes), and you can increase the limit for each resource type in the web ACL
         * <code>AssociationConfig</code>, for additional processing fees.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amplify, use the CloudFront limit.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The options for oversize handling are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CONTINUE</code> - Inspect the available body contents normally, according to the rule inspection
         * criteria.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action to the
         * request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can combine the <code>MATCH</code> or <code>NO_MATCH</code> settings for oversize handling with your rule
         * and web ACL action settings, so that you block any request whose body is over the limit.
         * </p>
         * <p>
         * Default: <code>CONTINUE</code>
         * </p>
         * 
         * @param oversizeHandling
         *        What WAF should do if the body is larger than WAF can inspect. </p>
         *        <p>
         *        WAF does not support inspecting the entire contents of the web request body if the body exceeds the
         *        limit for the resource type. When a web request body is larger than the limit, the underlying host
         *        service only forwards the contents that are within the limit to WAF for inspection.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For CloudFront, API Gateway, Amazon Cognito, App Runner, and Verified Access, the default limit is 16
         *        KB (16,384 bytes), and you can increase the limit for each resource type in the web ACL
         *        <code>AssociationConfig</code>, for additional processing fees.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amplify, use the CloudFront limit.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The options for oversize handling are the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONTINUE</code> - Inspect the available body contents normally, according to the rule inspection
         *        criteria.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MATCH</code> - Treat the web request as matching the rule statement. WAF applies the rule action
         *        to the request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO_MATCH</code> - Treat the web request as not matching the rule statement.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can combine the <code>MATCH</code> or <code>NO_MATCH</code> settings for oversize handling with
         *        your rule and web ACL action settings, so that you block any request whose body is over the limit.
         *        </p>
         *        <p>
         *        Default: <code>CONTINUE</code>
         * @see OversizeHandling
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OversizeHandling
         */
        Builder oversizeHandling(OversizeHandling oversizeHandling);
    }

    static final class BuilderImpl implements Builder {
        private JsonMatchPattern matchPattern;

        private String matchScope;

        private String invalidFallbackBehavior;

        private String oversizeHandling;

        private BuilderImpl() {
        }

        private BuilderImpl(JsonBody model) {
            matchPattern(model.matchPattern);
            matchScope(model.matchScope);
            invalidFallbackBehavior(model.invalidFallbackBehavior);
            oversizeHandling(model.oversizeHandling);
        }

        public final JsonMatchPattern.Builder getMatchPattern() {
            return matchPattern != null ? matchPattern.toBuilder() : null;
        }

        public final void setMatchPattern(JsonMatchPattern.BuilderImpl matchPattern) {
            this.matchPattern = matchPattern != null ? matchPattern.build() : null;
        }

        @Override
        public final Builder matchPattern(JsonMatchPattern matchPattern) {
            this.matchPattern = matchPattern;
            return this;
        }

        public final String getMatchScope() {
            return matchScope;
        }

        public final void setMatchScope(String matchScope) {
            this.matchScope = matchScope;
        }

        @Override
        public final Builder matchScope(String matchScope) {
            this.matchScope = matchScope;
            return this;
        }

        @Override
        public final Builder matchScope(JsonMatchScope matchScope) {
            this.matchScope(matchScope == null ? null : matchScope.toString());
            return this;
        }

        public final String getInvalidFallbackBehavior() {
            return invalidFallbackBehavior;
        }

        public final void setInvalidFallbackBehavior(String invalidFallbackBehavior) {
            this.invalidFallbackBehavior = invalidFallbackBehavior;
        }

        @Override
        public final Builder invalidFallbackBehavior(String invalidFallbackBehavior) {
            this.invalidFallbackBehavior = invalidFallbackBehavior;
            return this;
        }

        @Override
        public final Builder invalidFallbackBehavior(BodyParsingFallbackBehavior invalidFallbackBehavior) {
            this.invalidFallbackBehavior(invalidFallbackBehavior == null ? null : invalidFallbackBehavior.toString());
            return this;
        }

        public final String getOversizeHandling() {
            return oversizeHandling;
        }

        public final void setOversizeHandling(String oversizeHandling) {
            this.oversizeHandling = oversizeHandling;
        }

        @Override
        public final Builder oversizeHandling(String oversizeHandling) {
            this.oversizeHandling = oversizeHandling;
            return this;
        }

        @Override
        public final Builder oversizeHandling(OversizeHandling oversizeHandling) {
            this.oversizeHandling(oversizeHandling == null ? null : oversizeHandling.toString());
            return this;
        }

        @Override
        public JsonBody build() {
            return new JsonBody(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
