/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.voiceid.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains all the information about a fraudster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Fraudster implements SdkPojo, Serializable, ToCopyableBuilder<Fraudster.Builder, Fraudster> {
    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(Fraudster::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<String> DOMAIN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainId").getter(getter(Fraudster::domainId)).setter(setter(Builder::domainId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainId").build()).build();

    private static final SdkField<String> GENERATED_FRAUDSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GeneratedFraudsterId").getter(getter(Fraudster::generatedFraudsterId))
            .setter(setter(Builder::generatedFraudsterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GeneratedFraudsterId").build())
            .build();

    private static final SdkField<List<String>> WATCHLIST_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("WatchlistIds")
            .getter(getter(Fraudster::watchlistIds))
            .setter(setter(Builder::watchlistIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WatchlistIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATED_AT_FIELD,
            DOMAIN_ID_FIELD, GENERATED_FRAUDSTER_ID_FIELD, WATCHLIST_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant createdAt;

    private final String domainId;

    private final String generatedFraudsterId;

    private final List<String> watchlistIds;

    private Fraudster(BuilderImpl builder) {
        this.createdAt = builder.createdAt;
        this.domainId = builder.domainId;
        this.generatedFraudsterId = builder.generatedFraudsterId;
        this.watchlistIds = builder.watchlistIds;
    }

    /**
     * <p>
     * The timestamp of when Voice ID identified the fraudster.
     * </p>
     * 
     * @return The timestamp of when Voice ID identified the fraudster.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The identifier of the domain that contains the fraudster.
     * </p>
     * 
     * @return The identifier of the domain that contains the fraudster.
     */
    public final String domainId() {
        return domainId;
    }

    /**
     * <p>
     * The service-generated identifier for the fraudster.
     * </p>
     * 
     * @return The service-generated identifier for the fraudster.
     */
    public final String generatedFraudsterId() {
        return generatedFraudsterId;
    }

    /**
     * For responses, this returns true if the service returned a value for the WatchlistIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasWatchlistIds() {
        return watchlistIds != null && !(watchlistIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifier of the watchlists the fraudster is a part of.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWatchlistIds} method.
     * </p>
     * 
     * @return The identifier of the watchlists the fraudster is a part of.
     */
    public final List<String> watchlistIds() {
        return watchlistIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(domainId());
        hashCode = 31 * hashCode + Objects.hashCode(generatedFraudsterId());
        hashCode = 31 * hashCode + Objects.hashCode(hasWatchlistIds() ? watchlistIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Fraudster)) {
            return false;
        }
        Fraudster other = (Fraudster) obj;
        return Objects.equals(createdAt(), other.createdAt()) && Objects.equals(domainId(), other.domainId())
                && Objects.equals(generatedFraudsterId(), other.generatedFraudsterId())
                && hasWatchlistIds() == other.hasWatchlistIds() && Objects.equals(watchlistIds(), other.watchlistIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Fraudster").add("CreatedAt", createdAt()).add("DomainId", domainId())
                .add("GeneratedFraudsterId", generatedFraudsterId())
                .add("WatchlistIds", hasWatchlistIds() ? watchlistIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "DomainId":
            return Optional.ofNullable(clazz.cast(domainId()));
        case "GeneratedFraudsterId":
            return Optional.ofNullable(clazz.cast(generatedFraudsterId()));
        case "WatchlistIds":
            return Optional.ofNullable(clazz.cast(watchlistIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CreatedAt", CREATED_AT_FIELD);
        map.put("DomainId", DOMAIN_ID_FIELD);
        map.put("GeneratedFraudsterId", GENERATED_FRAUDSTER_ID_FIELD);
        map.put("WatchlistIds", WATCHLIST_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Fraudster, T> g) {
        return obj -> g.apply((Fraudster) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Fraudster> {
        /**
         * <p>
         * The timestamp of when Voice ID identified the fraudster.
         * </p>
         * 
         * @param createdAt
         *        The timestamp of when Voice ID identified the fraudster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The identifier of the domain that contains the fraudster.
         * </p>
         * 
         * @param domainId
         *        The identifier of the domain that contains the fraudster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainId(String domainId);

        /**
         * <p>
         * The service-generated identifier for the fraudster.
         * </p>
         * 
         * @param generatedFraudsterId
         *        The service-generated identifier for the fraudster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder generatedFraudsterId(String generatedFraudsterId);

        /**
         * <p>
         * The identifier of the watchlists the fraudster is a part of.
         * </p>
         * 
         * @param watchlistIds
         *        The identifier of the watchlists the fraudster is a part of.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder watchlistIds(Collection<String> watchlistIds);

        /**
         * <p>
         * The identifier of the watchlists the fraudster is a part of.
         * </p>
         * 
         * @param watchlistIds
         *        The identifier of the watchlists the fraudster is a part of.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder watchlistIds(String... watchlistIds);
    }

    static final class BuilderImpl implements Builder {
        private Instant createdAt;

        private String domainId;

        private String generatedFraudsterId;

        private List<String> watchlistIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Fraudster model) {
            createdAt(model.createdAt);
            domainId(model.domainId);
            generatedFraudsterId(model.generatedFraudsterId);
            watchlistIds(model.watchlistIds);
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getDomainId() {
            return domainId;
        }

        public final void setDomainId(String domainId) {
            this.domainId = domainId;
        }

        @Override
        public final Builder domainId(String domainId) {
            this.domainId = domainId;
            return this;
        }

        public final String getGeneratedFraudsterId() {
            return generatedFraudsterId;
        }

        public final void setGeneratedFraudsterId(String generatedFraudsterId) {
            this.generatedFraudsterId = generatedFraudsterId;
        }

        @Override
        public final Builder generatedFraudsterId(String generatedFraudsterId) {
            this.generatedFraudsterId = generatedFraudsterId;
            return this;
        }

        public final Collection<String> getWatchlistIds() {
            if (watchlistIds instanceof SdkAutoConstructList) {
                return null;
            }
            return watchlistIds;
        }

        public final void setWatchlistIds(Collection<String> watchlistIds) {
            this.watchlistIds = ResponseWatchlistIdsCopier.copy(watchlistIds);
        }

        @Override
        public final Builder watchlistIds(Collection<String> watchlistIds) {
            this.watchlistIds = ResponseWatchlistIdsCopier.copy(watchlistIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder watchlistIds(String... watchlistIds) {
            watchlistIds(Arrays.asList(watchlistIds));
            return this;
        }

        @Override
        public Fraudster build() {
            return new Fraudster(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
