/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the properties of a security policy that you specify. For more information about security policies, see <a
 * href="https://docs.aws.amazon.com/transfer/latest/userguide/security-policies.html">Working with security policies
 * for servers</a> or <a
 * href="https://docs.aws.amazon.com/transfer/latest/userguide/security-policies-connectors.html">Working with security
 * policies for SFTP connectors</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribedSecurityPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<DescribedSecurityPolicy.Builder, DescribedSecurityPolicy> {
    private static final SdkField<Boolean> FIPS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN).memberName("Fips")
            .getter(getter(DescribedSecurityPolicy::fips)).setter(setter(Builder::fips))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Fips").build()).build();

    private static final SdkField<String> SECURITY_POLICY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityPolicyName").getter(getter(DescribedSecurityPolicy::securityPolicyName))
            .setter(setter(Builder::securityPolicyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityPolicyName").build())
            .build();

    private static final SdkField<List<String>> SSH_CIPHERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SshCiphers")
            .getter(getter(DescribedSecurityPolicy::sshCiphers))
            .setter(setter(Builder::sshCiphers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SshCiphers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SSH_KEXS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SshKexs")
            .getter(getter(DescribedSecurityPolicy::sshKexs))
            .setter(setter(Builder::sshKexs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SshKexs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SSH_MACS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SshMacs")
            .getter(getter(DescribedSecurityPolicy::sshMacs))
            .setter(setter(Builder::sshMacs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SshMacs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> TLS_CIPHERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TlsCiphers")
            .getter(getter(DescribedSecurityPolicy::tlsCiphers))
            .setter(setter(Builder::tlsCiphers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TlsCiphers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SSH_HOST_KEY_ALGORITHMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SshHostKeyAlgorithms")
            .getter(getter(DescribedSecurityPolicy::sshHostKeyAlgorithms))
            .setter(setter(Builder::sshHostKeyAlgorithms))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SshHostKeyAlgorithms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DescribedSecurityPolicy::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<String>> PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Protocols")
            .getter(getter(DescribedSecurityPolicy::protocolsAsStrings))
            .setter(setter(Builder::protocolsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FIPS_FIELD,
            SECURITY_POLICY_NAME_FIELD, SSH_CIPHERS_FIELD, SSH_KEXS_FIELD, SSH_MACS_FIELD, TLS_CIPHERS_FIELD,
            SSH_HOST_KEY_ALGORITHMS_FIELD, TYPE_FIELD, PROTOCOLS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean fips;

    private final String securityPolicyName;

    private final List<String> sshCiphers;

    private final List<String> sshKexs;

    private final List<String> sshMacs;

    private final List<String> tlsCiphers;

    private final List<String> sshHostKeyAlgorithms;

    private final String type;

    private final List<String> protocols;

    private DescribedSecurityPolicy(BuilderImpl builder) {
        this.fips = builder.fips;
        this.securityPolicyName = builder.securityPolicyName;
        this.sshCiphers = builder.sshCiphers;
        this.sshKexs = builder.sshKexs;
        this.sshMacs = builder.sshMacs;
        this.tlsCiphers = builder.tlsCiphers;
        this.sshHostKeyAlgorithms = builder.sshHostKeyAlgorithms;
        this.type = builder.type;
        this.protocols = builder.protocols;
    }

    /**
     * <p>
     * Specifies whether this policy enables Federal Information Processing Standards (FIPS). This parameter applies to
     * both server and connector security policies.
     * </p>
     * 
     * @return Specifies whether this policy enables Federal Information Processing Standards (FIPS). This parameter
     *         applies to both server and connector security policies.
     */
    public final Boolean fips() {
        return fips;
    }

    /**
     * <p>
     * The text name of the specified security policy.
     * </p>
     * 
     * @return The text name of the specified security policy.
     */
    public final String securityPolicyName() {
        return securityPolicyName;
    }

    /**
     * For responses, this returns true if the service returned a value for the SshCiphers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSshCiphers() {
        return sshCiphers != null && !(sshCiphers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Lists the enabled Secure Shell (SSH) cipher encryption algorithms in the security policy that is attached to the
     * server or connector. This parameter applies to both server and connector security policies.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSshCiphers} method.
     * </p>
     * 
     * @return Lists the enabled Secure Shell (SSH) cipher encryption algorithms in the security policy that is attached
     *         to the server or connector. This parameter applies to both server and connector security policies.
     */
    public final List<String> sshCiphers() {
        return sshCiphers;
    }

    /**
     * For responses, this returns true if the service returned a value for the SshKexs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSshKexs() {
        return sshKexs != null && !(sshKexs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Lists the enabled SSH key exchange (KEX) encryption algorithms in the security policy that is attached to the
     * server or connector. This parameter applies to both server and connector security policies.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSshKexs} method.
     * </p>
     * 
     * @return Lists the enabled SSH key exchange (KEX) encryption algorithms in the security policy that is attached to
     *         the server or connector. This parameter applies to both server and connector security policies.
     */
    public final List<String> sshKexs() {
        return sshKexs;
    }

    /**
     * For responses, this returns true if the service returned a value for the SshMacs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSshMacs() {
        return sshMacs != null && !(sshMacs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Lists the enabled SSH message authentication code (MAC) encryption algorithms in the security policy that is
     * attached to the server or connector. This parameter applies to both server and connector security policies.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSshMacs} method.
     * </p>
     * 
     * @return Lists the enabled SSH message authentication code (MAC) encryption algorithms in the security policy that
     *         is attached to the server or connector. This parameter applies to both server and connector security
     *         policies.
     */
    public final List<String> sshMacs() {
        return sshMacs;
    }

    /**
     * For responses, this returns true if the service returned a value for the TlsCiphers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTlsCiphers() {
        return tlsCiphers != null && !(tlsCiphers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Lists the enabled Transport Layer Security (TLS) cipher encryption algorithms in the security policy that is
     * attached to the server.
     * </p>
     * <note>
     * <p>
     * This parameter only applies to security policies for servers.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTlsCiphers} method.
     * </p>
     * 
     * @return Lists the enabled Transport Layer Security (TLS) cipher encryption algorithms in the security policy that
     *         is attached to the server.</p> <note>
     *         <p>
     *         This parameter only applies to security policies for servers.
     *         </p>
     */
    public final List<String> tlsCiphers() {
        return tlsCiphers;
    }

    /**
     * For responses, this returns true if the service returned a value for the SshHostKeyAlgorithms property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSshHostKeyAlgorithms() {
        return sshHostKeyAlgorithms != null && !(sshHostKeyAlgorithms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Lists the host key algorithms for the security policy.
     * </p>
     * <note>
     * <p>
     * This parameter only applies to security policies for connectors.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSshHostKeyAlgorithms} method.
     * </p>
     * 
     * @return Lists the host key algorithms for the security policy.</p> <note>
     *         <p>
     *         This parameter only applies to security policies for connectors.
     *         </p>
     */
    public final List<String> sshHostKeyAlgorithms() {
        return sshHostKeyAlgorithms;
    }

    /**
     * <p>
     * The resource type to which the security policy applies, either server or connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SecurityPolicyResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The resource type to which the security policy applies, either server or connector.
     * @see SecurityPolicyResourceType
     */
    public final SecurityPolicyResourceType type() {
        return SecurityPolicyResourceType.fromValue(type);
    }

    /**
     * <p>
     * The resource type to which the security policy applies, either server or connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SecurityPolicyResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The resource type to which the security policy applies, either server or connector.
     * @see SecurityPolicyResourceType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Lists the file transfer protocols that the security policy applies to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return Lists the file transfer protocols that the security policy applies to.
     */
    public final List<SecurityPolicyProtocol> protocols() {
        return SecurityPolicyProtocolsCopier.copyStringToEnum(protocols);
    }

    /**
     * For responses, this returns true if the service returned a value for the Protocols property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProtocols() {
        return protocols != null && !(protocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Lists the file transfer protocols that the security policy applies to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return Lists the file transfer protocols that the security policy applies to.
     */
    public final List<String> protocolsAsStrings() {
        return protocols;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fips());
        hashCode = 31 * hashCode + Objects.hashCode(securityPolicyName());
        hashCode = 31 * hashCode + Objects.hashCode(hasSshCiphers() ? sshCiphers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSshKexs() ? sshKexs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSshMacs() ? sshMacs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTlsCiphers() ? tlsCiphers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSshHostKeyAlgorithms() ? sshHostKeyAlgorithms() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasProtocols() ? protocolsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribedSecurityPolicy)) {
            return false;
        }
        DescribedSecurityPolicy other = (DescribedSecurityPolicy) obj;
        return Objects.equals(fips(), other.fips()) && Objects.equals(securityPolicyName(), other.securityPolicyName())
                && hasSshCiphers() == other.hasSshCiphers() && Objects.equals(sshCiphers(), other.sshCiphers())
                && hasSshKexs() == other.hasSshKexs() && Objects.equals(sshKexs(), other.sshKexs())
                && hasSshMacs() == other.hasSshMacs() && Objects.equals(sshMacs(), other.sshMacs())
                && hasTlsCiphers() == other.hasTlsCiphers() && Objects.equals(tlsCiphers(), other.tlsCiphers())
                && hasSshHostKeyAlgorithms() == other.hasSshHostKeyAlgorithms()
                && Objects.equals(sshHostKeyAlgorithms(), other.sshHostKeyAlgorithms())
                && Objects.equals(typeAsString(), other.typeAsString()) && hasProtocols() == other.hasProtocols()
                && Objects.equals(protocolsAsStrings(), other.protocolsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribedSecurityPolicy").add("Fips", fips()).add("SecurityPolicyName", securityPolicyName())
                .add("SshCiphers", hasSshCiphers() ? sshCiphers() : null).add("SshKexs", hasSshKexs() ? sshKexs() : null)
                .add("SshMacs", hasSshMacs() ? sshMacs() : null).add("TlsCiphers", hasTlsCiphers() ? tlsCiphers() : null)
                .add("SshHostKeyAlgorithms", hasSshHostKeyAlgorithms() ? sshHostKeyAlgorithms() : null)
                .add("Type", typeAsString()).add("Protocols", hasProtocols() ? protocolsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Fips":
            return Optional.ofNullable(clazz.cast(fips()));
        case "SecurityPolicyName":
            return Optional.ofNullable(clazz.cast(securityPolicyName()));
        case "SshCiphers":
            return Optional.ofNullable(clazz.cast(sshCiphers()));
        case "SshKexs":
            return Optional.ofNullable(clazz.cast(sshKexs()));
        case "SshMacs":
            return Optional.ofNullable(clazz.cast(sshMacs()));
        case "TlsCiphers":
            return Optional.ofNullable(clazz.cast(tlsCiphers()));
        case "SshHostKeyAlgorithms":
            return Optional.ofNullable(clazz.cast(sshHostKeyAlgorithms()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Protocols":
            return Optional.ofNullable(clazz.cast(protocolsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Fips", FIPS_FIELD);
        map.put("SecurityPolicyName", SECURITY_POLICY_NAME_FIELD);
        map.put("SshCiphers", SSH_CIPHERS_FIELD);
        map.put("SshKexs", SSH_KEXS_FIELD);
        map.put("SshMacs", SSH_MACS_FIELD);
        map.put("TlsCiphers", TLS_CIPHERS_FIELD);
        map.put("SshHostKeyAlgorithms", SSH_HOST_KEY_ALGORITHMS_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("Protocols", PROTOCOLS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DescribedSecurityPolicy, T> g) {
        return obj -> g.apply((DescribedSecurityPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DescribedSecurityPolicy> {
        /**
         * <p>
         * Specifies whether this policy enables Federal Information Processing Standards (FIPS). This parameter applies
         * to both server and connector security policies.
         * </p>
         * 
         * @param fips
         *        Specifies whether this policy enables Federal Information Processing Standards (FIPS). This parameter
         *        applies to both server and connector security policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fips(Boolean fips);

        /**
         * <p>
         * The text name of the specified security policy.
         * </p>
         * 
         * @param securityPolicyName
         *        The text name of the specified security policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityPolicyName(String securityPolicyName);

        /**
         * <p>
         * Lists the enabled Secure Shell (SSH) cipher encryption algorithms in the security policy that is attached to
         * the server or connector. This parameter applies to both server and connector security policies.
         * </p>
         * 
         * @param sshCiphers
         *        Lists the enabled Secure Shell (SSH) cipher encryption algorithms in the security policy that is
         *        attached to the server or connector. This parameter applies to both server and connector security
         *        policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshCiphers(Collection<String> sshCiphers);

        /**
         * <p>
         * Lists the enabled Secure Shell (SSH) cipher encryption algorithms in the security policy that is attached to
         * the server or connector. This parameter applies to both server and connector security policies.
         * </p>
         * 
         * @param sshCiphers
         *        Lists the enabled Secure Shell (SSH) cipher encryption algorithms in the security policy that is
         *        attached to the server or connector. This parameter applies to both server and connector security
         *        policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshCiphers(String... sshCiphers);

        /**
         * <p>
         * Lists the enabled SSH key exchange (KEX) encryption algorithms in the security policy that is attached to the
         * server or connector. This parameter applies to both server and connector security policies.
         * </p>
         * 
         * @param sshKexs
         *        Lists the enabled SSH key exchange (KEX) encryption algorithms in the security policy that is attached
         *        to the server or connector. This parameter applies to both server and connector security policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshKexs(Collection<String> sshKexs);

        /**
         * <p>
         * Lists the enabled SSH key exchange (KEX) encryption algorithms in the security policy that is attached to the
         * server or connector. This parameter applies to both server and connector security policies.
         * </p>
         * 
         * @param sshKexs
         *        Lists the enabled SSH key exchange (KEX) encryption algorithms in the security policy that is attached
         *        to the server or connector. This parameter applies to both server and connector security policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshKexs(String... sshKexs);

        /**
         * <p>
         * Lists the enabled SSH message authentication code (MAC) encryption algorithms in the security policy that is
         * attached to the server or connector. This parameter applies to both server and connector security policies.
         * </p>
         * 
         * @param sshMacs
         *        Lists the enabled SSH message authentication code (MAC) encryption algorithms in the security policy
         *        that is attached to the server or connector. This parameter applies to both server and connector
         *        security policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshMacs(Collection<String> sshMacs);

        /**
         * <p>
         * Lists the enabled SSH message authentication code (MAC) encryption algorithms in the security policy that is
         * attached to the server or connector. This parameter applies to both server and connector security policies.
         * </p>
         * 
         * @param sshMacs
         *        Lists the enabled SSH message authentication code (MAC) encryption algorithms in the security policy
         *        that is attached to the server or connector. This parameter applies to both server and connector
         *        security policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshMacs(String... sshMacs);

        /**
         * <p>
         * Lists the enabled Transport Layer Security (TLS) cipher encryption algorithms in the security policy that is
         * attached to the server.
         * </p>
         * <note>
         * <p>
         * This parameter only applies to security policies for servers.
         * </p>
         * </note>
         * 
         * @param tlsCiphers
         *        Lists the enabled Transport Layer Security (TLS) cipher encryption algorithms in the security policy
         *        that is attached to the server.</p> <note>
         *        <p>
         *        This parameter only applies to security policies for servers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tlsCiphers(Collection<String> tlsCiphers);

        /**
         * <p>
         * Lists the enabled Transport Layer Security (TLS) cipher encryption algorithms in the security policy that is
         * attached to the server.
         * </p>
         * <note>
         * <p>
         * This parameter only applies to security policies for servers.
         * </p>
         * </note>
         * 
         * @param tlsCiphers
         *        Lists the enabled Transport Layer Security (TLS) cipher encryption algorithms in the security policy
         *        that is attached to the server.</p> <note>
         *        <p>
         *        This parameter only applies to security policies for servers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tlsCiphers(String... tlsCiphers);

        /**
         * <p>
         * Lists the host key algorithms for the security policy.
         * </p>
         * <note>
         * <p>
         * This parameter only applies to security policies for connectors.
         * </p>
         * </note>
         * 
         * @param sshHostKeyAlgorithms
         *        Lists the host key algorithms for the security policy.</p> <note>
         *        <p>
         *        This parameter only applies to security policies for connectors.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshHostKeyAlgorithms(Collection<String> sshHostKeyAlgorithms);

        /**
         * <p>
         * Lists the host key algorithms for the security policy.
         * </p>
         * <note>
         * <p>
         * This parameter only applies to security policies for connectors.
         * </p>
         * </note>
         * 
         * @param sshHostKeyAlgorithms
         *        Lists the host key algorithms for the security policy.</p> <note>
         *        <p>
         *        This parameter only applies to security policies for connectors.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshHostKeyAlgorithms(String... sshHostKeyAlgorithms);

        /**
         * <p>
         * The resource type to which the security policy applies, either server or connector.
         * </p>
         * 
         * @param type
         *        The resource type to which the security policy applies, either server or connector.
         * @see SecurityPolicyResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SecurityPolicyResourceType
         */
        Builder type(String type);

        /**
         * <p>
         * The resource type to which the security policy applies, either server or connector.
         * </p>
         * 
         * @param type
         *        The resource type to which the security policy applies, either server or connector.
         * @see SecurityPolicyResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SecurityPolicyResourceType
         */
        Builder type(SecurityPolicyResourceType type);

        /**
         * <p>
         * Lists the file transfer protocols that the security policy applies to.
         * </p>
         * 
         * @param protocols
         *        Lists the file transfer protocols that the security policy applies to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(Collection<String> protocols);

        /**
         * <p>
         * Lists the file transfer protocols that the security policy applies to.
         * </p>
         * 
         * @param protocols
         *        Lists the file transfer protocols that the security policy applies to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(String... protocols);

        /**
         * <p>
         * Lists the file transfer protocols that the security policy applies to.
         * </p>
         * 
         * @param protocols
         *        Lists the file transfer protocols that the security policy applies to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Collection<SecurityPolicyProtocol> protocols);

        /**
         * <p>
         * Lists the file transfer protocols that the security policy applies to.
         * </p>
         * 
         * @param protocols
         *        Lists the file transfer protocols that the security policy applies to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(SecurityPolicyProtocol... protocols);
    }

    static final class BuilderImpl implements Builder {
        private Boolean fips;

        private String securityPolicyName;

        private List<String> sshCiphers = DefaultSdkAutoConstructList.getInstance();

        private List<String> sshKexs = DefaultSdkAutoConstructList.getInstance();

        private List<String> sshMacs = DefaultSdkAutoConstructList.getInstance();

        private List<String> tlsCiphers = DefaultSdkAutoConstructList.getInstance();

        private List<String> sshHostKeyAlgorithms = DefaultSdkAutoConstructList.getInstance();

        private String type;

        private List<String> protocols = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DescribedSecurityPolicy model) {
            fips(model.fips);
            securityPolicyName(model.securityPolicyName);
            sshCiphers(model.sshCiphers);
            sshKexs(model.sshKexs);
            sshMacs(model.sshMacs);
            tlsCiphers(model.tlsCiphers);
            sshHostKeyAlgorithms(model.sshHostKeyAlgorithms);
            type(model.type);
            protocolsWithStrings(model.protocols);
        }

        public final Boolean getFips() {
            return fips;
        }

        public final void setFips(Boolean fips) {
            this.fips = fips;
        }

        @Override
        public final Builder fips(Boolean fips) {
            this.fips = fips;
            return this;
        }

        public final String getSecurityPolicyName() {
            return securityPolicyName;
        }

        public final void setSecurityPolicyName(String securityPolicyName) {
            this.securityPolicyName = securityPolicyName;
        }

        @Override
        public final Builder securityPolicyName(String securityPolicyName) {
            this.securityPolicyName = securityPolicyName;
            return this;
        }

        public final Collection<String> getSshCiphers() {
            if (sshCiphers instanceof SdkAutoConstructList) {
                return null;
            }
            return sshCiphers;
        }

        public final void setSshCiphers(Collection<String> sshCiphers) {
            this.sshCiphers = SecurityPolicyOptionsCopier.copy(sshCiphers);
        }

        @Override
        public final Builder sshCiphers(Collection<String> sshCiphers) {
            this.sshCiphers = SecurityPolicyOptionsCopier.copy(sshCiphers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sshCiphers(String... sshCiphers) {
            sshCiphers(Arrays.asList(sshCiphers));
            return this;
        }

        public final Collection<String> getSshKexs() {
            if (sshKexs instanceof SdkAutoConstructList) {
                return null;
            }
            return sshKexs;
        }

        public final void setSshKexs(Collection<String> sshKexs) {
            this.sshKexs = SecurityPolicyOptionsCopier.copy(sshKexs);
        }

        @Override
        public final Builder sshKexs(Collection<String> sshKexs) {
            this.sshKexs = SecurityPolicyOptionsCopier.copy(sshKexs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sshKexs(String... sshKexs) {
            sshKexs(Arrays.asList(sshKexs));
            return this;
        }

        public final Collection<String> getSshMacs() {
            if (sshMacs instanceof SdkAutoConstructList) {
                return null;
            }
            return sshMacs;
        }

        public final void setSshMacs(Collection<String> sshMacs) {
            this.sshMacs = SecurityPolicyOptionsCopier.copy(sshMacs);
        }

        @Override
        public final Builder sshMacs(Collection<String> sshMacs) {
            this.sshMacs = SecurityPolicyOptionsCopier.copy(sshMacs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sshMacs(String... sshMacs) {
            sshMacs(Arrays.asList(sshMacs));
            return this;
        }

        public final Collection<String> getTlsCiphers() {
            if (tlsCiphers instanceof SdkAutoConstructList) {
                return null;
            }
            return tlsCiphers;
        }

        public final void setTlsCiphers(Collection<String> tlsCiphers) {
            this.tlsCiphers = SecurityPolicyOptionsCopier.copy(tlsCiphers);
        }

        @Override
        public final Builder tlsCiphers(Collection<String> tlsCiphers) {
            this.tlsCiphers = SecurityPolicyOptionsCopier.copy(tlsCiphers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tlsCiphers(String... tlsCiphers) {
            tlsCiphers(Arrays.asList(tlsCiphers));
            return this;
        }

        public final Collection<String> getSshHostKeyAlgorithms() {
            if (sshHostKeyAlgorithms instanceof SdkAutoConstructList) {
                return null;
            }
            return sshHostKeyAlgorithms;
        }

        public final void setSshHostKeyAlgorithms(Collection<String> sshHostKeyAlgorithms) {
            this.sshHostKeyAlgorithms = SecurityPolicyOptionsCopier.copy(sshHostKeyAlgorithms);
        }

        @Override
        public final Builder sshHostKeyAlgorithms(Collection<String> sshHostKeyAlgorithms) {
            this.sshHostKeyAlgorithms = SecurityPolicyOptionsCopier.copy(sshHostKeyAlgorithms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sshHostKeyAlgorithms(String... sshHostKeyAlgorithms) {
            sshHostKeyAlgorithms(Arrays.asList(sshHostKeyAlgorithms));
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(SecurityPolicyResourceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Collection<String> getProtocols() {
            if (protocols instanceof SdkAutoConstructList) {
                return null;
            }
            return protocols;
        }

        public final void setProtocols(Collection<String> protocols) {
            this.protocols = SecurityPolicyProtocolsCopier.copy(protocols);
        }

        @Override
        public final Builder protocolsWithStrings(Collection<String> protocols) {
            this.protocols = SecurityPolicyProtocolsCopier.copy(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocolsWithStrings(String... protocols) {
            protocolsWithStrings(Arrays.asList(protocols));
            return this;
        }

        @Override
        public final Builder protocols(Collection<SecurityPolicyProtocol> protocols) {
            this.protocols = SecurityPolicyProtocolsCopier.copyEnumToString(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocols(SecurityPolicyProtocol... protocols) {
            protocols(Arrays.asList(protocols));
            return this;
        }

        @Override
        public DescribedSecurityPolicy build() {
            return new DescribedSecurityPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
