/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmincidents.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The record of the incident that's created when an incident occurs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IncidentRecord implements SdkPojo, Serializable, ToCopyableBuilder<IncidentRecord.Builder, IncidentRecord> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(IncidentRecord::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<List<AutomationExecution>> AUTOMATION_EXECUTIONS_FIELD = SdkField
            .<List<AutomationExecution>> builder(MarshallingType.LIST)
            .memberName("automationExecutions")
            .getter(getter(IncidentRecord::automationExecutions))
            .setter(setter(Builder::automationExecutions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("automationExecutions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AutomationExecution> builder(MarshallingType.SDK_POJO)
                                            .constructor(AutomationExecution::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ChatChannel> CHAT_CHANNEL_FIELD = SdkField.<ChatChannel> builder(MarshallingType.SDK_POJO)
            .memberName("chatChannel").getter(getter(IncidentRecord::chatChannel)).setter(setter(Builder::chatChannel))
            .constructor(ChatChannel::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("chatChannel").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime").getter(getter(IncidentRecord::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build()).build();

    private static final SdkField<String> DEDUPE_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dedupeString").getter(getter(IncidentRecord::dedupeString)).setter(setter(Builder::dedupeString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dedupeString").build()).build();

    private static final SdkField<Integer> IMPACT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("impact").getter(getter(IncidentRecord::impact)).setter(setter(Builder::impact))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("impact").build()).build();

    private static final SdkField<IncidentRecordSource> INCIDENT_RECORD_SOURCE_FIELD = SdkField
            .<IncidentRecordSource> builder(MarshallingType.SDK_POJO).memberName("incidentRecordSource")
            .getter(getter(IncidentRecord::incidentRecordSource)).setter(setter(Builder::incidentRecordSource))
            .constructor(IncidentRecordSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("incidentRecordSource").build())
            .build();

    private static final SdkField<String> LAST_MODIFIED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("lastModifiedBy").getter(getter(IncidentRecord::lastModifiedBy)).setter(setter(Builder::lastModifiedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastModifiedBy").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastModifiedTime").getter(getter(IncidentRecord::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastModifiedTime").build()).build();

    private static final SdkField<List<NotificationTargetItem>> NOTIFICATION_TARGETS_FIELD = SdkField
            .<List<NotificationTargetItem>> builder(MarshallingType.LIST)
            .memberName("notificationTargets")
            .getter(getter(IncidentRecord::notificationTargets))
            .setter(setter(Builder::notificationTargets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notificationTargets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NotificationTargetItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(NotificationTargetItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> RESOLVED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("resolvedTime").getter(getter(IncidentRecord::resolvedTime)).setter(setter(Builder::resolvedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resolvedTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(IncidentRecord::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("summary")
            .getter(getter(IncidentRecord::summary)).setter(setter(Builder::summary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("summary").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("title")
            .getter(getter(IncidentRecord::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("title").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD,
            AUTOMATION_EXECUTIONS_FIELD, CHAT_CHANNEL_FIELD, CREATION_TIME_FIELD, DEDUPE_STRING_FIELD, IMPACT_FIELD,
            INCIDENT_RECORD_SOURCE_FIELD, LAST_MODIFIED_BY_FIELD, LAST_MODIFIED_TIME_FIELD, NOTIFICATION_TARGETS_FIELD,
            RESOLVED_TIME_FIELD, STATUS_FIELD, SUMMARY_FIELD, TITLE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final List<AutomationExecution> automationExecutions;

    private final ChatChannel chatChannel;

    private final Instant creationTime;

    private final String dedupeString;

    private final Integer impact;

    private final IncidentRecordSource incidentRecordSource;

    private final String lastModifiedBy;

    private final Instant lastModifiedTime;

    private final List<NotificationTargetItem> notificationTargets;

    private final Instant resolvedTime;

    private final String status;

    private final String summary;

    private final String title;

    private IncidentRecord(BuilderImpl builder) {
        this.arn = builder.arn;
        this.automationExecutions = builder.automationExecutions;
        this.chatChannel = builder.chatChannel;
        this.creationTime = builder.creationTime;
        this.dedupeString = builder.dedupeString;
        this.impact = builder.impact;
        this.incidentRecordSource = builder.incidentRecordSource;
        this.lastModifiedBy = builder.lastModifiedBy;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.notificationTargets = builder.notificationTargets;
        this.resolvedTime = builder.resolvedTime;
        this.status = builder.status;
        this.summary = builder.summary;
        this.title = builder.title;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the incident record.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the incident record.
     */
    public final String arn() {
        return arn;
    }

    /**
     * For responses, this returns true if the service returned a value for the AutomationExecutions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAutomationExecutions() {
        return automationExecutions != null && !(automationExecutions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The runbook, or automation document, that's run at the beginning of the incident.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAutomationExecutions} method.
     * </p>
     * 
     * @return The runbook, or automation document, that's run at the beginning of the incident.
     */
    public final List<AutomationExecution> automationExecutions() {
        return automationExecutions;
    }

    /**
     * <p>
     * The chat channel used for collaboration during an incident.
     * </p>
     * 
     * @return The chat channel used for collaboration during an incident.
     */
    public final ChatChannel chatChannel() {
        return chatChannel;
    }

    /**
     * <p>
     * The time that Incident Manager created the incident record.
     * </p>
     * 
     * @return The time that Incident Manager created the incident record.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The string Incident Manager uses to prevent duplicate incidents from being created by the same incident in the
     * same account.
     * </p>
     * 
     * @return The string Incident Manager uses to prevent duplicate incidents from being created by the same incident
     *         in the same account.
     */
    public final String dedupeString() {
        return dedupeString;
    }

    /**
     * <p>
     * The impact of the incident on customers and applications.
     * </p>
     * 
     * @return The impact of the incident on customers and applications.
     */
    public final Integer impact() {
        return impact;
    }

    /**
     * <p>
     * Details about the action that started the incident.
     * </p>
     * 
     * @return Details about the action that started the incident.
     */
    public final IncidentRecordSource incidentRecordSource() {
        return incidentRecordSource;
    }

    /**
     * <p>
     * Who modified the incident most recently.
     * </p>
     * 
     * @return Who modified the incident most recently.
     */
    public final String lastModifiedBy() {
        return lastModifiedBy;
    }

    /**
     * <p>
     * The time at which the incident was most recently modified.
     * </p>
     * 
     * @return The time at which the incident was most recently modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the NotificationTargets property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNotificationTargets() {
        return notificationTargets != null && !(notificationTargets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon SNS targets that are notified when updates are made to an incident.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotificationTargets} method.
     * </p>
     * 
     * @return The Amazon SNS targets that are notified when updates are made to an incident.
     */
    public final List<NotificationTargetItem> notificationTargets() {
        return notificationTargets;
    }

    /**
     * <p>
     * The time at which the incident was resolved. This appears as a timeline event.
     * </p>
     * 
     * @return The time at which the incident was resolved. This appears as a timeline event.
     */
    public final Instant resolvedTime() {
        return resolvedTime;
    }

    /**
     * <p>
     * The current status of the incident.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link IncidentRecordStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the incident.
     * @see IncidentRecordStatus
     */
    public final IncidentRecordStatus status() {
        return IncidentRecordStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the incident.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link IncidentRecordStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the incident.
     * @see IncidentRecordStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The summary of the incident. The summary is a brief synopsis of what occurred, what's currently happening, and
     * context of the incident.
     * </p>
     * 
     * @return The summary of the incident. The summary is a brief synopsis of what occurred, what's currently
     *         happening, and context of the incident.
     */
    public final String summary() {
        return summary;
    }

    /**
     * <p>
     * The title of the incident.
     * </p>
     * 
     * @return The title of the incident.
     */
    public final String title() {
        return title;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(hasAutomationExecutions() ? automationExecutions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(chatChannel());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(dedupeString());
        hashCode = 31 * hashCode + Objects.hashCode(impact());
        hashCode = 31 * hashCode + Objects.hashCode(incidentRecordSource());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedBy());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasNotificationTargets() ? notificationTargets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resolvedTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(summary());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IncidentRecord)) {
            return false;
        }
        IncidentRecord other = (IncidentRecord) obj;
        return Objects.equals(arn(), other.arn()) && hasAutomationExecutions() == other.hasAutomationExecutions()
                && Objects.equals(automationExecutions(), other.automationExecutions())
                && Objects.equals(chatChannel(), other.chatChannel()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(dedupeString(), other.dedupeString()) && Objects.equals(impact(), other.impact())
                && Objects.equals(incidentRecordSource(), other.incidentRecordSource())
                && Objects.equals(lastModifiedBy(), other.lastModifiedBy())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && hasNotificationTargets() == other.hasNotificationTargets()
                && Objects.equals(notificationTargets(), other.notificationTargets())
                && Objects.equals(resolvedTime(), other.resolvedTime())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(summary(), other.summary())
                && Objects.equals(title(), other.title());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IncidentRecord").add("Arn", arn())
                .add("AutomationExecutions", hasAutomationExecutions() ? automationExecutions() : null)
                .add("ChatChannel", chatChannel()).add("CreationTime", creationTime()).add("DedupeString", dedupeString())
                .add("Impact", impact()).add("IncidentRecordSource", incidentRecordSource())
                .add("LastModifiedBy", lastModifiedBy()).add("LastModifiedTime", lastModifiedTime())
                .add("NotificationTargets", hasNotificationTargets() ? notificationTargets() : null)
                .add("ResolvedTime", resolvedTime()).add("Status", statusAsString()).add("Summary", summary())
                .add("Title", title()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "automationExecutions":
            return Optional.ofNullable(clazz.cast(automationExecutions()));
        case "chatChannel":
            return Optional.ofNullable(clazz.cast(chatChannel()));
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "dedupeString":
            return Optional.ofNullable(clazz.cast(dedupeString()));
        case "impact":
            return Optional.ofNullable(clazz.cast(impact()));
        case "incidentRecordSource":
            return Optional.ofNullable(clazz.cast(incidentRecordSource()));
        case "lastModifiedBy":
            return Optional.ofNullable(clazz.cast(lastModifiedBy()));
        case "lastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "notificationTargets":
            return Optional.ofNullable(clazz.cast(notificationTargets()));
        case "resolvedTime":
            return Optional.ofNullable(clazz.cast(resolvedTime()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "summary":
            return Optional.ofNullable(clazz.cast(summary()));
        case "title":
            return Optional.ofNullable(clazz.cast(title()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IncidentRecord, T> g) {
        return obj -> g.apply((IncidentRecord) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IncidentRecord> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the incident record.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the incident record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The runbook, or automation document, that's run at the beginning of the incident.
         * </p>
         * 
         * @param automationExecutions
         *        The runbook, or automation document, that's run at the beginning of the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automationExecutions(Collection<AutomationExecution> automationExecutions);

        /**
         * <p>
         * The runbook, or automation document, that's run at the beginning of the incident.
         * </p>
         * 
         * @param automationExecutions
         *        The runbook, or automation document, that's run at the beginning of the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automationExecutions(AutomationExecution... automationExecutions);

        /**
         * <p>
         * The runbook, or automation document, that's run at the beginning of the incident.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssmincidents.model.AutomationExecution.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.ssmincidents.model.AutomationExecution#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssmincidents.model.AutomationExecution.Builder#build()} is called
         * immediately and its result is passed to {@link #automationExecutions(List<AutomationExecution>)}.
         * 
         * @param automationExecutions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssmincidents.model.AutomationExecution.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #automationExecutions(java.util.Collection<AutomationExecution>)
         */
        Builder automationExecutions(Consumer<AutomationExecution.Builder>... automationExecutions);

        /**
         * <p>
         * The chat channel used for collaboration during an incident.
         * </p>
         * 
         * @param chatChannel
         *        The chat channel used for collaboration during an incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder chatChannel(ChatChannel chatChannel);

        /**
         * <p>
         * The chat channel used for collaboration during an incident.
         * </p>
         * This is a convenience method that creates an instance of the {@link ChatChannel.Builder} avoiding the need to
         * create one manually via {@link ChatChannel#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ChatChannel.Builder#build()} is called immediately and its result
         * is passed to {@link #chatChannel(ChatChannel)}.
         * 
         * @param chatChannel
         *        a consumer that will call methods on {@link ChatChannel.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #chatChannel(ChatChannel)
         */
        default Builder chatChannel(Consumer<ChatChannel.Builder> chatChannel) {
            return chatChannel(ChatChannel.builder().applyMutation(chatChannel).build());
        }

        /**
         * <p>
         * The time that Incident Manager created the incident record.
         * </p>
         * 
         * @param creationTime
         *        The time that Incident Manager created the incident record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The string Incident Manager uses to prevent duplicate incidents from being created by the same incident in
         * the same account.
         * </p>
         * 
         * @param dedupeString
         *        The string Incident Manager uses to prevent duplicate incidents from being created by the same
         *        incident in the same account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dedupeString(String dedupeString);

        /**
         * <p>
         * The impact of the incident on customers and applications.
         * </p>
         * 
         * @param impact
         *        The impact of the incident on customers and applications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder impact(Integer impact);

        /**
         * <p>
         * Details about the action that started the incident.
         * </p>
         * 
         * @param incidentRecordSource
         *        Details about the action that started the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder incidentRecordSource(IncidentRecordSource incidentRecordSource);

        /**
         * <p>
         * Details about the action that started the incident.
         * </p>
         * This is a convenience method that creates an instance of the {@link IncidentRecordSource.Builder} avoiding
         * the need to create one manually via {@link IncidentRecordSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IncidentRecordSource.Builder#build()} is called immediately and
         * its result is passed to {@link #incidentRecordSource(IncidentRecordSource)}.
         * 
         * @param incidentRecordSource
         *        a consumer that will call methods on {@link IncidentRecordSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #incidentRecordSource(IncidentRecordSource)
         */
        default Builder incidentRecordSource(Consumer<IncidentRecordSource.Builder> incidentRecordSource) {
            return incidentRecordSource(IncidentRecordSource.builder().applyMutation(incidentRecordSource).build());
        }

        /**
         * <p>
         * Who modified the incident most recently.
         * </p>
         * 
         * @param lastModifiedBy
         *        Who modified the incident most recently.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedBy(String lastModifiedBy);

        /**
         * <p>
         * The time at which the incident was most recently modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The time at which the incident was most recently modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The Amazon SNS targets that are notified when updates are made to an incident.
         * </p>
         * 
         * @param notificationTargets
         *        The Amazon SNS targets that are notified when updates are made to an incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTargets(Collection<NotificationTargetItem> notificationTargets);

        /**
         * <p>
         * The Amazon SNS targets that are notified when updates are made to an incident.
         * </p>
         * 
         * @param notificationTargets
         *        The Amazon SNS targets that are notified when updates are made to an incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTargets(NotificationTargetItem... notificationTargets);

        /**
         * <p>
         * The Amazon SNS targets that are notified when updates are made to an incident.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssmincidents.model.NotificationTargetItem.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.ssmincidents.model.NotificationTargetItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssmincidents.model.NotificationTargetItem.Builder#build()} is called
         * immediately and its result is passed to {@link #notificationTargets(List<NotificationTargetItem>)}.
         * 
         * @param notificationTargets
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssmincidents.model.NotificationTargetItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationTargets(java.util.Collection<NotificationTargetItem>)
         */
        Builder notificationTargets(Consumer<NotificationTargetItem.Builder>... notificationTargets);

        /**
         * <p>
         * The time at which the incident was resolved. This appears as a timeline event.
         * </p>
         * 
         * @param resolvedTime
         *        The time at which the incident was resolved. This appears as a timeline event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedTime(Instant resolvedTime);

        /**
         * <p>
         * The current status of the incident.
         * </p>
         * 
         * @param status
         *        The current status of the incident.
         * @see IncidentRecordStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IncidentRecordStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the incident.
         * </p>
         * 
         * @param status
         *        The current status of the incident.
         * @see IncidentRecordStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IncidentRecordStatus
         */
        Builder status(IncidentRecordStatus status);

        /**
         * <p>
         * The summary of the incident. The summary is a brief synopsis of what occurred, what's currently happening,
         * and context of the incident.
         * </p>
         * 
         * @param summary
         *        The summary of the incident. The summary is a brief synopsis of what occurred, what's currently
         *        happening, and context of the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder summary(String summary);

        /**
         * <p>
         * The title of the incident.
         * </p>
         * 
         * @param title
         *        The title of the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private List<AutomationExecution> automationExecutions = DefaultSdkAutoConstructList.getInstance();

        private ChatChannel chatChannel;

        private Instant creationTime;

        private String dedupeString;

        private Integer impact;

        private IncidentRecordSource incidentRecordSource;

        private String lastModifiedBy;

        private Instant lastModifiedTime;

        private List<NotificationTargetItem> notificationTargets = DefaultSdkAutoConstructList.getInstance();

        private Instant resolvedTime;

        private String status;

        private String summary;

        private String title;

        private BuilderImpl() {
        }

        private BuilderImpl(IncidentRecord model) {
            arn(model.arn);
            automationExecutions(model.automationExecutions);
            chatChannel(model.chatChannel);
            creationTime(model.creationTime);
            dedupeString(model.dedupeString);
            impact(model.impact);
            incidentRecordSource(model.incidentRecordSource);
            lastModifiedBy(model.lastModifiedBy);
            lastModifiedTime(model.lastModifiedTime);
            notificationTargets(model.notificationTargets);
            resolvedTime(model.resolvedTime);
            status(model.status);
            summary(model.summary);
            title(model.title);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final List<AutomationExecution.Builder> getAutomationExecutions() {
            List<AutomationExecution.Builder> result = AutomationExecutionSetCopier.copyToBuilder(this.automationExecutions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAutomationExecutions(Collection<AutomationExecution.BuilderImpl> automationExecutions) {
            this.automationExecutions = AutomationExecutionSetCopier.copyFromBuilder(automationExecutions);
        }

        @Override
        public final Builder automationExecutions(Collection<AutomationExecution> automationExecutions) {
            this.automationExecutions = AutomationExecutionSetCopier.copy(automationExecutions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder automationExecutions(AutomationExecution... automationExecutions) {
            automationExecutions(Arrays.asList(automationExecutions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder automationExecutions(Consumer<AutomationExecution.Builder>... automationExecutions) {
            automationExecutions(Stream.of(automationExecutions).map(c -> AutomationExecution.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final ChatChannel.Builder getChatChannel() {
            return chatChannel != null ? chatChannel.toBuilder() : null;
        }

        public final void setChatChannel(ChatChannel.BuilderImpl chatChannel) {
            this.chatChannel = chatChannel != null ? chatChannel.build() : null;
        }

        @Override
        public final Builder chatChannel(ChatChannel chatChannel) {
            this.chatChannel = chatChannel;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getDedupeString() {
            return dedupeString;
        }

        public final void setDedupeString(String dedupeString) {
            this.dedupeString = dedupeString;
        }

        @Override
        public final Builder dedupeString(String dedupeString) {
            this.dedupeString = dedupeString;
            return this;
        }

        public final Integer getImpact() {
            return impact;
        }

        public final void setImpact(Integer impact) {
            this.impact = impact;
        }

        @Override
        public final Builder impact(Integer impact) {
            this.impact = impact;
            return this;
        }

        public final IncidentRecordSource.Builder getIncidentRecordSource() {
            return incidentRecordSource != null ? incidentRecordSource.toBuilder() : null;
        }

        public final void setIncidentRecordSource(IncidentRecordSource.BuilderImpl incidentRecordSource) {
            this.incidentRecordSource = incidentRecordSource != null ? incidentRecordSource.build() : null;
        }

        @Override
        public final Builder incidentRecordSource(IncidentRecordSource incidentRecordSource) {
            this.incidentRecordSource = incidentRecordSource;
            return this;
        }

        public final String getLastModifiedBy() {
            return lastModifiedBy;
        }

        public final void setLastModifiedBy(String lastModifiedBy) {
            this.lastModifiedBy = lastModifiedBy;
        }

        @Override
        public final Builder lastModifiedBy(String lastModifiedBy) {
            this.lastModifiedBy = lastModifiedBy;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final List<NotificationTargetItem.Builder> getNotificationTargets() {
            List<NotificationTargetItem.Builder> result = NotificationTargetSetCopier.copyToBuilder(this.notificationTargets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNotificationTargets(Collection<NotificationTargetItem.BuilderImpl> notificationTargets) {
            this.notificationTargets = NotificationTargetSetCopier.copyFromBuilder(notificationTargets);
        }

        @Override
        public final Builder notificationTargets(Collection<NotificationTargetItem> notificationTargets) {
            this.notificationTargets = NotificationTargetSetCopier.copy(notificationTargets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationTargets(NotificationTargetItem... notificationTargets) {
            notificationTargets(Arrays.asList(notificationTargets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationTargets(Consumer<NotificationTargetItem.Builder>... notificationTargets) {
            notificationTargets(Stream.of(notificationTargets)
                    .map(c -> NotificationTargetItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getResolvedTime() {
            return resolvedTime;
        }

        public final void setResolvedTime(Instant resolvedTime) {
            this.resolvedTime = resolvedTime;
        }

        @Override
        public final Builder resolvedTime(Instant resolvedTime) {
            this.resolvedTime = resolvedTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(IncidentRecordStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getSummary() {
            return summary;
        }

        public final void setSummary(String summary) {
            this.summary = summary;
        }

        @Override
        public final Builder summary(String summary) {
            this.summary = summary;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        @Override
        public IncidentRecord build() {
            return new IncidentRecord(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
