/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An array of search criteria that targets managed nodes using a key-value pair that you specify.
 * </p>
 * <note>
 * <p>
 * One or more targets must be specified for maintenance window Run Command-type tasks. Depending on the task, targets
 * are optional for other maintenance window task types (Automation, Lambda, and Step Functions). For more information
 * about running tasks that don't specify targets, see <a
 * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/maintenance-windows-targetless-tasks.html"
 * >Registering maintenance window tasks without targets</a> in the <i>Amazon Web Services Systems Manager User
 * Guide</i>.
 * </p>
 * </note>
 * <p>
 * Supported formats include the following.
 * </p>
 * <p>
 * <b>For all Systems Manager tools:</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Key=tag-key,Values=tag-value-1,tag-value-2</code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>For Automation and Change Manager:</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Key=tag:tag-key,Values=tag-value</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=ResourceGroup,Values=resource-group-name</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=ParameterValues,Values=value-1,value-2,value-3</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * To target all instances in the Amazon Web Services Region:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Key=AWS::EC2::Instance,Values=*</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=InstanceIds,Values=*</code>
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * <b>For Run Command and Maintenance Windows:</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Key=InstanceIds,Values=instance-id-1,instance-id-2,instance-id-3</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=tag:tag-key,Values=tag-value-1,tag-value-2</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=resource-groups:Name,Values=resource-group-name</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * Additionally, Maintenance Windows support targeting resource types:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Key=resource-groups:ResourceTypeFilters,Values=resource-type-1,resource-type-2</code>
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * <b>For State Manager:</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Key=InstanceIds,Values=instance-id-1,instance-id-2,instance-id-3</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Key=tag:tag-key,Values=tag-value-1,tag-value-2</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * To target all instances in the Amazon Web Services Region:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Key=InstanceIds,Values=*</code>
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * For more information about how to send commands that target managed nodes using <code>Key,Value</code> parameters,
 * see <a href=
 * "https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html#send-commands-targeting"
 * >Targeting multiple managed nodes</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Target implements SdkPojo, Serializable, ToCopyableBuilder<Target.Builder, Target> {
    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Key")
            .getter(getter(Target::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Key").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(Target::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_FIELD, VALUES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String key;

    private final List<String> values;

    private Target(BuilderImpl builder) {
        this.key = builder.key;
        this.values = builder.values;
    }

    /**
     * <p>
     * User-defined criteria for sending commands that target managed nodes that meet the criteria.
     * </p>
     * 
     * @return User-defined criteria for sending commands that target managed nodes that meet the criteria.
     */
    public final String key() {
        return key;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * User-defined criteria that maps to <code>Key</code>. For example, if you specified <code>tag:ServerRole</code>,
     * you could specify <code>value:WebServer</code> to run a command on instances that include EC2 tags of
     * <code>ServerRole,WebServer</code>.
     * </p>
     * <p>
     * Depending on the type of target, the maximum number of values for a key might be lower than the global maximum of
     * 50.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return User-defined criteria that maps to <code>Key</code>. For example, if you specified
     *         <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on instances
     *         that include EC2 tags of <code>ServerRole,WebServer</code>. </p>
     *         <p>
     *         Depending on the type of target, the maximum number of values for a key might be lower than the global
     *         maximum of 50.
     */
    public final List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Target)) {
            return false;
        }
        Target other = (Target) obj;
        return Objects.equals(key(), other.key()) && hasValues() == other.hasValues() && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Target").add("Key", key()).add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Key", KEY_FIELD);
        map.put("Values", VALUES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Target, T> g) {
        return obj -> g.apply((Target) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Target> {
        /**
         * <p>
         * User-defined criteria for sending commands that target managed nodes that meet the criteria.
         * </p>
         * 
         * @param key
         *        User-defined criteria for sending commands that target managed nodes that meet the criteria.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * User-defined criteria that maps to <code>Key</code>. For example, if you specified
         * <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on instances
         * that include EC2 tags of <code>ServerRole,WebServer</code>.
         * </p>
         * <p>
         * Depending on the type of target, the maximum number of values for a key might be lower than the global
         * maximum of 50.
         * </p>
         * 
         * @param values
         *        User-defined criteria that maps to <code>Key</code>. For example, if you specified
         *        <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on
         *        instances that include EC2 tags of <code>ServerRole,WebServer</code>. </p>
         *        <p>
         *        Depending on the type of target, the maximum number of values for a key might be lower than the global
         *        maximum of 50.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * User-defined criteria that maps to <code>Key</code>. For example, if you specified
         * <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on instances
         * that include EC2 tags of <code>ServerRole,WebServer</code>.
         * </p>
         * <p>
         * Depending on the type of target, the maximum number of values for a key might be lower than the global
         * maximum of 50.
         * </p>
         * 
         * @param values
         *        User-defined criteria that maps to <code>Key</code>. For example, if you specified
         *        <code>tag:ServerRole</code>, you could specify <code>value:WebServer</code> to run a command on
         *        instances that include EC2 tags of <code>ServerRole,WebServer</code>. </p>
         *        <p>
         *        Depending on the type of target, the maximum number of values for a key might be lower than the global
         *        maximum of 50.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String key;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Target model) {
            key(model.key);
            values(model.values);
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<String> values) {
            this.values = TargetValuesCopier.copy(values);
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = TargetValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        @Override
        public Target build() {
            return new Target(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
