/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Metadata includes information like the Amazon Resource Name (ARN) of the last user to update the parameter and the
 * date and time the parameter was last used.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ParameterMetadata implements SdkPojo, Serializable,
        ToCopyableBuilder<ParameterMetadata.Builder, ParameterMetadata> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ParameterMetadata::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ARN")
            .getter(getter(ParameterMetadata::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(ParameterMetadata::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KeyId")
            .getter(getter(ParameterMetadata::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedDate").getter(getter(ParameterMetadata::lastModifiedDate))
            .setter(setter(Builder::lastModifiedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedDate").build()).build();

    private static final SdkField<String> LAST_MODIFIED_USER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastModifiedUser").getter(getter(ParameterMetadata::lastModifiedUser))
            .setter(setter(Builder::lastModifiedUser))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedUser").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ParameterMetadata::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> ALLOWED_PATTERN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AllowedPattern").getter(getter(ParameterMetadata::allowedPattern))
            .setter(setter(Builder::allowedPattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedPattern").build()).build();

    private static final SdkField<Long> VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Version")
            .getter(getter(ParameterMetadata::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<String> TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Tier")
            .getter(getter(ParameterMetadata::tierAsString)).setter(setter(Builder::tier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tier").build()).build();

    private static final SdkField<List<ParameterInlinePolicy>> POLICIES_FIELD = SdkField
            .<List<ParameterInlinePolicy>> builder(MarshallingType.LIST)
            .memberName("Policies")
            .getter(getter(ParameterMetadata::policies))
            .setter(setter(Builder::policies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ParameterInlinePolicy> builder(MarshallingType.SDK_POJO)
                                            .constructor(ParameterInlinePolicy::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataType").getter(getter(ParameterMetadata::dataType)).setter(setter(Builder::dataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            TYPE_FIELD, KEY_ID_FIELD, LAST_MODIFIED_DATE_FIELD, LAST_MODIFIED_USER_FIELD, DESCRIPTION_FIELD,
            ALLOWED_PATTERN_FIELD, VERSION_FIELD, TIER_FIELD, POLICIES_FIELD, DATA_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String type;

    private final String keyId;

    private final Instant lastModifiedDate;

    private final String lastModifiedUser;

    private final String description;

    private final String allowedPattern;

    private final Long version;

    private final String tier;

    private final List<ParameterInlinePolicy> policies;

    private final String dataType;

    private ParameterMetadata(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.type = builder.type;
        this.keyId = builder.keyId;
        this.lastModifiedDate = builder.lastModifiedDate;
        this.lastModifiedUser = builder.lastModifiedUser;
        this.description = builder.description;
        this.allowedPattern = builder.allowedPattern;
        this.version = builder.version;
        this.tier = builder.tier;
        this.policies = builder.policies;
        this.dataType = builder.dataType;
    }

    /**
     * <p>
     * The parameter name.
     * </p>
     * 
     * @return The parameter name.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the parameter.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the parameter.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The type of parameter. Valid parameter types include the following: <code>String</code>, <code>StringList</code>,
     * and <code>SecureString</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter. Valid parameter types include the following: <code>String</code>,
     *         <code>StringList</code>, and <code>SecureString</code>.
     * @see ParameterType
     */
    public final ParameterType type() {
        return ParameterType.fromValue(type);
    }

    /**
     * <p>
     * The type of parameter. Valid parameter types include the following: <code>String</code>, <code>StringList</code>,
     * and <code>SecureString</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter. Valid parameter types include the following: <code>String</code>,
     *         <code>StringList</code>, and <code>SecureString</code>.
     * @see ParameterType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The alias of the Key Management Service (KMS) key used to encrypt the parameter. Applies to
     * <code>SecureString</code> parameters only.
     * </p>
     * 
     * @return The alias of the Key Management Service (KMS) key used to encrypt the parameter. Applies to
     *         <code>SecureString</code> parameters only.
     */
    public final String keyId() {
        return keyId;
    }

    /**
     * <p>
     * Date the parameter was last changed or updated.
     * </p>
     * 
     * @return Date the parameter was last changed or updated.
     */
    public final Instant lastModifiedDate() {
        return lastModifiedDate;
    }

    /**
     * <p>
     * Amazon Resource Name (ARN) of the Amazon Web Services user who last changed the parameter.
     * </p>
     * 
     * @return Amazon Resource Name (ARN) of the Amazon Web Services user who last changed the parameter.
     */
    public final String lastModifiedUser() {
        return lastModifiedUser;
    }

    /**
     * <p>
     * Description of the parameter actions.
     * </p>
     * 
     * @return Description of the parameter actions.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A parameter name can include only the following letters and symbols.
     * </p>
     * <p>
     * a-zA-Z0-9_.-
     * </p>
     * 
     * @return A parameter name can include only the following letters and symbols.</p>
     *         <p>
     *         a-zA-Z0-9_.-
     */
    public final String allowedPattern() {
        return allowedPattern;
    }

    /**
     * <p>
     * The parameter version.
     * </p>
     * 
     * @return The parameter version.
     */
    public final Long version() {
        return version;
    }

    /**
     * <p>
     * The parameter tier.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ParameterTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The parameter tier.
     * @see ParameterTier
     */
    public final ParameterTier tier() {
        return ParameterTier.fromValue(tier);
    }

    /**
     * <p>
     * The parameter tier.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link ParameterTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The parameter tier.
     * @see ParameterTier
     */
    public final String tierAsString() {
        return tier;
    }

    /**
     * For responses, this returns true if the service returned a value for the Policies property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPolicies() {
        return policies != null && !(policies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of policies associated with a parameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPolicies} method.
     * </p>
     * 
     * @return A list of policies associated with a parameter.
     */
    public final List<ParameterInlinePolicy> policies() {
        return policies;
    }

    /**
     * <p>
     * The data type of the parameter, such as <code>text</code> or <code>aws:ec2:image</code>. The default is
     * <code>text</code>.
     * </p>
     * 
     * @return The data type of the parameter, such as <code>text</code> or <code>aws:ec2:image</code>. The default is
     *         <code>text</code>.
     */
    public final String dataType() {
        return dataType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedDate());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedUser());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(allowedPattern());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(tierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPolicies() ? policies() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataType());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ParameterMetadata)) {
            return false;
        }
        ParameterMetadata other = (ParameterMetadata) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(keyId(), other.keyId())
                && Objects.equals(lastModifiedDate(), other.lastModifiedDate())
                && Objects.equals(lastModifiedUser(), other.lastModifiedUser())
                && Objects.equals(description(), other.description()) && Objects.equals(allowedPattern(), other.allowedPattern())
                && Objects.equals(version(), other.version()) && Objects.equals(tierAsString(), other.tierAsString())
                && hasPolicies() == other.hasPolicies() && Objects.equals(policies(), other.policies())
                && Objects.equals(dataType(), other.dataType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ParameterMetadata").add("Name", name()).add("ARN", arn()).add("Type", typeAsString())
                .add("KeyId", keyId()).add("LastModifiedDate", lastModifiedDate()).add("LastModifiedUser", lastModifiedUser())
                .add("Description", description()).add("AllowedPattern", allowedPattern()).add("Version", version())
                .add("Tier", tierAsString()).add("Policies", hasPolicies() ? policies() : null).add("DataType", dataType())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "LastModifiedDate":
            return Optional.ofNullable(clazz.cast(lastModifiedDate()));
        case "LastModifiedUser":
            return Optional.ofNullable(clazz.cast(lastModifiedUser()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "AllowedPattern":
            return Optional.ofNullable(clazz.cast(allowedPattern()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Tier":
            return Optional.ofNullable(clazz.cast(tierAsString()));
        case "Policies":
            return Optional.ofNullable(clazz.cast(policies()));
        case "DataType":
            return Optional.ofNullable(clazz.cast(dataType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("ARN", ARN_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("KeyId", KEY_ID_FIELD);
        map.put("LastModifiedDate", LAST_MODIFIED_DATE_FIELD);
        map.put("LastModifiedUser", LAST_MODIFIED_USER_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("AllowedPattern", ALLOWED_PATTERN_FIELD);
        map.put("Version", VERSION_FIELD);
        map.put("Tier", TIER_FIELD);
        map.put("Policies", POLICIES_FIELD);
        map.put("DataType", DATA_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ParameterMetadata, T> g) {
        return obj -> g.apply((ParameterMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ParameterMetadata> {
        /**
         * <p>
         * The parameter name.
         * </p>
         * 
         * @param name
         *        The parameter name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the parameter.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The type of parameter. Valid parameter types include the following: <code>String</code>,
         * <code>StringList</code>, and <code>SecureString</code>.
         * </p>
         * 
         * @param type
         *        The type of parameter. Valid parameter types include the following: <code>String</code>,
         *        <code>StringList</code>, and <code>SecureString</code>.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of parameter. Valid parameter types include the following: <code>String</code>,
         * <code>StringList</code>, and <code>SecureString</code>.
         * </p>
         * 
         * @param type
         *        The type of parameter. Valid parameter types include the following: <code>String</code>,
         *        <code>StringList</code>, and <code>SecureString</code>.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(ParameterType type);

        /**
         * <p>
         * The alias of the Key Management Service (KMS) key used to encrypt the parameter. Applies to
         * <code>SecureString</code> parameters only.
         * </p>
         * 
         * @param keyId
         *        The alias of the Key Management Service (KMS) key used to encrypt the parameter. Applies to
         *        <code>SecureString</code> parameters only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * Date the parameter was last changed or updated.
         * </p>
         * 
         * @param lastModifiedDate
         *        Date the parameter was last changed or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedDate(Instant lastModifiedDate);

        /**
         * <p>
         * Amazon Resource Name (ARN) of the Amazon Web Services user who last changed the parameter.
         * </p>
         * 
         * @param lastModifiedUser
         *        Amazon Resource Name (ARN) of the Amazon Web Services user who last changed the parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedUser(String lastModifiedUser);

        /**
         * <p>
         * Description of the parameter actions.
         * </p>
         * 
         * @param description
         *        Description of the parameter actions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A parameter name can include only the following letters and symbols.
         * </p>
         * <p>
         * a-zA-Z0-9_.-
         * </p>
         * 
         * @param allowedPattern
         *        A parameter name can include only the following letters and symbols.</p>
         *        <p>
         *        a-zA-Z0-9_.-
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedPattern(String allowedPattern);

        /**
         * <p>
         * The parameter version.
         * </p>
         * 
         * @param version
         *        The parameter version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Long version);

        /**
         * <p>
         * The parameter tier.
         * </p>
         * 
         * @param tier
         *        The parameter tier.
         * @see ParameterTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterTier
         */
        Builder tier(String tier);

        /**
         * <p>
         * The parameter tier.
         * </p>
         * 
         * @param tier
         *        The parameter tier.
         * @see ParameterTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterTier
         */
        Builder tier(ParameterTier tier);

        /**
         * <p>
         * A list of policies associated with a parameter.
         * </p>
         * 
         * @param policies
         *        A list of policies associated with a parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(Collection<ParameterInlinePolicy> policies);

        /**
         * <p>
         * A list of policies associated with a parameter.
         * </p>
         * 
         * @param policies
         *        A list of policies associated with a parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(ParameterInlinePolicy... policies);

        /**
         * <p>
         * A list of policies associated with a parameter.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.ParameterInlinePolicy.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ssm.model.ParameterInlinePolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.ParameterInlinePolicy.Builder#build()} is called immediately
         * and its result is passed to {@link #policies(List<ParameterInlinePolicy>)}.
         * 
         * @param policies
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.ParameterInlinePolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #policies(java.util.Collection<ParameterInlinePolicy>)
         */
        Builder policies(Consumer<ParameterInlinePolicy.Builder>... policies);

        /**
         * <p>
         * The data type of the parameter, such as <code>text</code> or <code>aws:ec2:image</code>. The default is
         * <code>text</code>.
         * </p>
         * 
         * @param dataType
         *        The data type of the parameter, such as <code>text</code> or <code>aws:ec2:image</code>. The default
         *        is <code>text</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String type;

        private String keyId;

        private Instant lastModifiedDate;

        private String lastModifiedUser;

        private String description;

        private String allowedPattern;

        private Long version;

        private String tier;

        private List<ParameterInlinePolicy> policies = DefaultSdkAutoConstructList.getInstance();

        private String dataType;

        private BuilderImpl() {
        }

        private BuilderImpl(ParameterMetadata model) {
            name(model.name);
            arn(model.arn);
            type(model.type);
            keyId(model.keyId);
            lastModifiedDate(model.lastModifiedDate);
            lastModifiedUser(model.lastModifiedUser);
            description(model.description);
            allowedPattern(model.allowedPattern);
            version(model.version);
            tier(model.tier);
            policies(model.policies);
            dataType(model.dataType);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ParameterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getKeyId() {
            return keyId;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final Instant getLastModifiedDate() {
            return lastModifiedDate;
        }

        public final void setLastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
        }

        @Override
        public final Builder lastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
            return this;
        }

        public final String getLastModifiedUser() {
            return lastModifiedUser;
        }

        public final void setLastModifiedUser(String lastModifiedUser) {
            this.lastModifiedUser = lastModifiedUser;
        }

        @Override
        public final Builder lastModifiedUser(String lastModifiedUser) {
            this.lastModifiedUser = lastModifiedUser;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getAllowedPattern() {
            return allowedPattern;
        }

        public final void setAllowedPattern(String allowedPattern) {
            this.allowedPattern = allowedPattern;
        }

        @Override
        public final Builder allowedPattern(String allowedPattern) {
            this.allowedPattern = allowedPattern;
            return this;
        }

        public final Long getVersion() {
            return version;
        }

        public final void setVersion(Long version) {
            this.version = version;
        }

        @Override
        public final Builder version(Long version) {
            this.version = version;
            return this;
        }

        public final String getTier() {
            return tier;
        }

        public final void setTier(String tier) {
            this.tier = tier;
        }

        @Override
        public final Builder tier(String tier) {
            this.tier = tier;
            return this;
        }

        @Override
        public final Builder tier(ParameterTier tier) {
            this.tier(tier == null ? null : tier.toString());
            return this;
        }

        public final List<ParameterInlinePolicy.Builder> getPolicies() {
            List<ParameterInlinePolicy.Builder> result = ParameterPolicyListCopier.copyToBuilder(this.policies);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPolicies(Collection<ParameterInlinePolicy.BuilderImpl> policies) {
            this.policies = ParameterPolicyListCopier.copyFromBuilder(policies);
        }

        @Override
        public final Builder policies(Collection<ParameterInlinePolicy> policies) {
            this.policies = ParameterPolicyListCopier.copy(policies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policies(ParameterInlinePolicy... policies) {
            policies(Arrays.asList(policies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policies(Consumer<ParameterInlinePolicy.Builder>... policies) {
            policies(Stream.of(policies).map(c -> ParameterInlinePolicy.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getDataType() {
            return dataType;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        @Override
        public ParameterMetadata build() {
            return new ParameterMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
