/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the name of a SSM document.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentIdentifier implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentIdentifier.Builder, DocumentIdentifier> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(DocumentIdentifier::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedDate").getter(getter(DocumentIdentifier::createdDate)).setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedDate").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayName").getter(getter(DocumentIdentifier::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Owner")
            .getter(getter(DocumentIdentifier::owner)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Owner").build()).build();

    private static final SdkField<String> VERSION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionName").getter(getter(DocumentIdentifier::versionName)).setter(setter(Builder::versionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionName").build()).build();

    private static final SdkField<List<String>> PLATFORM_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PlatformTypes")
            .getter(getter(DocumentIdentifier::platformTypesAsStrings))
            .setter(setter(Builder::platformTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentVersion").getter(getter(DocumentIdentifier::documentVersion))
            .setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<String> DOCUMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentType").getter(getter(DocumentIdentifier::documentTypeAsString))
            .setter(setter(Builder::documentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentType").build()).build();

    private static final SdkField<String> SCHEMA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SchemaVersion").getter(getter(DocumentIdentifier::schemaVersion)).setter(setter(Builder::schemaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaVersion").build()).build();

    private static final SdkField<String> DOCUMENT_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentFormat").getter(getter(DocumentIdentifier::documentFormatAsString))
            .setter(setter(Builder::documentFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentFormat").build()).build();

    private static final SdkField<String> TARGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetType").getter(getter(DocumentIdentifier::targetType)).setter(setter(Builder::targetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(DocumentIdentifier::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DocumentRequires>> REQUIRES_FIELD = SdkField
            .<List<DocumentRequires>> builder(MarshallingType.LIST)
            .memberName("Requires")
            .getter(getter(DocumentIdentifier::requires))
            .setter(setter(Builder::requires))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Requires").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentRequires> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentRequires::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> REVIEW_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReviewStatus").getter(getter(DocumentIdentifier::reviewStatusAsString))
            .setter(setter(Builder::reviewStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReviewStatus").build()).build();

    private static final SdkField<String> AUTHOR_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Author")
            .getter(getter(DocumentIdentifier::author)).setter(setter(Builder::author))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Author").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CREATED_DATE_FIELD, DISPLAY_NAME_FIELD, OWNER_FIELD, VERSION_NAME_FIELD, PLATFORM_TYPES_FIELD,
            DOCUMENT_VERSION_FIELD, DOCUMENT_TYPE_FIELD, SCHEMA_VERSION_FIELD, DOCUMENT_FORMAT_FIELD, TARGET_TYPE_FIELD,
            TAGS_FIELD, REQUIRES_FIELD, REVIEW_STATUS_FIELD, AUTHOR_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Instant createdDate;

    private final String displayName;

    private final String owner;

    private final String versionName;

    private final List<String> platformTypes;

    private final String documentVersion;

    private final String documentType;

    private final String schemaVersion;

    private final String documentFormat;

    private final String targetType;

    private final List<Tag> tags;

    private final List<DocumentRequires> requires;

    private final String reviewStatus;

    private final String author;

    private DocumentIdentifier(BuilderImpl builder) {
        this.name = builder.name;
        this.createdDate = builder.createdDate;
        this.displayName = builder.displayName;
        this.owner = builder.owner;
        this.versionName = builder.versionName;
        this.platformTypes = builder.platformTypes;
        this.documentVersion = builder.documentVersion;
        this.documentType = builder.documentType;
        this.schemaVersion = builder.schemaVersion;
        this.documentFormat = builder.documentFormat;
        this.targetType = builder.targetType;
        this.tags = builder.tags;
        this.requires = builder.requires;
        this.reviewStatus = builder.reviewStatus;
        this.author = builder.author;
    }

    /**
     * <p>
     * The name of the SSM document.
     * </p>
     * 
     * @return The name of the SSM document.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The date the SSM document was created.
     * </p>
     * 
     * @return The date the SSM document was created.
     */
    public final Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * An optional field where you can specify a friendly name for the SSM document. This value can differ for each
     * version of the document. If you want to update this value, see <a>UpdateDocument</a>.
     * </p>
     * 
     * @return An optional field where you can specify a friendly name for the SSM document. This value can differ for
     *         each version of the document. If you want to update this value, see <a>UpdateDocument</a>.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The Amazon Web Services user that created the document.
     * </p>
     * 
     * @return The Amazon Web Services user that created the document.
     */
    public final String owner() {
        return owner;
    }

    /**
     * <p>
     * An optional field specifying the version of the artifact associated with the document. For example, 12.6. This
     * value is unique across all versions of a document, and can't be changed.
     * </p>
     * 
     * @return An optional field specifying the version of the artifact associated with the document. For example, 12.6.
     *         This value is unique across all versions of a document, and can't be changed.
     */
    public final String versionName() {
        return versionName;
    }

    /**
     * <p>
     * The operating system platform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatformTypes} method.
     * </p>
     * 
     * @return The operating system platform.
     */
    public final List<PlatformType> platformTypes() {
        return PlatformTypeListCopier.copyStringToEnum(platformTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the PlatformTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlatformTypes() {
        return platformTypes != null && !(platformTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The operating system platform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatformTypes} method.
     * </p>
     * 
     * @return The operating system platform.
     */
    public final List<String> platformTypesAsStrings() {
        return platformTypes;
    }

    /**
     * <p>
     * The document version.
     * </p>
     * 
     * @return The document version.
     */
    public final String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * The document type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The document type.
     * @see DocumentType
     */
    public final DocumentType documentType() {
        return DocumentType.fromValue(documentType);
    }

    /**
     * <p>
     * The document type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The document type.
     * @see DocumentType
     */
    public final String documentTypeAsString() {
        return documentType;
    }

    /**
     * <p>
     * The schema version.
     * </p>
     * 
     * @return The schema version.
     */
    public final String schemaVersion() {
        return schemaVersion;
    }

    /**
     * <p>
     * The document format, either JSON or YAML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return The document format, either JSON or YAML.
     * @see DocumentFormat
     */
    public final DocumentFormat documentFormat() {
        return DocumentFormat.fromValue(documentFormat);
    }

    /**
     * <p>
     * The document format, either JSON or YAML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return The document format, either JSON or YAML.
     * @see DocumentFormat
     */
    public final String documentFormatAsString() {
        return documentFormat;
    }

    /**
     * <p>
     * The target type which defines the kinds of resources the document can run on. For example,
     * <code>/AWS::EC2::Instance</code>. For a list of valid resource types, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">Amazon
     * Web Services resource and property types reference</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * 
     * @return The target type which defines the kinds of resources the document can run on. For example,
     *         <code>/AWS::EC2::Instance</code>. For a list of valid resource types, see <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
     *         >Amazon Web Services resource and property types reference</a> in the <i>CloudFormation User Guide</i>.
     */
    public final String targetType() {
        return targetType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags, or metadata, that have been applied to the document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags, or metadata, that have been applied to the document.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the Requires property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRequires() {
        return requires != null && !(requires instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code> document
     * requires an <code>ApplicationConfigurationSchema</code> document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequires} method.
     * </p>
     * 
     * @return A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
     *         document requires an <code>ApplicationConfigurationSchema</code> document.
     */
    public final List<DocumentRequires> requires() {
        return requires;
    }

    /**
     * <p>
     * The current status of a document review.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reviewStatus} will
     * return {@link ReviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #reviewStatusAsString}.
     * </p>
     * 
     * @return The current status of a document review.
     * @see ReviewStatus
     */
    public final ReviewStatus reviewStatus() {
        return ReviewStatus.fromValue(reviewStatus);
    }

    /**
     * <p>
     * The current status of a document review.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reviewStatus} will
     * return {@link ReviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #reviewStatusAsString}.
     * </p>
     * 
     * @return The current status of a document review.
     * @see ReviewStatus
     */
    public final String reviewStatusAsString() {
        return reviewStatus;
    }

    /**
     * <p>
     * The user in your organization who created the document.
     * </p>
     * 
     * @return The user in your organization who created the document.
     */
    public final String author() {
        return author;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(versionName());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlatformTypes() ? platformTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetType());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRequires() ? requires() : null);
        hashCode = 31 * hashCode + Objects.hashCode(reviewStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(author());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentIdentifier)) {
            return false;
        }
        DocumentIdentifier other = (DocumentIdentifier) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(createdDate(), other.createdDate())
                && Objects.equals(displayName(), other.displayName()) && Objects.equals(owner(), other.owner())
                && Objects.equals(versionName(), other.versionName()) && hasPlatformTypes() == other.hasPlatformTypes()
                && Objects.equals(platformTypesAsStrings(), other.platformTypesAsStrings())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(documentTypeAsString(), other.documentTypeAsString())
                && Objects.equals(schemaVersion(), other.schemaVersion())
                && Objects.equals(documentFormatAsString(), other.documentFormatAsString())
                && Objects.equals(targetType(), other.targetType()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasRequires() == other.hasRequires()
                && Objects.equals(requires(), other.requires())
                && Objects.equals(reviewStatusAsString(), other.reviewStatusAsString())
                && Objects.equals(author(), other.author());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DocumentIdentifier").add("Name", name()).add("CreatedDate", createdDate())
                .add("DisplayName", displayName()).add("Owner", owner()).add("VersionName", versionName())
                .add("PlatformTypes", hasPlatformTypes() ? platformTypesAsStrings() : null)
                .add("DocumentVersion", documentVersion()).add("DocumentType", documentTypeAsString())
                .add("SchemaVersion", schemaVersion()).add("DocumentFormat", documentFormatAsString())
                .add("TargetType", targetType()).add("Tags", hasTags() ? tags() : null)
                .add("Requires", hasRequires() ? requires() : null).add("ReviewStatus", reviewStatusAsString())
                .add("Author", author()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "CreatedDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "Owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "VersionName":
            return Optional.ofNullable(clazz.cast(versionName()));
        case "PlatformTypes":
            return Optional.ofNullable(clazz.cast(platformTypesAsStrings()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "DocumentType":
            return Optional.ofNullable(clazz.cast(documentTypeAsString()));
        case "SchemaVersion":
            return Optional.ofNullable(clazz.cast(schemaVersion()));
        case "DocumentFormat":
            return Optional.ofNullable(clazz.cast(documentFormatAsString()));
        case "TargetType":
            return Optional.ofNullable(clazz.cast(targetType()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Requires":
            return Optional.ofNullable(clazz.cast(requires()));
        case "ReviewStatus":
            return Optional.ofNullable(clazz.cast(reviewStatusAsString()));
        case "Author":
            return Optional.ofNullable(clazz.cast(author()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DocumentIdentifier, T> g) {
        return obj -> g.apply((DocumentIdentifier) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentIdentifier> {
        /**
         * <p>
         * The name of the SSM document.
         * </p>
         * 
         * @param name
         *        The name of the SSM document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The date the SSM document was created.
         * </p>
         * 
         * @param createdDate
         *        The date the SSM document was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * An optional field where you can specify a friendly name for the SSM document. This value can differ for each
         * version of the document. If you want to update this value, see <a>UpdateDocument</a>.
         * </p>
         * 
         * @param displayName
         *        An optional field where you can specify a friendly name for the SSM document. This value can differ
         *        for each version of the document. If you want to update this value, see <a>UpdateDocument</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The Amazon Web Services user that created the document.
         * </p>
         * 
         * @param owner
         *        The Amazon Web Services user that created the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * An optional field specifying the version of the artifact associated with the document. For example, 12.6.
         * This value is unique across all versions of a document, and can't be changed.
         * </p>
         * 
         * @param versionName
         *        An optional field specifying the version of the artifact associated with the document. For example,
         *        12.6. This value is unique across all versions of a document, and can't be changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionName(String versionName);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypesWithStrings(Collection<String> platformTypes);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypesWithStrings(String... platformTypes);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(Collection<PlatformType> platformTypes);

        /**
         * <p>
         * The operating system platform.
         * </p>
         * 
         * @param platformTypes
         *        The operating system platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(PlatformType... platformTypes);

        /**
         * <p>
         * The document version.
         * </p>
         * 
         * @param documentVersion
         *        The document version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * The document type.
         * </p>
         * 
         * @param documentType
         *        The document type.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(String documentType);

        /**
         * <p>
         * The document type.
         * </p>
         * 
         * @param documentType
         *        The document type.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(DocumentType documentType);

        /**
         * <p>
         * The schema version.
         * </p>
         * 
         * @param schemaVersion
         *        The schema version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The document format, either JSON or YAML.
         * </p>
         * 
         * @param documentFormat
         *        The document format, either JSON or YAML.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(String documentFormat);

        /**
         * <p>
         * The document format, either JSON or YAML.
         * </p>
         * 
         * @param documentFormat
         *        The document format, either JSON or YAML.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(DocumentFormat documentFormat);

        /**
         * <p>
         * The target type which defines the kinds of resources the document can run on. For example,
         * <code>/AWS::EC2::Instance</code>. For a list of valid resource types, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         * >Amazon Web Services resource and property types reference</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * 
         * @param targetType
         *        The target type which defines the kinds of resources the document can run on. For example,
         *        <code>/AWS::EC2::Instance</code>. For a list of valid resource types, see <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         *        >Amazon Web Services resource and property types reference</a> in the <i>CloudFormation User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetType(String targetType);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ssm.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ssm.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ssm.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         * document requires an <code>ApplicationConfigurationSchema</code> document.
         * </p>
         * 
         * @param requires
         *        A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         *        document requires an <code>ApplicationConfigurationSchema</code> document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requires(Collection<DocumentRequires> requires);

        /**
         * <p>
         * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         * document requires an <code>ApplicationConfigurationSchema</code> document.
         * </p>
         * 
         * @param requires
         *        A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         *        document requires an <code>ApplicationConfigurationSchema</code> document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requires(DocumentRequires... requires);

        /**
         * <p>
         * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         * document requires an <code>ApplicationConfigurationSchema</code> document.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.DocumentRequires.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ssm.model.DocumentRequires#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.DocumentRequires.Builder#build()} is called immediately and
         * its result is passed to {@link #requires(List<DocumentRequires>)}.
         * 
         * @param requires
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.DocumentRequires.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #requires(java.util.Collection<DocumentRequires>)
         */
        Builder requires(Consumer<DocumentRequires.Builder>... requires);

        /**
         * <p>
         * The current status of a document review.
         * </p>
         * 
         * @param reviewStatus
         *        The current status of a document review.
         * @see ReviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReviewStatus
         */
        Builder reviewStatus(String reviewStatus);

        /**
         * <p>
         * The current status of a document review.
         * </p>
         * 
         * @param reviewStatus
         *        The current status of a document review.
         * @see ReviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReviewStatus
         */
        Builder reviewStatus(ReviewStatus reviewStatus);

        /**
         * <p>
         * The user in your organization who created the document.
         * </p>
         * 
         * @param author
         *        The user in your organization who created the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder author(String author);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Instant createdDate;

        private String displayName;

        private String owner;

        private String versionName;

        private List<String> platformTypes = DefaultSdkAutoConstructList.getInstance();

        private String documentVersion;

        private String documentType;

        private String schemaVersion;

        private String documentFormat;

        private String targetType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<DocumentRequires> requires = DefaultSdkAutoConstructList.getInstance();

        private String reviewStatus;

        private String author;

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentIdentifier model) {
            name(model.name);
            createdDate(model.createdDate);
            displayName(model.displayName);
            owner(model.owner);
            versionName(model.versionName);
            platformTypesWithStrings(model.platformTypes);
            documentVersion(model.documentVersion);
            documentType(model.documentType);
            schemaVersion(model.schemaVersion);
            documentFormat(model.documentFormat);
            targetType(model.targetType);
            tags(model.tags);
            requires(model.requires);
            reviewStatus(model.reviewStatus);
            author(model.author);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getOwner() {
            return owner;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final String getVersionName() {
            return versionName;
        }

        public final void setVersionName(String versionName) {
            this.versionName = versionName;
        }

        @Override
        public final Builder versionName(String versionName) {
            this.versionName = versionName;
            return this;
        }

        public final Collection<String> getPlatformTypes() {
            if (platformTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return platformTypes;
        }

        public final void setPlatformTypes(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
        }

        @Override
        public final Builder platformTypesWithStrings(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformTypesWithStrings(String... platformTypes) {
            platformTypesWithStrings(Arrays.asList(platformTypes));
            return this;
        }

        @Override
        public final Builder platformTypes(Collection<PlatformType> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copyEnumToString(platformTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformTypes(PlatformType... platformTypes) {
            platformTypes(Arrays.asList(platformTypes));
            return this;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final String getDocumentType() {
            return documentType;
        }

        public final void setDocumentType(String documentType) {
            this.documentType = documentType;
        }

        @Override
        public final Builder documentType(String documentType) {
            this.documentType = documentType;
            return this;
        }

        @Override
        public final Builder documentType(DocumentType documentType) {
            this.documentType(documentType == null ? null : documentType.toString());
            return this;
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        public final String getDocumentFormat() {
            return documentFormat;
        }

        public final void setDocumentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
        }

        @Override
        public final Builder documentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
            return this;
        }

        @Override
        public final Builder documentFormat(DocumentFormat documentFormat) {
            this.documentFormat(documentFormat == null ? null : documentFormat.toString());
            return this;
        }

        public final String getTargetType() {
            return targetType;
        }

        public final void setTargetType(String targetType) {
            this.targetType = targetType;
        }

        @Override
        public final Builder targetType(String targetType) {
            this.targetType = targetType;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DocumentRequires.Builder> getRequires() {
            List<DocumentRequires.Builder> result = DocumentRequiresListCopier.copyToBuilder(this.requires);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRequires(Collection<DocumentRequires.BuilderImpl> requires) {
            this.requires = DocumentRequiresListCopier.copyFromBuilder(requires);
        }

        @Override
        public final Builder requires(Collection<DocumentRequires> requires) {
            this.requires = DocumentRequiresListCopier.copy(requires);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requires(DocumentRequires... requires) {
            requires(Arrays.asList(requires));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requires(Consumer<DocumentRequires.Builder>... requires) {
            requires(Stream.of(requires).map(c -> DocumentRequires.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getReviewStatus() {
            return reviewStatus;
        }

        public final void setReviewStatus(String reviewStatus) {
            this.reviewStatus = reviewStatus;
        }

        @Override
        public final Builder reviewStatus(String reviewStatus) {
            this.reviewStatus = reviewStatus;
            return this;
        }

        @Override
        public final Builder reviewStatus(ReviewStatus reviewStatus) {
            this.reviewStatus(reviewStatus == null ? null : reviewStatus.toString());
            return this;
        }

        public final String getAuthor() {
            return author;
        }

        public final void setAuthor(String author) {
            this.author = author;
        }

        @Override
        public final Builder author(String author) {
            this.author = author;
            return this;
        }

        @Override
        public DocumentIdentifier build() {
            return new DocumentIdentifier(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
