/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sns.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The user-specified message attribute value. For string data types, the value attribute has the same restrictions on
 * the content as the message body. For more information, see <a
 * href="https://docs.aws.amazon.com/sns/latest/api/API_Publish.html">Publish</a>.
 * </p>
 * <p>
 * Name, type, and value must not be empty or null. In addition, the message body should not be empty or null. All parts
 * of the message attribute, including name, type, and value, are included in the message size restriction, which is
 * currently 256 KB (262,144 bytes). For more information, see <a
 * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html">Amazon SNS message attributes</a> and <a
 * href="https://docs.aws.amazon.com/sns/latest/dg/sms_publish-to-phone.html">Publishing to a mobile phone</a> in the
 * <i>Amazon SNS Developer Guide.</i>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MessageAttributeValue implements SdkPojo, Serializable,
        ToCopyableBuilder<MessageAttributeValue.Builder, MessageAttributeValue> {
    private static final SdkField<String> DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataType").getter(getter(MessageAttributeValue::dataType)).setter(setter(Builder::dataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataType").build()).build();

    private static final SdkField<String> STRING_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StringValue").getter(getter(MessageAttributeValue::stringValue)).setter(setter(Builder::stringValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StringValue").build()).build();

    private static final SdkField<SdkBytes> BINARY_VALUE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("BinaryValue").getter(getter(MessageAttributeValue::binaryValue)).setter(setter(Builder::binaryValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BinaryValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_TYPE_FIELD,
            STRING_VALUE_FIELD, BINARY_VALUE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String dataType;

    private final String stringValue;

    private final SdkBytes binaryValue;

    private MessageAttributeValue(BuilderImpl builder) {
        this.dataType = builder.dataType;
        this.stringValue = builder.stringValue;
        this.binaryValue = builder.binaryValue;
    }

    /**
     * <p>
     * Amazon SNS supports the following logical data types: String, String.Array, Number, and Binary. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes">Message
     * Attribute Data Types</a>.
     * </p>
     * 
     * @return Amazon SNS supports the following logical data types: String, String.Array, Number, and Binary. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes"
     *         >Message Attribute Data Types</a>.
     */
    public final String dataType() {
        return dataType;
    }

    /**
     * <p>
     * Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
     * href="https://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
     * </p>
     * 
     * @return Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
     *         href="https://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
     */
    public final String stringValue() {
        return stringValue;
    }

    /**
     * <p>
     * Binary type attributes can store any binary data, for example, compressed data, encrypted data, or images.
     * </p>
     * 
     * @return Binary type attributes can store any binary data, for example, compressed data, encrypted data, or
     *         images.
     */
    public final SdkBytes binaryValue() {
        return binaryValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dataType());
        hashCode = 31 * hashCode + Objects.hashCode(stringValue());
        hashCode = 31 * hashCode + Objects.hashCode(binaryValue());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageAttributeValue)) {
            return false;
        }
        MessageAttributeValue other = (MessageAttributeValue) obj;
        return Objects.equals(dataType(), other.dataType()) && Objects.equals(stringValue(), other.stringValue())
                && Objects.equals(binaryValue(), other.binaryValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MessageAttributeValue").add("DataType", dataType()).add("StringValue", stringValue())
                .add("BinaryValue", binaryValue()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DataType":
            return Optional.ofNullable(clazz.cast(dataType()));
        case "StringValue":
            return Optional.ofNullable(clazz.cast(stringValue()));
        case "BinaryValue":
            return Optional.ofNullable(clazz.cast(binaryValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DataType", DATA_TYPE_FIELD);
        map.put("StringValue", STRING_VALUE_FIELD);
        map.put("BinaryValue", BINARY_VALUE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MessageAttributeValue, T> g) {
        return obj -> g.apply((MessageAttributeValue) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MessageAttributeValue> {
        /**
         * <p>
         * Amazon SNS supports the following logical data types: String, String.Array, Number, and Binary. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes"
         * >Message Attribute Data Types</a>.
         * </p>
         * 
         * @param dataType
         *        Amazon SNS supports the following logical data types: String, String.Array, Number, and Binary. For
         *        more information, see <a href=
         *        "https://docs.aws.amazon.com/sns/latest/dg/SNSMessageAttributes.html#SNSMessageAttributes.DataTypes"
         *        >Message Attribute Data Types</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);

        /**
         * <p>
         * Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
         * href="https://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
         * </p>
         * 
         * @param stringValue
         *        Strings are Unicode with UTF8 binary encoding. For a list of code values, see <a
         *        href="https://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">ASCII Printable Characters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringValue(String stringValue);

        /**
         * <p>
         * Binary type attributes can store any binary data, for example, compressed data, encrypted data, or images.
         * </p>
         * 
         * @param binaryValue
         *        Binary type attributes can store any binary data, for example, compressed data, encrypted data, or
         *        images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder binaryValue(SdkBytes binaryValue);
    }

    static final class BuilderImpl implements Builder {
        private String dataType;

        private String stringValue;

        private SdkBytes binaryValue;

        private BuilderImpl() {
        }

        private BuilderImpl(MessageAttributeValue model) {
            dataType(model.dataType);
            stringValue(model.stringValue);
            binaryValue(model.binaryValue);
        }

        public final String getDataType() {
            return dataType;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        public final String getStringValue() {
            return stringValue;
        }

        public final void setStringValue(String stringValue) {
            this.stringValue = stringValue;
        }

        @Override
        public final Builder stringValue(String stringValue) {
            this.stringValue = stringValue;
            return this;
        }

        public final ByteBuffer getBinaryValue() {
            return binaryValue == null ? null : binaryValue.asByteBuffer();
        }

        public final void setBinaryValue(ByteBuffer binaryValue) {
            binaryValue(binaryValue == null ? null : SdkBytes.fromByteBuffer(binaryValue));
        }

        @Override
        public final Builder binaryValue(SdkBytes binaryValue) {
            this.binaryValue = binaryValue;
            return this;
        }

        @Override
        public MessageAttributeValue build() {
            return new MessageAttributeValue(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
