/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.signer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The hash algorithms that are available to a code-signing job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HashAlgorithmOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<HashAlgorithmOptions.Builder, HashAlgorithmOptions> {
    private static final SdkField<List<String>> ALLOWED_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("allowedValues")
            .getter(getter(HashAlgorithmOptions::allowedValuesAsStrings))
            .setter(setter(Builder::allowedValuesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowedValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DEFAULT_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("defaultValue").getter(getter(HashAlgorithmOptions::defaultValueAsString))
            .setter(setter(Builder::defaultValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOWED_VALUES_FIELD,
            DEFAULT_VALUE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> allowedValues;

    private final String defaultValue;

    private HashAlgorithmOptions(BuilderImpl builder) {
        this.allowedValues = builder.allowedValues;
        this.defaultValue = builder.defaultValue;
    }

    /**
     * <p>
     * The set of accepted hash algorithms allowed in a code-signing job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowedValues} method.
     * </p>
     * 
     * @return The set of accepted hash algorithms allowed in a code-signing job.
     */
    public final List<HashAlgorithm> allowedValues() {
        return HashAlgorithmsCopier.copyStringToEnum(allowedValues);
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowedValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowedValues() {
        return allowedValues != null && !(allowedValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of accepted hash algorithms allowed in a code-signing job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowedValues} method.
     * </p>
     * 
     * @return The set of accepted hash algorithms allowed in a code-signing job.
     */
    public final List<String> allowedValuesAsStrings() {
        return allowedValues;
    }

    /**
     * <p>
     * The default hash algorithm that is used in a code-signing job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #defaultValue} will
     * return {@link HashAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #defaultValueAsString}.
     * </p>
     * 
     * @return The default hash algorithm that is used in a code-signing job.
     * @see HashAlgorithm
     */
    public final HashAlgorithm defaultValue() {
        return HashAlgorithm.fromValue(defaultValue);
    }

    /**
     * <p>
     * The default hash algorithm that is used in a code-signing job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #defaultValue} will
     * return {@link HashAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #defaultValueAsString}.
     * </p>
     * 
     * @return The default hash algorithm that is used in a code-signing job.
     * @see HashAlgorithm
     */
    public final String defaultValueAsString() {
        return defaultValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowedValues() ? allowedValuesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(defaultValueAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HashAlgorithmOptions)) {
            return false;
        }
        HashAlgorithmOptions other = (HashAlgorithmOptions) obj;
        return hasAllowedValues() == other.hasAllowedValues()
                && Objects.equals(allowedValuesAsStrings(), other.allowedValuesAsStrings())
                && Objects.equals(defaultValueAsString(), other.defaultValueAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HashAlgorithmOptions")
                .add("AllowedValues", hasAllowedValues() ? allowedValuesAsStrings() : null)
                .add("DefaultValue", defaultValueAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "allowedValues":
            return Optional.ofNullable(clazz.cast(allowedValuesAsStrings()));
        case "defaultValue":
            return Optional.ofNullable(clazz.cast(defaultValueAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("allowedValues", ALLOWED_VALUES_FIELD);
        map.put("defaultValue", DEFAULT_VALUE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HashAlgorithmOptions, T> g) {
        return obj -> g.apply((HashAlgorithmOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HashAlgorithmOptions> {
        /**
         * <p>
         * The set of accepted hash algorithms allowed in a code-signing job.
         * </p>
         * 
         * @param allowedValues
         *        The set of accepted hash algorithms allowed in a code-signing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedValuesWithStrings(Collection<String> allowedValues);

        /**
         * <p>
         * The set of accepted hash algorithms allowed in a code-signing job.
         * </p>
         * 
         * @param allowedValues
         *        The set of accepted hash algorithms allowed in a code-signing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedValuesWithStrings(String... allowedValues);

        /**
         * <p>
         * The set of accepted hash algorithms allowed in a code-signing job.
         * </p>
         * 
         * @param allowedValues
         *        The set of accepted hash algorithms allowed in a code-signing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedValues(Collection<HashAlgorithm> allowedValues);

        /**
         * <p>
         * The set of accepted hash algorithms allowed in a code-signing job.
         * </p>
         * 
         * @param allowedValues
         *        The set of accepted hash algorithms allowed in a code-signing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedValues(HashAlgorithm... allowedValues);

        /**
         * <p>
         * The default hash algorithm that is used in a code-signing job.
         * </p>
         * 
         * @param defaultValue
         *        The default hash algorithm that is used in a code-signing job.
         * @see HashAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HashAlgorithm
         */
        Builder defaultValue(String defaultValue);

        /**
         * <p>
         * The default hash algorithm that is used in a code-signing job.
         * </p>
         * 
         * @param defaultValue
         *        The default hash algorithm that is used in a code-signing job.
         * @see HashAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HashAlgorithm
         */
        Builder defaultValue(HashAlgorithm defaultValue);
    }

    static final class BuilderImpl implements Builder {
        private List<String> allowedValues = DefaultSdkAutoConstructList.getInstance();

        private String defaultValue;

        private BuilderImpl() {
        }

        private BuilderImpl(HashAlgorithmOptions model) {
            allowedValuesWithStrings(model.allowedValues);
            defaultValue(model.defaultValue);
        }

        public final Collection<String> getAllowedValues() {
            if (allowedValues instanceof SdkAutoConstructList) {
                return null;
            }
            return allowedValues;
        }

        public final void setAllowedValues(Collection<String> allowedValues) {
            this.allowedValues = HashAlgorithmsCopier.copy(allowedValues);
        }

        @Override
        public final Builder allowedValuesWithStrings(Collection<String> allowedValues) {
            this.allowedValues = HashAlgorithmsCopier.copy(allowedValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedValuesWithStrings(String... allowedValues) {
            allowedValuesWithStrings(Arrays.asList(allowedValues));
            return this;
        }

        @Override
        public final Builder allowedValues(Collection<HashAlgorithm> allowedValues) {
            this.allowedValues = HashAlgorithmsCopier.copyEnumToString(allowedValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedValues(HashAlgorithm... allowedValues) {
            allowedValues(Arrays.asList(allowedValues));
            return this;
        }

        public final String getDefaultValue() {
            return defaultValue;
        }

        public final void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public final Builder defaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        @Override
        public final Builder defaultValue(HashAlgorithm defaultValue) {
            this.defaultValue(defaultValue == null ? null : defaultValue.toString());
            return this;
        }

        @Override
        public HashAlgorithmOptions build() {
            return new HashAlgorithmOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
