/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sfn;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.sfn.model.ActivityDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.ActivityLimitExceededException;
import software.amazon.awssdk.services.sfn.model.ActivityWorkerLimitExceededException;
import software.amazon.awssdk.services.sfn.model.CreateActivityRequest;
import software.amazon.awssdk.services.sfn.model.CreateActivityResponse;
import software.amazon.awssdk.services.sfn.model.CreateStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.CreateStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.DeleteActivityRequest;
import software.amazon.awssdk.services.sfn.model.DeleteActivityResponse;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.DescribeActivityRequest;
import software.amazon.awssdk.services.sfn.model.DescribeActivityResponse;
import software.amazon.awssdk.services.sfn.model.DescribeExecutionRequest;
import software.amazon.awssdk.services.sfn.model.DescribeExecutionResponse;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineForExecutionRequest;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineForExecutionResponse;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.ExecutionAlreadyExistsException;
import software.amazon.awssdk.services.sfn.model.ExecutionDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.ExecutionLimitExceededException;
import software.amazon.awssdk.services.sfn.model.GetActivityTaskRequest;
import software.amazon.awssdk.services.sfn.model.GetActivityTaskResponse;
import software.amazon.awssdk.services.sfn.model.GetExecutionHistoryRequest;
import software.amazon.awssdk.services.sfn.model.GetExecutionHistoryResponse;
import software.amazon.awssdk.services.sfn.model.InvalidArnException;
import software.amazon.awssdk.services.sfn.model.InvalidDefinitionException;
import software.amazon.awssdk.services.sfn.model.InvalidExecutionInputException;
import software.amazon.awssdk.services.sfn.model.InvalidLoggingConfigurationException;
import software.amazon.awssdk.services.sfn.model.InvalidNameException;
import software.amazon.awssdk.services.sfn.model.InvalidOutputException;
import software.amazon.awssdk.services.sfn.model.InvalidTokenException;
import software.amazon.awssdk.services.sfn.model.ListActivitiesRequest;
import software.amazon.awssdk.services.sfn.model.ListActivitiesResponse;
import software.amazon.awssdk.services.sfn.model.ListExecutionsRequest;
import software.amazon.awssdk.services.sfn.model.ListExecutionsResponse;
import software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest;
import software.amazon.awssdk.services.sfn.model.ListStateMachinesResponse;
import software.amazon.awssdk.services.sfn.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.sfn.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.sfn.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.sfn.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sfn.model.SendTaskFailureRequest;
import software.amazon.awssdk.services.sfn.model.SendTaskFailureResponse;
import software.amazon.awssdk.services.sfn.model.SendTaskHeartbeatRequest;
import software.amazon.awssdk.services.sfn.model.SendTaskHeartbeatResponse;
import software.amazon.awssdk.services.sfn.model.SendTaskSuccessRequest;
import software.amazon.awssdk.services.sfn.model.SendTaskSuccessResponse;
import software.amazon.awssdk.services.sfn.model.SfnException;
import software.amazon.awssdk.services.sfn.model.StartExecutionRequest;
import software.amazon.awssdk.services.sfn.model.StartExecutionResponse;
import software.amazon.awssdk.services.sfn.model.StateMachineAlreadyExistsException;
import software.amazon.awssdk.services.sfn.model.StateMachineDeletingException;
import software.amazon.awssdk.services.sfn.model.StateMachineDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.StateMachineLimitExceededException;
import software.amazon.awssdk.services.sfn.model.StateMachineTypeNotSupportedException;
import software.amazon.awssdk.services.sfn.model.StopExecutionRequest;
import software.amazon.awssdk.services.sfn.model.StopExecutionResponse;
import software.amazon.awssdk.services.sfn.model.TagResourceRequest;
import software.amazon.awssdk.services.sfn.model.TagResourceResponse;
import software.amazon.awssdk.services.sfn.model.TaskDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.TaskTimedOutException;
import software.amazon.awssdk.services.sfn.model.TooManyTagsException;
import software.amazon.awssdk.services.sfn.model.UntagResourceRequest;
import software.amazon.awssdk.services.sfn.model.UntagResourceResponse;
import software.amazon.awssdk.services.sfn.model.UpdateStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.UpdateStateMachineResponse;
import software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable;
import software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable;
import software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable;
import software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable;

/**
 * Service client for accessing AWS SFN. This can be created using the static {@link #builder()} method.
 *
 * <fullname>AWS Step Functions</fullname>
 * <p>
 * AWS Step Functions is a service that lets you coordinate the components of distributed applications and microservices
 * using visual workflows.
 * </p>
 * <p>
 * You can use Step Functions to build applications from individual components, each of which performs a discrete
 * function, or <i>task</i>, allowing you to scale and change applications quickly. Step Functions provides a console
 * that helps visualize the components of your application as a series of steps. Step Functions automatically triggers
 * and tracks each step, and retries steps when there are errors, so your application executes predictably and in the
 * right order every time. Step Functions logs the state of each step, so you can quickly diagnose and debug any issues.
 * </p>
 * <p>
 * Step Functions manages operations and underlying infrastructure to ensure your application is available at any scale.
 * You can run tasks on AWS, your own servers, or any system that has access to AWS. You can access and use Step
 * Functions using the console, the AWS SDKs, or an HTTP API. For more information about Step Functions, see the <i> <a
 * href="https://docs.aws.amazon.com/step-functions/latest/dg/welcome.html">AWS Step Functions Developer Guide</a> </i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface SfnClient extends SdkClient {
    String SERVICE_NAME = "states";

    /**
     * Create a {@link SfnClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static SfnClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link SfnClient}.
     */
    static SfnClientBuilder builder() {
        return new DefaultSfnClientBuilder();
    }

    /**
     * <p>
     * Creates an activity. An activity is a task that you write in any programming language and host on any machine
     * that has access to AWS Step Functions. Activities must poll Step Functions using the <code>GetActivityTask</code>
     * API action and respond using <code>SendTask*</code> API actions. This function lets Step Functions know the
     * existence of your activity and returns an identifier for use in a state machine and when polling from the
     * activity.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note> <note>
     * <p>
     * <code>CreateActivity</code> is an idempotent API. Subsequent requests won’t create a duplicate resource if it was
     * already created. <code>CreateActivity</code>'s idempotency check is based on the activity <code>name</code>. If a
     * following request has different <code>tags</code> values, Step Functions will ignore these differences and treat
     * it as an idempotent request of the previous. In this case, <code>tags</code> will not be updated, even if they
     * are different.
     * </p>
     * </note>
     *
     * @param createActivityRequest
     * @return Result of the CreateActivity operation returned by the service.
     * @throws ActivityLimitExceededException
     *         The maximum number of activities has been reached. Existing activities must be deleted before a new
     *         activity can be created.
     * @throws InvalidNameException
     *         The provided name is invalid.
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the AWS Step
     *         Functions Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.CreateActivity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/CreateActivity" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateActivityResponse createActivity(CreateActivityRequest createActivityRequest)
            throws ActivityLimitExceededException, InvalidNameException, TooManyTagsException, AwsServiceException,
            SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an activity. An activity is a task that you write in any programming language and host on any machine
     * that has access to AWS Step Functions. Activities must poll Step Functions using the <code>GetActivityTask</code>
     * API action and respond using <code>SendTask*</code> API actions. This function lets Step Functions know the
     * existence of your activity and returns an identifier for use in a state machine and when polling from the
     * activity.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note> <note>
     * <p>
     * <code>CreateActivity</code> is an idempotent API. Subsequent requests won’t create a duplicate resource if it was
     * already created. <code>CreateActivity</code>'s idempotency check is based on the activity <code>name</code>. If a
     * following request has different <code>tags</code> values, Step Functions will ignore these differences and treat
     * it as an idempotent request of the previous. In this case, <code>tags</code> will not be updated, even if they
     * are different.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateActivityRequest.Builder} avoiding the need to
     * create one manually via {@link CreateActivityRequest#builder()}
     * </p>
     *
     * @param createActivityRequest
     *        A {@link Consumer} that will call methods on {@link CreateActivityInput.Builder} to create a request.
     * @return Result of the CreateActivity operation returned by the service.
     * @throws ActivityLimitExceededException
     *         The maximum number of activities has been reached. Existing activities must be deleted before a new
     *         activity can be created.
     * @throws InvalidNameException
     *         The provided name is invalid.
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the AWS Step
     *         Functions Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.CreateActivity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/CreateActivity" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateActivityResponse createActivity(Consumer<CreateActivityRequest.Builder> createActivityRequest)
            throws ActivityLimitExceededException, InvalidNameException, TooManyTagsException, AwsServiceException,
            SdkClientException, SfnException {
        return createActivity(CreateActivityRequest.builder().applyMutation(createActivityRequest).build());
    }

    /**
     * <p>
     * Creates a state machine. A state machine consists of a collection of states that can do work (<code>Task</code>
     * states), determine to which states to transition next (<code>Choice</code> states), stop an execution with an
     * error (<code>Fail</code> states), and so on. State machines are specified using a JSON-based, structured
     * language. For more information, see <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html">Amazon States
     * Language</a> in the AWS Step Functions User Guide.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note> <note>
     * <p>
     * <code>CreateStateMachine</code> is an idempotent API. Subsequent requests won’t create a duplicate resource if it
     * was already created. <code>CreateStateMachine</code>'s idempotency check is based on the state machine
     * <code>name</code>, <code>definition</code>, <code>type</code>, and <code>LoggingConfiguration</code>. If a
     * following request has a different <code>roleArn</code> or <code>tags</code>, Step Functions will ignore these
     * differences and treat it as an idempotent request of the previous. In this case, <code>roleArn</code> and
     * <code>tags</code> will not be updated, even if they are different.
     * </p>
     * </note>
     *
     * @param createStateMachineRequest
     * @return Result of the CreateStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidDefinitionException
     *         The provided Amazon States Language definition is invalid.
     * @throws InvalidNameException
     *         The provided name is invalid.
     * @throws InvalidLoggingConfigurationException
     * @throws StateMachineAlreadyExistsException
     *         A state machine with the same name but a different definition or role ARN already exists.
     * @throws StateMachineDeletingException
     *         The specified state machine is being deleted.
     * @throws StateMachineLimitExceededException
     *         The maximum number of state machines has been reached. Existing state machines must be deleted before a
     *         new state machine can be created.
     * @throws StateMachineTypeNotSupportedException
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the AWS Step
     *         Functions Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.CreateStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/CreateStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateStateMachineResponse createStateMachine(CreateStateMachineRequest createStateMachineRequest)
            throws InvalidArnException, InvalidDefinitionException, InvalidNameException, InvalidLoggingConfigurationException,
            StateMachineAlreadyExistsException, StateMachineDeletingException, StateMachineLimitExceededException,
            StateMachineTypeNotSupportedException, TooManyTagsException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a state machine. A state machine consists of a collection of states that can do work (<code>Task</code>
     * states), determine to which states to transition next (<code>Choice</code> states), stop an execution with an
     * error (<code>Fail</code> states), and so on. State machines are specified using a JSON-based, structured
     * language. For more information, see <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html">Amazon States
     * Language</a> in the AWS Step Functions User Guide.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note> <note>
     * <p>
     * <code>CreateStateMachine</code> is an idempotent API. Subsequent requests won’t create a duplicate resource if it
     * was already created. <code>CreateStateMachine</code>'s idempotency check is based on the state machine
     * <code>name</code>, <code>definition</code>, <code>type</code>, and <code>LoggingConfiguration</code>. If a
     * following request has a different <code>roleArn</code> or <code>tags</code>, Step Functions will ignore these
     * differences and treat it as an idempotent request of the previous. In this case, <code>roleArn</code> and
     * <code>tags</code> will not be updated, even if they are different.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateStateMachineRequest.Builder} avoiding the
     * need to create one manually via {@link CreateStateMachineRequest#builder()}
     * </p>
     *
     * @param createStateMachineRequest
     *        A {@link Consumer} that will call methods on {@link CreateStateMachineInput.Builder} to create a request.
     * @return Result of the CreateStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidDefinitionException
     *         The provided Amazon States Language definition is invalid.
     * @throws InvalidNameException
     *         The provided name is invalid.
     * @throws InvalidLoggingConfigurationException
     * @throws StateMachineAlreadyExistsException
     *         A state machine with the same name but a different definition or role ARN already exists.
     * @throws StateMachineDeletingException
     *         The specified state machine is being deleted.
     * @throws StateMachineLimitExceededException
     *         The maximum number of state machines has been reached. Existing state machines must be deleted before a
     *         new state machine can be created.
     * @throws StateMachineTypeNotSupportedException
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the AWS Step
     *         Functions Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.CreateStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/CreateStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateStateMachineResponse createStateMachine(Consumer<CreateStateMachineRequest.Builder> createStateMachineRequest)
            throws InvalidArnException, InvalidDefinitionException, InvalidNameException, InvalidLoggingConfigurationException,
            StateMachineAlreadyExistsException, StateMachineDeletingException, StateMachineLimitExceededException,
            StateMachineTypeNotSupportedException, TooManyTagsException, AwsServiceException, SdkClientException, SfnException {
        return createStateMachine(CreateStateMachineRequest.builder().applyMutation(createStateMachineRequest).build());
    }

    /**
     * <p>
     * Deletes an activity.
     * </p>
     *
     * @param deleteActivityRequest
     * @return Result of the DeleteActivity operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DeleteActivity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteActivity" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteActivityResponse deleteActivity(DeleteActivityRequest deleteActivityRequest) throws InvalidArnException,
            AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes an activity.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteActivityRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteActivityRequest#builder()}
     * </p>
     *
     * @param deleteActivityRequest
     *        A {@link Consumer} that will call methods on {@link DeleteActivityInput.Builder} to create a request.
     * @return Result of the DeleteActivity operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DeleteActivity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteActivity" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteActivityResponse deleteActivity(Consumer<DeleteActivityRequest.Builder> deleteActivityRequest)
            throws InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        return deleteActivity(DeleteActivityRequest.builder().applyMutation(deleteActivityRequest).build());
    }

    /**
     * <p>
     * Deletes a state machine. This is an asynchronous operation: It sets the state machine's status to
     * <code>DELETING</code> and begins the deletion process.
     * </p>
     * <note>
     * <p>
     * For <code>EXPRESS</code>state machines, the deletion will happen eventually (usually less than a minute). Running
     * executions may emit logs after <code>DeleteStateMachine</code> API is called.
     * </p>
     * </note>
     *
     * @param deleteStateMachineRequest
     * @return Result of the DeleteStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DeleteStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteStateMachineResponse deleteStateMachine(DeleteStateMachineRequest deleteStateMachineRequest)
            throws InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a state machine. This is an asynchronous operation: It sets the state machine's status to
     * <code>DELETING</code> and begins the deletion process.
     * </p>
     * <note>
     * <p>
     * For <code>EXPRESS</code>state machines, the deletion will happen eventually (usually less than a minute). Running
     * executions may emit logs after <code>DeleteStateMachine</code> API is called.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteStateMachineRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteStateMachineRequest#builder()}
     * </p>
     *
     * @param deleteStateMachineRequest
     *        A {@link Consumer} that will call methods on {@link DeleteStateMachineInput.Builder} to create a request.
     * @return Result of the DeleteStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DeleteStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteStateMachineResponse deleteStateMachine(Consumer<DeleteStateMachineRequest.Builder> deleteStateMachineRequest)
            throws InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        return deleteStateMachine(DeleteStateMachineRequest.builder().applyMutation(deleteStateMachineRequest).build());
    }

    /**
     * <p>
     * Describes an activity.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param describeActivityRequest
     * @return Result of the DescribeActivity operation returned by the service.
     * @throws ActivityDoesNotExistException
     *         The specified activity does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeActivity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeActivity" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeActivityResponse describeActivity(DescribeActivityRequest describeActivityRequest)
            throws ActivityDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes an activity.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeActivityRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeActivityRequest#builder()}
     * </p>
     *
     * @param describeActivityRequest
     *        A {@link Consumer} that will call methods on {@link DescribeActivityInput.Builder} to create a request.
     * @return Result of the DescribeActivity operation returned by the service.
     * @throws ActivityDoesNotExistException
     *         The specified activity does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeActivity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeActivity" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeActivityResponse describeActivity(Consumer<DescribeActivityRequest.Builder> describeActivityRequest)
            throws ActivityDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        return describeActivity(DescribeActivityRequest.builder().applyMutation(describeActivityRequest).build());
    }

    /**
     * <p>
     * Describes an execution.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param describeExecutionRequest
     * @return Result of the DescribeExecution operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeExecutionResponse describeExecution(DescribeExecutionRequest describeExecutionRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes an execution.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeExecutionRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeExecutionRequest#builder()}
     * </p>
     *
     * @param describeExecutionRequest
     *        A {@link Consumer} that will call methods on {@link DescribeExecutionInput.Builder} to create a request.
     * @return Result of the DescribeExecution operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeExecutionResponse describeExecution(Consumer<DescribeExecutionRequest.Builder> describeExecutionRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        return describeExecution(DescribeExecutionRequest.builder().applyMutation(describeExecutionRequest).build());
    }

    /**
     * <p>
     * Describes a state machine.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param describeStateMachineRequest
     * @return Result of the DescribeStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeStateMachine" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeStateMachineResponse describeStateMachine(DescribeStateMachineRequest describeStateMachineRequest)
            throws InvalidArnException, StateMachineDoesNotExistException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a state machine.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeStateMachineRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeStateMachineRequest#builder()}
     * </p>
     *
     * @param describeStateMachineRequest
     *        A {@link Consumer} that will call methods on {@link DescribeStateMachineInput.Builder} to create a
     *        request.
     * @return Result of the DescribeStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeStateMachine" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeStateMachineResponse describeStateMachine(
            Consumer<DescribeStateMachineRequest.Builder> describeStateMachineRequest) throws InvalidArnException,
            StateMachineDoesNotExistException, AwsServiceException, SdkClientException, SfnException {
        return describeStateMachine(DescribeStateMachineRequest.builder().applyMutation(describeStateMachineRequest).build());
    }

    /**
     * <p>
     * Describes the state machine associated with a specific execution.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param describeStateMachineForExecutionRequest
     * @return Result of the DescribeStateMachineForExecution operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeStateMachineForExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeStateMachineForExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeStateMachineForExecutionResponse describeStateMachineForExecution(
            DescribeStateMachineForExecutionRequest describeStateMachineForExecutionRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the state machine associated with a specific execution.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeStateMachineForExecutionRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeStateMachineForExecutionRequest#builder()}
     * </p>
     *
     * @param describeStateMachineForExecutionRequest
     *        A {@link Consumer} that will call methods on {@link DescribeStateMachineForExecutionInput.Builder} to
     *        create a request.
     * @return Result of the DescribeStateMachineForExecution operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.DescribeStateMachineForExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeStateMachineForExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeStateMachineForExecutionResponse describeStateMachineForExecution(
            Consumer<DescribeStateMachineForExecutionRequest.Builder> describeStateMachineForExecutionRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        return describeStateMachineForExecution(DescribeStateMachineForExecutionRequest.builder()
                .applyMutation(describeStateMachineForExecutionRequest).build());
    }

    /**
     * <p>
     * Used by workers to retrieve a task (with the specified activity ARN) which has been scheduled for execution by a
     * running state machine. This initiates a long poll, where the service holds the HTTP connection open and responds
     * as soon as a task becomes available (i.e. an execution of a task of this type is needed.) The maximum time the
     * service holds on to the request before responding is 60 seconds. If no task is available within 60 seconds, the
     * poll returns a <code>taskToken</code> with a null string.
     * </p>
     * <important>
     * <p>
     * Workers should set their client side socket timeout to at least 65 seconds (5 seconds higher than the maximum
     * time the service may hold the poll request).
     * </p>
     * <p>
     * Polling with <code>GetActivityTask</code> can cause latency in some implementations. See <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/bp-activity-pollers.html">Avoid Latency When Polling
     * for Activity Tasks</a> in the Step Functions Developer Guide.
     * </p>
     * </important>
     *
     * @param getActivityTaskRequest
     * @return Result of the GetActivityTask operation returned by the service.
     * @throws ActivityDoesNotExistException
     *         The specified activity does not exist.
     * @throws ActivityWorkerLimitExceededException
     *         The maximum number of workers concurrently polling for activity tasks has been reached.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.GetActivityTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetActivityTask" target="_top">AWS API
     *      Documentation</a>
     */
    default GetActivityTaskResponse getActivityTask(GetActivityTaskRequest getActivityTaskRequest)
            throws ActivityDoesNotExistException, ActivityWorkerLimitExceededException, InvalidArnException, AwsServiceException,
            SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Used by workers to retrieve a task (with the specified activity ARN) which has been scheduled for execution by a
     * running state machine. This initiates a long poll, where the service holds the HTTP connection open and responds
     * as soon as a task becomes available (i.e. an execution of a task of this type is needed.) The maximum time the
     * service holds on to the request before responding is 60 seconds. If no task is available within 60 seconds, the
     * poll returns a <code>taskToken</code> with a null string.
     * </p>
     * <important>
     * <p>
     * Workers should set their client side socket timeout to at least 65 seconds (5 seconds higher than the maximum
     * time the service may hold the poll request).
     * </p>
     * <p>
     * Polling with <code>GetActivityTask</code> can cause latency in some implementations. See <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/bp-activity-pollers.html">Avoid Latency When Polling
     * for Activity Tasks</a> in the Step Functions Developer Guide.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetActivityTaskRequest.Builder} avoiding the need
     * to create one manually via {@link GetActivityTaskRequest#builder()}
     * </p>
     *
     * @param getActivityTaskRequest
     *        A {@link Consumer} that will call methods on {@link GetActivityTaskInput.Builder} to create a request.
     * @return Result of the GetActivityTask operation returned by the service.
     * @throws ActivityDoesNotExistException
     *         The specified activity does not exist.
     * @throws ActivityWorkerLimitExceededException
     *         The maximum number of workers concurrently polling for activity tasks has been reached.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.GetActivityTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetActivityTask" target="_top">AWS API
     *      Documentation</a>
     */
    default GetActivityTaskResponse getActivityTask(Consumer<GetActivityTaskRequest.Builder> getActivityTaskRequest)
            throws ActivityDoesNotExistException, ActivityWorkerLimitExceededException, InvalidArnException, AwsServiceException,
            SdkClientException, SfnException {
        return getActivityTask(GetActivityTaskRequest.builder().applyMutation(getActivityTaskRequest).build());
    }

    /**
     * <p>
     * Returns the history of the specified execution as a list of events. By default, the results are returned in
     * ascending order of the <code>timeStamp</code> of the events. Use the <code>reverseOrder</code> parameter to get
     * the latest events first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param getExecutionHistoryRequest
     * @return Result of the GetExecutionHistory operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.GetExecutionHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetExecutionHistory" target="_top">AWS API
     *      Documentation</a>
     */
    default GetExecutionHistoryResponse getExecutionHistory(GetExecutionHistoryRequest getExecutionHistoryRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, InvalidTokenException, AwsServiceException,
            SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the history of the specified execution as a list of events. By default, the results are returned in
     * ascending order of the <code>timeStamp</code> of the events. Use the <code>reverseOrder</code> parameter to get
     * the latest events first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetExecutionHistoryRequest.Builder} avoiding the
     * need to create one manually via {@link GetExecutionHistoryRequest#builder()}
     * </p>
     *
     * @param getExecutionHistoryRequest
     *        A {@link Consumer} that will call methods on {@link GetExecutionHistoryInput.Builder} to create a request.
     * @return Result of the GetExecutionHistory operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.GetExecutionHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetExecutionHistory" target="_top">AWS API
     *      Documentation</a>
     */
    default GetExecutionHistoryResponse getExecutionHistory(
            Consumer<GetExecutionHistoryRequest.Builder> getExecutionHistoryRequest) throws ExecutionDoesNotExistException,
            InvalidArnException, InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        return getExecutionHistory(GetExecutionHistoryRequest.builder().applyMutation(getExecutionHistoryRequest).build());
    }

    /**
     * <p>
     * Returns the history of the specified execution as a list of events. By default, the results are returned in
     * ascending order of the <code>timeStamp</code> of the events. Use the <code>reverseOrder</code> parameter to get
     * the latest events first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getExecutionHistory(software.amazon.awssdk.services.sfn.model.GetExecutionHistoryRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable responses = client.getExecutionHistoryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable responses = client
     *             .getExecutionHistoryPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.GetExecutionHistoryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable responses = client.getExecutionHistoryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getExecutionHistory(software.amazon.awssdk.services.sfn.model.GetExecutionHistoryRequest)} operation.</b>
     * </p>
     *
     * @param getExecutionHistoryRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.GetExecutionHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetExecutionHistory" target="_top">AWS API
     *      Documentation</a>
     */
    default GetExecutionHistoryIterable getExecutionHistoryPaginator(GetExecutionHistoryRequest getExecutionHistoryRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, InvalidTokenException, AwsServiceException,
            SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the history of the specified execution as a list of events. By default, the results are returned in
     * ascending order of the <code>timeStamp</code> of the events. Use the <code>reverseOrder</code> parameter to get
     * the latest events first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getExecutionHistory(software.amazon.awssdk.services.sfn.model.GetExecutionHistoryRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable responses = client.getExecutionHistoryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable responses = client
     *             .getExecutionHistoryPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.GetExecutionHistoryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.GetExecutionHistoryIterable responses = client.getExecutionHistoryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getExecutionHistory(software.amazon.awssdk.services.sfn.model.GetExecutionHistoryRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link GetExecutionHistoryRequest.Builder} avoiding the
     * need to create one manually via {@link GetExecutionHistoryRequest#builder()}
     * </p>
     *
     * @param getExecutionHistoryRequest
     *        A {@link Consumer} that will call methods on {@link GetExecutionHistoryInput.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.GetExecutionHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetExecutionHistory" target="_top">AWS API
     *      Documentation</a>
     */
    default GetExecutionHistoryIterable getExecutionHistoryPaginator(
            Consumer<GetExecutionHistoryRequest.Builder> getExecutionHistoryRequest) throws ExecutionDoesNotExistException,
            InvalidArnException, InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        return getExecutionHistoryPaginator(GetExecutionHistoryRequest.builder().applyMutation(getExecutionHistoryRequest)
                .build());
    }

    /**
     * <p>
     * Lists the existing activities.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @return Result of the ListActivities operation returned by the service.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListActivities
     * @see #listActivities(ListActivitiesRequest)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListActivities" target="_top">AWS API
     *      Documentation</a>
     */
    default ListActivitiesResponse listActivities() throws InvalidTokenException, AwsServiceException, SdkClientException,
            SfnException {
        return listActivities(ListActivitiesRequest.builder().build());
    }

    /**
     * <p>
     * Lists the existing activities.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param listActivitiesRequest
     * @return Result of the ListActivities operation returned by the service.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListActivities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListActivities" target="_top">AWS API
     *      Documentation</a>
     */
    default ListActivitiesResponse listActivities(ListActivitiesRequest listActivitiesRequest) throws InvalidTokenException,
            AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the existing activities.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListActivitiesRequest.Builder} avoiding the need to
     * create one manually via {@link ListActivitiesRequest#builder()}
     * </p>
     *
     * @param listActivitiesRequest
     *        A {@link Consumer} that will call methods on {@link ListActivitiesInput.Builder} to create a request.
     * @return Result of the ListActivities operation returned by the service.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListActivities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListActivities" target="_top">AWS API
     *      Documentation</a>
     */
    default ListActivitiesResponse listActivities(Consumer<ListActivitiesRequest.Builder> listActivitiesRequest)
            throws InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        return listActivities(ListActivitiesRequest.builder().applyMutation(listActivitiesRequest).build());
    }

    /**
     * <p>
     * Lists the existing activities.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of {@link #listActivities(software.amazon.awssdk.services.sfn.model.ListActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listActivities(software.amazon.awssdk.services.sfn.model.ListActivitiesRequest)} operation.</b>
     * </p>
     *
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListActivities
     * @see #listActivitiesPaginator(ListActivitiesRequest)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListActivities" target="_top">AWS API
     *      Documentation</a>
     */
    default ListActivitiesIterable listActivitiesPaginator() throws InvalidTokenException, AwsServiceException,
            SdkClientException, SfnException {
        return listActivitiesPaginator(ListActivitiesRequest.builder().build());
    }

    /**
     * <p>
     * Lists the existing activities.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of {@link #listActivities(software.amazon.awssdk.services.sfn.model.ListActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listActivities(software.amazon.awssdk.services.sfn.model.ListActivitiesRequest)} operation.</b>
     * </p>
     *
     * @param listActivitiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListActivities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListActivities" target="_top">AWS API
     *      Documentation</a>
     */
    default ListActivitiesIterable listActivitiesPaginator(ListActivitiesRequest listActivitiesRequest)
            throws InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the existing activities.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of {@link #listActivities(software.amazon.awssdk.services.sfn.model.ListActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListActivitiesIterable responses = client.listActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listActivities(software.amazon.awssdk.services.sfn.model.ListActivitiesRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListActivitiesRequest.Builder} avoiding the need to
     * create one manually via {@link ListActivitiesRequest#builder()}
     * </p>
     *
     * @param listActivitiesRequest
     *        A {@link Consumer} that will call methods on {@link ListActivitiesInput.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListActivities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListActivities" target="_top">AWS API
     *      Documentation</a>
     */
    default ListActivitiesIterable listActivitiesPaginator(Consumer<ListActivitiesRequest.Builder> listActivitiesRequest)
            throws InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        return listActivitiesPaginator(ListActivitiesRequest.builder().applyMutation(listActivitiesRequest).build());
    }

    /**
     * <p>
     * Lists the executions of a state machine that meet the filtering criteria. Results are sorted by time, with the
     * most recent execution first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param listExecutionsRequest
     * @return Result of the ListExecutions operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws StateMachineTypeNotSupportedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListExecutionsResponse listExecutions(ListExecutionsRequest listExecutionsRequest) throws InvalidArnException,
            InvalidTokenException, StateMachineDoesNotExistException, StateMachineTypeNotSupportedException, AwsServiceException,
            SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the executions of a state machine that meet the filtering criteria. Results are sorted by time, with the
     * most recent execution first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListExecutionsRequest.Builder} avoiding the need to
     * create one manually via {@link ListExecutionsRequest#builder()}
     * </p>
     *
     * @param listExecutionsRequest
     *        A {@link Consumer} that will call methods on {@link ListExecutionsInput.Builder} to create a request.
     * @return Result of the ListExecutions operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws StateMachineTypeNotSupportedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListExecutionsResponse listExecutions(Consumer<ListExecutionsRequest.Builder> listExecutionsRequest)
            throws InvalidArnException, InvalidTokenException, StateMachineDoesNotExistException,
            StateMachineTypeNotSupportedException, AwsServiceException, SdkClientException, SfnException {
        return listExecutions(ListExecutionsRequest.builder().applyMutation(listExecutionsRequest).build());
    }

    /**
     * <p>
     * Lists the executions of a state machine that meet the filtering criteria. Results are sorted by time, with the
     * most recent execution first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listExecutions(software.amazon.awssdk.services.sfn.model.ListExecutionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable responses = client.listExecutionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable responses = client.listExecutionsPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListExecutionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable responses = client.listExecutionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExecutions(software.amazon.awssdk.services.sfn.model.ListExecutionsRequest)} operation.</b>
     * </p>
     *
     * @param listExecutionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws StateMachineTypeNotSupportedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListExecutionsIterable listExecutionsPaginator(ListExecutionsRequest listExecutionsRequest)
            throws InvalidArnException, InvalidTokenException, StateMachineDoesNotExistException,
            StateMachineTypeNotSupportedException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the executions of a state machine that meet the filtering criteria. Results are sorted by time, with the
     * most recent execution first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listExecutions(software.amazon.awssdk.services.sfn.model.ListExecutionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable responses = client.listExecutionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable responses = client.listExecutionsPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListExecutionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListExecutionsIterable responses = client.listExecutionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExecutions(software.amazon.awssdk.services.sfn.model.ListExecutionsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListExecutionsRequest.Builder} avoiding the need to
     * create one manually via {@link ListExecutionsRequest#builder()}
     * </p>
     *
     * @param listExecutionsRequest
     *        A {@link Consumer} that will call methods on {@link ListExecutionsInput.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws StateMachineTypeNotSupportedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListExecutionsIterable listExecutionsPaginator(Consumer<ListExecutionsRequest.Builder> listExecutionsRequest)
            throws InvalidArnException, InvalidTokenException, StateMachineDoesNotExistException,
            StateMachineTypeNotSupportedException, AwsServiceException, SdkClientException, SfnException {
        return listExecutionsPaginator(ListExecutionsRequest.builder().applyMutation(listExecutionsRequest).build());
    }

    /**
     * <p>
     * Lists the existing state machines.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @return Result of the ListStateMachines operation returned by the service.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListStateMachines
     * @see #listStateMachines(ListStateMachinesRequest)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachines" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStateMachinesResponse listStateMachines() throws InvalidTokenException, AwsServiceException, SdkClientException,
            SfnException {
        return listStateMachines(ListStateMachinesRequest.builder().build());
    }

    /**
     * <p>
     * Lists the existing state machines.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param listStateMachinesRequest
     * @return Result of the ListStateMachines operation returned by the service.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListStateMachines
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachines" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStateMachinesResponse listStateMachines(ListStateMachinesRequest listStateMachinesRequest)
            throws InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the existing state machines.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStateMachinesRequest.Builder} avoiding the need
     * to create one manually via {@link ListStateMachinesRequest#builder()}
     * </p>
     *
     * @param listStateMachinesRequest
     *        A {@link Consumer} that will call methods on {@link ListStateMachinesInput.Builder} to create a request.
     * @return Result of the ListStateMachines operation returned by the service.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListStateMachines
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachines" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStateMachinesResponse listStateMachines(Consumer<ListStateMachinesRequest.Builder> listStateMachinesRequest)
            throws InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        return listStateMachines(ListStateMachinesRequest.builder().applyMutation(listStateMachinesRequest).build());
    }

    /**
     * <p>
     * Lists the existing state machines.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listStateMachines(software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client.listStateMachinesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client
     *             .listStateMachinesPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListStateMachinesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client.listStateMachinesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStateMachines(software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest)} operation.</b>
     * </p>
     *
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListStateMachines
     * @see #listStateMachinesPaginator(ListStateMachinesRequest)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachines" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStateMachinesIterable listStateMachinesPaginator() throws InvalidTokenException, AwsServiceException,
            SdkClientException, SfnException {
        return listStateMachinesPaginator(ListStateMachinesRequest.builder().build());
    }

    /**
     * <p>
     * Lists the existing state machines.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listStateMachines(software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client.listStateMachinesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client
     *             .listStateMachinesPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListStateMachinesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client.listStateMachinesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStateMachines(software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest)} operation.</b>
     * </p>
     *
     * @param listStateMachinesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListStateMachines
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachines" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStateMachinesIterable listStateMachinesPaginator(ListStateMachinesRequest listStateMachinesRequest)
            throws InvalidTokenException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the existing state machines.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listStateMachines(software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client.listStateMachinesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client
     *             .listStateMachinesPaginator(request);
     *     for (software.amazon.awssdk.services.sfn.model.ListStateMachinesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sfn.paginators.ListStateMachinesIterable responses = client.listStateMachinesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStateMachines(software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStateMachinesRequest.Builder} avoiding the need
     * to create one manually via {@link ListStateMachinesRequest#builder()}
     * </p>
     *
     * @param listStateMachinesRequest
     *        A {@link Consumer} that will call methods on {@link ListStateMachinesInput.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListStateMachines
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachines" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStateMachinesIterable listStateMachinesPaginator(
            Consumer<ListStateMachinesRequest.Builder> listStateMachinesRequest) throws InvalidTokenException,
            AwsServiceException, SdkClientException, SfnException {
        return listStateMachinesPaginator(ListStateMachinesRequest.builder().applyMutation(listStateMachinesRequest).build());
    }

    /**
     * <p>
     * List tags for a given resource.
     * </p>
     * <p>
     * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws ResourceNotFoundException
     *         Could not find the referenced resource. Only state machine and activity ARNs are supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidArnException, ResourceNotFoundException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List tags for a given resource.
     * </p>
     * <p>
     * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceInput.Builder} to create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws ResourceNotFoundException
     *         Could not find the referenced resource. Only state machine and activity ARNs are supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws InvalidArnException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, SfnException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report that the task identified by the <code>taskToken</code> failed.
     * </p>
     *
     * @param sendTaskFailureRequest
     * @return Result of the SendTaskFailure operation returned by the service.
     * @throws TaskDoesNotExistException
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws TaskTimedOutException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.SendTaskFailure
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskFailure" target="_top">AWS API
     *      Documentation</a>
     */
    default SendTaskFailureResponse sendTaskFailure(SendTaskFailureRequest sendTaskFailureRequest)
            throws TaskDoesNotExistException, InvalidTokenException, TaskTimedOutException, AwsServiceException,
            SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report that the task identified by the <code>taskToken</code> failed.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SendTaskFailureRequest.Builder} avoiding the need
     * to create one manually via {@link SendTaskFailureRequest#builder()}
     * </p>
     *
     * @param sendTaskFailureRequest
     *        A {@link Consumer} that will call methods on {@link SendTaskFailureInput.Builder} to create a request.
     * @return Result of the SendTaskFailure operation returned by the service.
     * @throws TaskDoesNotExistException
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws TaskTimedOutException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.SendTaskFailure
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskFailure" target="_top">AWS API
     *      Documentation</a>
     */
    default SendTaskFailureResponse sendTaskFailure(Consumer<SendTaskFailureRequest.Builder> sendTaskFailureRequest)
            throws TaskDoesNotExistException, InvalidTokenException, TaskTimedOutException, AwsServiceException,
            SdkClientException, SfnException {
        return sendTaskFailure(SendTaskFailureRequest.builder().applyMutation(sendTaskFailureRequest).build());
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report to Step Functions that the task represented by the specified
     * <code>taskToken</code> is still making progress. This action resets the <code>Heartbeat</code> clock. The
     * <code>Heartbeat</code> threshold is specified in the state machine's Amazon States Language definition (
     * <code>HeartbeatSeconds</code>). This action does not in itself create an event in the execution history. However,
     * if the task times out, the execution history contains an <code>ActivityTimedOut</code> entry for activities, or a
     * <code>TaskTimedOut</code> entry for for tasks using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-sync">job run</a> or
     * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token">
     * callback</a> pattern.
     * </p>
     * <note>
     * <p>
     * The <code>Timeout</code> of a task, defined in the state machine's Amazon States Language definition, is its
     * maximum allowed duration, regardless of the number of <a>SendTaskHeartbeat</a> requests received. Use
     * <code>HeartbeatSeconds</code> to configure the timeout interval for heartbeats.
     * </p>
     * </note>
     *
     * @param sendTaskHeartbeatRequest
     * @return Result of the SendTaskHeartbeat operation returned by the service.
     * @throws TaskDoesNotExistException
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws TaskTimedOutException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.SendTaskHeartbeat
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskHeartbeat" target="_top">AWS API
     *      Documentation</a>
     */
    default SendTaskHeartbeatResponse sendTaskHeartbeat(SendTaskHeartbeatRequest sendTaskHeartbeatRequest)
            throws TaskDoesNotExistException, InvalidTokenException, TaskTimedOutException, AwsServiceException,
            SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report to Step Functions that the task represented by the specified
     * <code>taskToken</code> is still making progress. This action resets the <code>Heartbeat</code> clock. The
     * <code>Heartbeat</code> threshold is specified in the state machine's Amazon States Language definition (
     * <code>HeartbeatSeconds</code>). This action does not in itself create an event in the execution history. However,
     * if the task times out, the execution history contains an <code>ActivityTimedOut</code> entry for activities, or a
     * <code>TaskTimedOut</code> entry for for tasks using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-sync">job run</a> or
     * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token">
     * callback</a> pattern.
     * </p>
     * <note>
     * <p>
     * The <code>Timeout</code> of a task, defined in the state machine's Amazon States Language definition, is its
     * maximum allowed duration, regardless of the number of <a>SendTaskHeartbeat</a> requests received. Use
     * <code>HeartbeatSeconds</code> to configure the timeout interval for heartbeats.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SendTaskHeartbeatRequest.Builder} avoiding the need
     * to create one manually via {@link SendTaskHeartbeatRequest#builder()}
     * </p>
     *
     * @param sendTaskHeartbeatRequest
     *        A {@link Consumer} that will call methods on {@link SendTaskHeartbeatInput.Builder} to create a request.
     * @return Result of the SendTaskHeartbeat operation returned by the service.
     * @throws TaskDoesNotExistException
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws TaskTimedOutException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.SendTaskHeartbeat
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskHeartbeat" target="_top">AWS API
     *      Documentation</a>
     */
    default SendTaskHeartbeatResponse sendTaskHeartbeat(Consumer<SendTaskHeartbeatRequest.Builder> sendTaskHeartbeatRequest)
            throws TaskDoesNotExistException, InvalidTokenException, TaskTimedOutException, AwsServiceException,
            SdkClientException, SfnException {
        return sendTaskHeartbeat(SendTaskHeartbeatRequest.builder().applyMutation(sendTaskHeartbeatRequest).build());
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report that the task identified by the <code>taskToken</code> completed successfully.
     * </p>
     *
     * @param sendTaskSuccessRequest
     * @return Result of the SendTaskSuccess operation returned by the service.
     * @throws TaskDoesNotExistException
     * @throws InvalidOutputException
     *         The provided JSON output data is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws TaskTimedOutException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.SendTaskSuccess
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskSuccess" target="_top">AWS API
     *      Documentation</a>
     */
    default SendTaskSuccessResponse sendTaskSuccess(SendTaskSuccessRequest sendTaskSuccessRequest)
            throws TaskDoesNotExistException, InvalidOutputException, InvalidTokenException, TaskTimedOutException,
            AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report that the task identified by the <code>taskToken</code> completed successfully.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SendTaskSuccessRequest.Builder} avoiding the need
     * to create one manually via {@link SendTaskSuccessRequest#builder()}
     * </p>
     *
     * @param sendTaskSuccessRequest
     *        A {@link Consumer} that will call methods on {@link SendTaskSuccessInput.Builder} to create a request.
     * @return Result of the SendTaskSuccess operation returned by the service.
     * @throws TaskDoesNotExistException
     * @throws InvalidOutputException
     *         The provided JSON output data is invalid.
     * @throws InvalidTokenException
     *         The provided token is invalid.
     * @throws TaskTimedOutException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.SendTaskSuccess
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskSuccess" target="_top">AWS API
     *      Documentation</a>
     */
    default SendTaskSuccessResponse sendTaskSuccess(Consumer<SendTaskSuccessRequest.Builder> sendTaskSuccessRequest)
            throws TaskDoesNotExistException, InvalidOutputException, InvalidTokenException, TaskTimedOutException,
            AwsServiceException, SdkClientException, SfnException {
        return sendTaskSuccess(SendTaskSuccessRequest.builder().applyMutation(sendTaskSuccessRequest).build());
    }

    /**
     * <p>
     * Starts a state machine execution.
     * </p>
     * <note>
     * <p>
     * <code>StartExecution</code> is idempotent. If <code>StartExecution</code> is called with the same name and input
     * as a running execution, the call will succeed and return the same response as the original request. If the
     * execution is closed or if the input is different, it will return a 400 <code>ExecutionAlreadyExists</code> error.
     * Names can be reused after 90 days.
     * </p>
     * </note>
     *
     * @param startExecutionRequest
     * @return Result of the StartExecution operation returned by the service.
     * @throws ExecutionLimitExceededException
     *         The maximum number of running executions has been reached. Running executions must end or be stopped
     *         before a new execution can be started.
     * @throws ExecutionAlreadyExistsException
     *         The execution has the same <code>name</code> as another execution (but a different <code>input</code>
     *         ).</p> <note>
     *         <p>
     *         Executions with the same <code>name</code> and <code>input</code> are considered idempotent.
     *         </p>
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidExecutionInputException
     *         The provided JSON input data is invalid.
     * @throws InvalidNameException
     *         The provided name is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws StateMachineDeletingException
     *         The specified state machine is being deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.StartExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/StartExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default StartExecutionResponse startExecution(StartExecutionRequest startExecutionRequest)
            throws ExecutionLimitExceededException, ExecutionAlreadyExistsException, InvalidArnException,
            InvalidExecutionInputException, InvalidNameException, StateMachineDoesNotExistException,
            StateMachineDeletingException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts a state machine execution.
     * </p>
     * <note>
     * <p>
     * <code>StartExecution</code> is idempotent. If <code>StartExecution</code> is called with the same name and input
     * as a running execution, the call will succeed and return the same response as the original request. If the
     * execution is closed or if the input is different, it will return a 400 <code>ExecutionAlreadyExists</code> error.
     * Names can be reused after 90 days.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartExecutionRequest.Builder} avoiding the need to
     * create one manually via {@link StartExecutionRequest#builder()}
     * </p>
     *
     * @param startExecutionRequest
     *        A {@link Consumer} that will call methods on {@link StartExecutionInput.Builder} to create a request.
     * @return Result of the StartExecution operation returned by the service.
     * @throws ExecutionLimitExceededException
     *         The maximum number of running executions has been reached. Running executions must end or be stopped
     *         before a new execution can be started.
     * @throws ExecutionAlreadyExistsException
     *         The execution has the same <code>name</code> as another execution (but a different <code>input</code>
     *         ).</p> <note>
     *         <p>
     *         Executions with the same <code>name</code> and <code>input</code> are considered idempotent.
     *         </p>
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidExecutionInputException
     *         The provided JSON input data is invalid.
     * @throws InvalidNameException
     *         The provided name is invalid.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws StateMachineDeletingException
     *         The specified state machine is being deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.StartExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/StartExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default StartExecutionResponse startExecution(Consumer<StartExecutionRequest.Builder> startExecutionRequest)
            throws ExecutionLimitExceededException, ExecutionAlreadyExistsException, InvalidArnException,
            InvalidExecutionInputException, InvalidNameException, StateMachineDoesNotExistException,
            StateMachineDeletingException, AwsServiceException, SdkClientException, SfnException {
        return startExecution(StartExecutionRequest.builder().applyMutation(startExecutionRequest).build());
    }

    /**
     * <p>
     * Stops an execution.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param stopExecutionRequest
     * @return Result of the StopExecution operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.StopExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/StopExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default StopExecutionResponse stopExecution(StopExecutionRequest stopExecutionRequest) throws ExecutionDoesNotExistException,
            InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Stops an execution.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StopExecutionRequest.Builder} avoiding the need to
     * create one manually via {@link StopExecutionRequest#builder()}
     * </p>
     *
     * @param stopExecutionRequest
     *        A {@link Consumer} that will call methods on {@link StopExecutionInput.Builder} to create a request.
     * @return Result of the StopExecution operation returned by the service.
     * @throws ExecutionDoesNotExistException
     *         The specified execution does not exist.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.StopExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/StopExecution" target="_top">AWS API
     *      Documentation</a>
     */
    default StopExecutionResponse stopExecution(Consumer<StopExecutionRequest.Builder> stopExecutionRequest)
            throws ExecutionDoesNotExistException, InvalidArnException, AwsServiceException, SdkClientException, SfnException {
        return stopExecution(StopExecutionRequest.builder().applyMutation(stopExecutionRequest).build());
    }

    /**
     * <p>
     * Add a tag to a Step Functions resource.
     * </p>
     * <p>
     * An array of key-value pairs. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>AWS Billing and Cost Management User Guide</i>, and <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using IAM
     * Tags</a>.
     * </p>
     * <p>
     * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws ResourceNotFoundException
     *         Could not find the referenced resource. Only state machine and activity ARNs are supported.
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the AWS Step
     *         Functions Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidArnException,
            ResourceNotFoundException, TooManyTagsException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Add a tag to a Step Functions resource.
     * </p>
     * <p>
     * An array of key-value pairs. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>AWS Billing and Cost Management User Guide</i>, and <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using IAM
     * Tags</a>.
     * </p>
     * <p>
     * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceInput.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws ResourceNotFoundException
     *         Could not find the referenced resource. Only state machine and activity ARNs are supported.
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the AWS Step
     *         Functions Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) throws InvalidArnException,
            ResourceNotFoundException, TooManyTagsException, AwsServiceException, SdkClientException, SfnException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Remove a tag from a Step Functions resource
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws ResourceNotFoundException
     *         Could not find the referenced resource. Only state machine and activity ARNs are supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidArnException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Remove a tag from a Step Functions resource
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceInput.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws ResourceNotFoundException
     *         Could not find the referenced resource. Only state machine and activity ARNs are supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws InvalidArnException, ResourceNotFoundException, AwsServiceException, SdkClientException, SfnException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates an existing state machine by modifying its <code>definition</code>, <code>roleArn</code>, or
     * <code>loggingConfiguration</code>. Running executions will continue to use the previous <code>definition</code>
     * and <code>roleArn</code>. You must include at least one of <code>definition</code> or <code>roleArn</code> or you
     * will receive a <code>MissingRequiredParameter</code> error.
     * </p>
     * <note>
     * <p>
     * All <code>StartExecution</code> calls within a few seconds will use the updated <code>definition</code> and
     * <code>roleArn</code>. Executions started immediately after calling <code>UpdateStateMachine</code> may use the
     * previous state machine <code>definition</code> and <code>roleArn</code>.
     * </p>
     * </note>
     *
     * @param updateStateMachineRequest
     * @return Result of the UpdateStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidDefinitionException
     *         The provided Amazon States Language definition is invalid.
     * @throws InvalidLoggingConfigurationException
     * @throws MissingRequiredParameterException
     *         Request is missing a required parameter. This error occurs if both <code>definition</code> and
     *         <code>roleArn</code> are not specified.
     * @throws StateMachineDeletingException
     *         The specified state machine is being deleted.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.UpdateStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UpdateStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateStateMachineResponse updateStateMachine(UpdateStateMachineRequest updateStateMachineRequest)
            throws InvalidArnException, InvalidDefinitionException, InvalidLoggingConfigurationException,
            MissingRequiredParameterException, StateMachineDeletingException, StateMachineDoesNotExistException,
            AwsServiceException, SdkClientException, SfnException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an existing state machine by modifying its <code>definition</code>, <code>roleArn</code>, or
     * <code>loggingConfiguration</code>. Running executions will continue to use the previous <code>definition</code>
     * and <code>roleArn</code>. You must include at least one of <code>definition</code> or <code>roleArn</code> or you
     * will receive a <code>MissingRequiredParameter</code> error.
     * </p>
     * <note>
     * <p>
     * All <code>StartExecution</code> calls within a few seconds will use the updated <code>definition</code> and
     * <code>roleArn</code>. Executions started immediately after calling <code>UpdateStateMachine</code> may use the
     * previous state machine <code>definition</code> and <code>roleArn</code>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateStateMachineRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateStateMachineRequest#builder()}
     * </p>
     *
     * @param updateStateMachineRequest
     *        A {@link Consumer} that will call methods on {@link UpdateStateMachineInput.Builder} to create a request.
     * @return Result of the UpdateStateMachine operation returned by the service.
     * @throws InvalidArnException
     *         The provided Amazon Resource Name (ARN) is invalid.
     * @throws InvalidDefinitionException
     *         The provided Amazon States Language definition is invalid.
     * @throws InvalidLoggingConfigurationException
     * @throws MissingRequiredParameterException
     *         Request is missing a required parameter. This error occurs if both <code>definition</code> and
     *         <code>roleArn</code> are not specified.
     * @throws StateMachineDeletingException
     *         The specified state machine is being deleted.
     * @throws StateMachineDoesNotExistException
     *         The specified state machine does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SfnException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SfnClient.UpdateStateMachine
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UpdateStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateStateMachineResponse updateStateMachine(Consumer<UpdateStateMachineRequest.Builder> updateStateMachineRequest)
            throws InvalidArnException, InvalidDefinitionException, InvalidLoggingConfigurationException,
            MissingRequiredParameterException, StateMachineDeletingException, StateMachineDoesNotExistException,
            AwsServiceException, SdkClientException, SfnException {
        return updateStateMachine(UpdateStateMachineRequest.builder().applyMutation(updateStateMachineRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("states");
    }
}
