/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to reorder the receipt rules within a receipt rule set. You use receipt rule sets to receive
 * email with Amazon SES. For more information, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-concepts.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReorderReceiptRuleSetRequest extends SesRequest implements
        ToCopyableBuilder<ReorderReceiptRuleSetRequest.Builder, ReorderReceiptRuleSetRequest> {
    private static final SdkField<String> RULE_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleSetName").getter(getter(ReorderReceiptRuleSetRequest::ruleSetName))
            .setter(setter(Builder::ruleSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleSetName").build()).build();

    private static final SdkField<List<String>> RULE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RuleNames")
            .getter(getter(ReorderReceiptRuleSetRequest::ruleNames))
            .setter(setter(Builder::ruleNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_SET_NAME_FIELD,
            RULE_NAMES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String ruleSetName;

    private final List<String> ruleNames;

    private ReorderReceiptRuleSetRequest(BuilderImpl builder) {
        super(builder);
        this.ruleSetName = builder.ruleSetName;
        this.ruleNames = builder.ruleNames;
    }

    /**
     * <p>
     * The name of the receipt rule set to reorder.
     * </p>
     * 
     * @return The name of the receipt rule set to reorder.
     */
    public final String ruleSetName() {
        return ruleSetName;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuleNames property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRuleNames() {
        return ruleNames != null && !(ruleNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The specified receipt rule set's receipt rules, in order.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuleNames} method.
     * </p>
     * 
     * @return The specified receipt rule set's receipt rules, in order.
     */
    public final List<String> ruleNames() {
        return ruleNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(ruleSetName());
        hashCode = 31 * hashCode + Objects.hashCode(hasRuleNames() ? ruleNames() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReorderReceiptRuleSetRequest)) {
            return false;
        }
        ReorderReceiptRuleSetRequest other = (ReorderReceiptRuleSetRequest) obj;
        return Objects.equals(ruleSetName(), other.ruleSetName()) && hasRuleNames() == other.hasRuleNames()
                && Objects.equals(ruleNames(), other.ruleNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReorderReceiptRuleSetRequest").add("RuleSetName", ruleSetName())
                .add("RuleNames", hasRuleNames() ? ruleNames() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleSetName":
            return Optional.ofNullable(clazz.cast(ruleSetName()));
        case "RuleNames":
            return Optional.ofNullable(clazz.cast(ruleNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RuleSetName", RULE_SET_NAME_FIELD);
        map.put("RuleNames", RULE_NAMES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ReorderReceiptRuleSetRequest, T> g) {
        return obj -> g.apply((ReorderReceiptRuleSetRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo, CopyableBuilder<Builder, ReorderReceiptRuleSetRequest> {
        /**
         * <p>
         * The name of the receipt rule set to reorder.
         * </p>
         * 
         * @param ruleSetName
         *        The name of the receipt rule set to reorder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleSetName(String ruleSetName);

        /**
         * <p>
         * The specified receipt rule set's receipt rules, in order.
         * </p>
         * 
         * @param ruleNames
         *        The specified receipt rule set's receipt rules, in order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleNames(Collection<String> ruleNames);

        /**
         * <p>
         * The specified receipt rule set's receipt rules, in order.
         * </p>
         * 
         * @param ruleNames
         *        The specified receipt rule set's receipt rules, in order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleNames(String... ruleNames);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String ruleSetName;

        private List<String> ruleNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ReorderReceiptRuleSetRequest model) {
            super(model);
            ruleSetName(model.ruleSetName);
            ruleNames(model.ruleNames);
        }

        public final String getRuleSetName() {
            return ruleSetName;
        }

        public final void setRuleSetName(String ruleSetName) {
            this.ruleSetName = ruleSetName;
        }

        @Override
        public final Builder ruleSetName(String ruleSetName) {
            this.ruleSetName = ruleSetName;
            return this;
        }

        public final Collection<String> getRuleNames() {
            if (ruleNames instanceof SdkAutoConstructList) {
                return null;
            }
            return ruleNames;
        }

        public final void setRuleNames(Collection<String> ruleNames) {
            this.ruleNames = ReceiptRuleNamesListCopier.copy(ruleNames);
        }

        @Override
        public final Builder ruleNames(Collection<String> ruleNames) {
            this.ruleNames = ReceiptRuleNamesListCopier.copy(ruleNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleNames(String... ruleNames) {
            ruleNames(Arrays.asList(ruleNames));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ReorderReceiptRuleSetRequest build() {
            return new ReorderReceiptRuleSetRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
