/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to specify the Amazon SNS topic to which Amazon SES publishes bounce, complaint, or delivery
 * notifications for emails sent with that identity as the source. For information about Amazon SES notifications, see
 * the <a href="https://docs.aws.amazon.com/ses/latest/dg/monitor-sending-activity-using-notifications-sns.html">Amazon
 * SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SetIdentityNotificationTopicRequest extends SesRequest implements
        ToCopyableBuilder<SetIdentityNotificationTopicRequest.Builder, SetIdentityNotificationTopicRequest> {
    private static final SdkField<String> IDENTITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Identity").getter(getter(SetIdentityNotificationTopicRequest::identity))
            .setter(setter(Builder::identity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Identity").build()).build();

    private static final SdkField<String> NOTIFICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotificationType").getter(getter(SetIdentityNotificationTopicRequest::notificationTypeAsString))
            .setter(setter(Builder::notificationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationType").build()).build();

    private static final SdkField<String> SNS_TOPIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnsTopic").getter(getter(SetIdentityNotificationTopicRequest::snsTopic))
            .setter(setter(Builder::snsTopic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsTopic").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IDENTITY_FIELD,
            NOTIFICATION_TYPE_FIELD, SNS_TOPIC_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Identity", IDENTITY_FIELD);
                    put("NotificationType", NOTIFICATION_TYPE_FIELD);
                    put("SnsTopic", SNS_TOPIC_FIELD);
                }
            });

    private final String identity;

    private final String notificationType;

    private final String snsTopic;

    private SetIdentityNotificationTopicRequest(BuilderImpl builder) {
        super(builder);
        this.identity = builder.identity;
        this.notificationType = builder.notificationType;
        this.snsTopic = builder.snsTopic;
    }

    /**
     * <p>
     * The identity (email address or domain) for the Amazon SNS topic.
     * </p>
     * <important>
     * <p>
     * You can only specify a verified identity for this parameter.
     * </p>
     * </important>
     * <p>
     * You can specify an identity by using its name or by using its Amazon Resource Name (ARN). The following examples
     * are all valid identities: <code>sender@example.com</code>, <code>example.com</code>,
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
     * </p>
     * 
     * @return The identity (email address or domain) for the Amazon SNS topic.</p> <important>
     *         <p>
     *         You can only specify a verified identity for this parameter.
     *         </p>
     *         </important>
     *         <p>
     *         You can specify an identity by using its name or by using its Amazon Resource Name (ARN). The following
     *         examples are all valid identities: <code>sender@example.com</code>, <code>example.com</code>,
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
     */
    public final String identity() {
        return identity;
    }

    /**
     * <p>
     * The type of notifications that are published to the specified Amazon SNS topic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return The type of notifications that are published to the specified Amazon SNS topic.
     * @see NotificationType
     */
    public final NotificationType notificationType() {
        return NotificationType.fromValue(notificationType);
    }

    /**
     * <p>
     * The type of notifications that are published to the specified Amazon SNS topic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return The type of notifications that are published to the specified Amazon SNS topic.
     * @see NotificationType
     */
    public final String notificationTypeAsString() {
        return notificationType;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon SNS topic. If the parameter is omitted from the request or a null
     * value is passed, <code>SnsTopic</code> is cleared and publishing is disabled.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon SNS topic. If the parameter is omitted from the request or a
     *         null value is passed, <code>SnsTopic</code> is cleared and publishing is disabled.
     */
    public final String snsTopic() {
        return snsTopic;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(identity());
        hashCode = 31 * hashCode + Objects.hashCode(notificationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(snsTopic());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SetIdentityNotificationTopicRequest)) {
            return false;
        }
        SetIdentityNotificationTopicRequest other = (SetIdentityNotificationTopicRequest) obj;
        return Objects.equals(identity(), other.identity())
                && Objects.equals(notificationTypeAsString(), other.notificationTypeAsString())
                && Objects.equals(snsTopic(), other.snsTopic());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SetIdentityNotificationTopicRequest").add("Identity", identity())
                .add("NotificationType", notificationTypeAsString()).add("SnsTopic", snsTopic()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Identity":
            return Optional.ofNullable(clazz.cast(identity()));
        case "NotificationType":
            return Optional.ofNullable(clazz.cast(notificationTypeAsString()));
        case "SnsTopic":
            return Optional.ofNullable(clazz.cast(snsTopic()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<SetIdentityNotificationTopicRequest, T> g) {
        return obj -> g.apply((SetIdentityNotificationTopicRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo, CopyableBuilder<Builder, SetIdentityNotificationTopicRequest> {
        /**
         * <p>
         * The identity (email address or domain) for the Amazon SNS topic.
         * </p>
         * <important>
         * <p>
         * You can only specify a verified identity for this parameter.
         * </p>
         * </important>
         * <p>
         * You can specify an identity by using its name or by using its Amazon Resource Name (ARN). The following
         * examples are all valid identities: <code>sender@example.com</code>, <code>example.com</code>,
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
         * </p>
         * 
         * @param identity
         *        The identity (email address or domain) for the Amazon SNS topic.</p> <important>
         *        <p>
         *        You can only specify a verified identity for this parameter.
         *        </p>
         *        </important>
         *        <p>
         *        You can specify an identity by using its name or by using its Amazon Resource Name (ARN). The
         *        following examples are all valid identities: <code>sender@example.com</code>, <code>example.com</code>, <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identity(String identity);

        /**
         * <p>
         * The type of notifications that are published to the specified Amazon SNS topic.
         * </p>
         * 
         * @param notificationType
         *        The type of notifications that are published to the specified Amazon SNS topic.
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(String notificationType);

        /**
         * <p>
         * The type of notifications that are published to the specified Amazon SNS topic.
         * </p>
         * 
         * @param notificationType
         *        The type of notifications that are published to the specified Amazon SNS topic.
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(NotificationType notificationType);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon SNS topic. If the parameter is omitted from the request or a
         * null value is passed, <code>SnsTopic</code> is cleared and publishing is disabled.
         * </p>
         * 
         * @param snsTopic
         *        The Amazon Resource Name (ARN) of the Amazon SNS topic. If the parameter is omitted from the request
         *        or a null value is passed, <code>SnsTopic</code> is cleared and publishing is disabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopic(String snsTopic);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String identity;

        private String notificationType;

        private String snsTopic;

        private BuilderImpl() {
        }

        private BuilderImpl(SetIdentityNotificationTopicRequest model) {
            super(model);
            identity(model.identity);
            notificationType(model.notificationType);
            snsTopic(model.snsTopic);
        }

        public final String getIdentity() {
            return identity;
        }

        public final void setIdentity(String identity) {
            this.identity = identity;
        }

        @Override
        public final Builder identity(String identity) {
            this.identity = identity;
            return this;
        }

        public final String getNotificationType() {
            return notificationType;
        }

        public final void setNotificationType(String notificationType) {
            this.notificationType = notificationType;
        }

        @Override
        public final Builder notificationType(String notificationType) {
            this.notificationType = notificationType;
            return this;
        }

        @Override
        public final Builder notificationType(NotificationType notificationType) {
            this.notificationType(notificationType == null ? null : notificationType.toString());
            return this;
        }

        public final String getSnsTopic() {
            return snsTopic;
        }

        public final void setSnsTopic(String snsTopic) {
            this.snsTopic = snsTopic;
        }

        @Override
        public final Builder snsTopic(String snsTopic) {
            this.snsTopic = snsTopic;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SetIdentityNotificationTopicRequest build() {
            return new SetIdentityNotificationTopicRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
