/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send a bounce message to the sender of an email you received through Amazon SES.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SendBounceRequest extends SesRequest implements
        ToCopyableBuilder<SendBounceRequest.Builder, SendBounceRequest> {
    private static final SdkField<String> ORIGINAL_MESSAGE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OriginalMessageId").getter(getter(SendBounceRequest::originalMessageId))
            .setter(setter(Builder::originalMessageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginalMessageId").build()).build();

    private static final SdkField<String> BOUNCE_SENDER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BounceSender").getter(getter(SendBounceRequest::bounceSender)).setter(setter(Builder::bounceSender))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BounceSender").build()).build();

    private static final SdkField<String> EXPLANATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Explanation").getter(getter(SendBounceRequest::explanation)).setter(setter(Builder::explanation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Explanation").build()).build();

    private static final SdkField<MessageDsn> MESSAGE_DSN_FIELD = SdkField.<MessageDsn> builder(MarshallingType.SDK_POJO)
            .memberName("MessageDsn").getter(getter(SendBounceRequest::messageDsn)).setter(setter(Builder::messageDsn))
            .constructor(MessageDsn::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageDsn").build()).build();

    private static final SdkField<List<BouncedRecipientInfo>> BOUNCED_RECIPIENT_INFO_LIST_FIELD = SdkField
            .<List<BouncedRecipientInfo>> builder(MarshallingType.LIST)
            .memberName("BouncedRecipientInfoList")
            .getter(getter(SendBounceRequest::bouncedRecipientInfoList))
            .setter(setter(Builder::bouncedRecipientInfoList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BouncedRecipientInfoList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BouncedRecipientInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(BouncedRecipientInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> BOUNCE_SENDER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BounceSenderArn").getter(getter(SendBounceRequest::bounceSenderArn))
            .setter(setter(Builder::bounceSenderArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BounceSenderArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(ORIGINAL_MESSAGE_ID_FIELD, BOUNCE_SENDER_FIELD, EXPLANATION_FIELD, MESSAGE_DSN_FIELD,
                    BOUNCED_RECIPIENT_INFO_LIST_FIELD, BOUNCE_SENDER_ARN_FIELD));

    private final String originalMessageId;

    private final String bounceSender;

    private final String explanation;

    private final MessageDsn messageDsn;

    private final List<BouncedRecipientInfo> bouncedRecipientInfoList;

    private final String bounceSenderArn;

    private SendBounceRequest(BuilderImpl builder) {
        super(builder);
        this.originalMessageId = builder.originalMessageId;
        this.bounceSender = builder.bounceSender;
        this.explanation = builder.explanation;
        this.messageDsn = builder.messageDsn;
        this.bouncedRecipientInfoList = builder.bouncedRecipientInfoList;
        this.bounceSenderArn = builder.bounceSenderArn;
    }

    /**
     * <p>
     * The message ID of the message to be bounced.
     * </p>
     * 
     * @return The message ID of the message to be bounced.
     */
    public final String originalMessageId() {
        return originalMessageId;
    }

    /**
     * <p>
     * The address to use in the "From" header of the bounce message. This must be an identity that you have verified
     * with Amazon SES.
     * </p>
     * 
     * @return The address to use in the "From" header of the bounce message. This must be an identity that you have
     *         verified with Amazon SES.
     */
    public final String bounceSender() {
        return bounceSender;
    }

    /**
     * <p>
     * Human-readable text for the bounce message to explain the failure. If not specified, the text is auto-generated
     * based on the bounced recipient information.
     * </p>
     * 
     * @return Human-readable text for the bounce message to explain the failure. If not specified, the text is
     *         auto-generated based on the bounced recipient information.
     */
    public final String explanation() {
        return explanation;
    }

    /**
     * <p>
     * Message-related DSN fields. If not specified, Amazon SES chooses the values.
     * </p>
     * 
     * @return Message-related DSN fields. If not specified, Amazon SES chooses the values.
     */
    public final MessageDsn messageDsn() {
        return messageDsn;
    }

    /**
     * For responses, this returns true if the service returned a value for the BouncedRecipientInfoList property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasBouncedRecipientInfoList() {
        return bouncedRecipientInfoList != null && !(bouncedRecipientInfoList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of recipients of the bounced message, including the information required to create the Delivery Status
     * Notifications (DSNs) for the recipients. You must specify at least one <code>BouncedRecipientInfo</code> in the
     * list.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBouncedRecipientInfoList} method.
     * </p>
     * 
     * @return A list of recipients of the bounced message, including the information required to create the Delivery
     *         Status Notifications (DSNs) for the recipients. You must specify at least one
     *         <code>BouncedRecipientInfo</code> in the list.
     */
    public final List<BouncedRecipientInfo> bouncedRecipientInfoList() {
        return bouncedRecipientInfoList;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the address in the "From" header of the bounce. For more
     * information about sending authorization, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization.html">Amazon SES Developer Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the address in the "From" header of the
     *         bounce. For more information about sending authorization, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization.html">Amazon SES Developer
     *         Guide</a>.
     */
    public final String bounceSenderArn() {
        return bounceSenderArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(originalMessageId());
        hashCode = 31 * hashCode + Objects.hashCode(bounceSender());
        hashCode = 31 * hashCode + Objects.hashCode(explanation());
        hashCode = 31 * hashCode + Objects.hashCode(messageDsn());
        hashCode = 31 * hashCode + Objects.hashCode(hasBouncedRecipientInfoList() ? bouncedRecipientInfoList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(bounceSenderArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendBounceRequest)) {
            return false;
        }
        SendBounceRequest other = (SendBounceRequest) obj;
        return Objects.equals(originalMessageId(), other.originalMessageId())
                && Objects.equals(bounceSender(), other.bounceSender()) && Objects.equals(explanation(), other.explanation())
                && Objects.equals(messageDsn(), other.messageDsn())
                && hasBouncedRecipientInfoList() == other.hasBouncedRecipientInfoList()
                && Objects.equals(bouncedRecipientInfoList(), other.bouncedRecipientInfoList())
                && Objects.equals(bounceSenderArn(), other.bounceSenderArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SendBounceRequest").add("OriginalMessageId", originalMessageId())
                .add("BounceSender", bounceSender()).add("Explanation", explanation()).add("MessageDsn", messageDsn())
                .add("BouncedRecipientInfoList", hasBouncedRecipientInfoList() ? bouncedRecipientInfoList() : null)
                .add("BounceSenderArn", bounceSenderArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OriginalMessageId":
            return Optional.ofNullable(clazz.cast(originalMessageId()));
        case "BounceSender":
            return Optional.ofNullable(clazz.cast(bounceSender()));
        case "Explanation":
            return Optional.ofNullable(clazz.cast(explanation()));
        case "MessageDsn":
            return Optional.ofNullable(clazz.cast(messageDsn()));
        case "BouncedRecipientInfoList":
            return Optional.ofNullable(clazz.cast(bouncedRecipientInfoList()));
        case "BounceSenderArn":
            return Optional.ofNullable(clazz.cast(bounceSenderArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SendBounceRequest, T> g) {
        return obj -> g.apply((SendBounceRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo, CopyableBuilder<Builder, SendBounceRequest> {
        /**
         * <p>
         * The message ID of the message to be bounced.
         * </p>
         * 
         * @param originalMessageId
         *        The message ID of the message to be bounced.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originalMessageId(String originalMessageId);

        /**
         * <p>
         * The address to use in the "From" header of the bounce message. This must be an identity that you have
         * verified with Amazon SES.
         * </p>
         * 
         * @param bounceSender
         *        The address to use in the "From" header of the bounce message. This must be an identity that you have
         *        verified with Amazon SES.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bounceSender(String bounceSender);

        /**
         * <p>
         * Human-readable text for the bounce message to explain the failure. If not specified, the text is
         * auto-generated based on the bounced recipient information.
         * </p>
         * 
         * @param explanation
         *        Human-readable text for the bounce message to explain the failure. If not specified, the text is
         *        auto-generated based on the bounced recipient information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder explanation(String explanation);

        /**
         * <p>
         * Message-related DSN fields. If not specified, Amazon SES chooses the values.
         * </p>
         * 
         * @param messageDsn
         *        Message-related DSN fields. If not specified, Amazon SES chooses the values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageDsn(MessageDsn messageDsn);

        /**
         * <p>
         * Message-related DSN fields. If not specified, Amazon SES chooses the values.
         * </p>
         * This is a convenience method that creates an instance of the {@link MessageDsn.Builder} avoiding the need to
         * create one manually via {@link MessageDsn#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MessageDsn.Builder#build()} is called immediately and its result
         * is passed to {@link #messageDsn(MessageDsn)}.
         * 
         * @param messageDsn
         *        a consumer that will call methods on {@link MessageDsn.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messageDsn(MessageDsn)
         */
        default Builder messageDsn(Consumer<MessageDsn.Builder> messageDsn) {
            return messageDsn(MessageDsn.builder().applyMutation(messageDsn).build());
        }

        /**
         * <p>
         * A list of recipients of the bounced message, including the information required to create the Delivery Status
         * Notifications (DSNs) for the recipients. You must specify at least one <code>BouncedRecipientInfo</code> in
         * the list.
         * </p>
         * 
         * @param bouncedRecipientInfoList
         *        A list of recipients of the bounced message, including the information required to create the Delivery
         *        Status Notifications (DSNs) for the recipients. You must specify at least one
         *        <code>BouncedRecipientInfo</code> in the list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bouncedRecipientInfoList(Collection<BouncedRecipientInfo> bouncedRecipientInfoList);

        /**
         * <p>
         * A list of recipients of the bounced message, including the information required to create the Delivery Status
         * Notifications (DSNs) for the recipients. You must specify at least one <code>BouncedRecipientInfo</code> in
         * the list.
         * </p>
         * 
         * @param bouncedRecipientInfoList
         *        A list of recipients of the bounced message, including the information required to create the Delivery
         *        Status Notifications (DSNs) for the recipients. You must specify at least one
         *        <code>BouncedRecipientInfo</code> in the list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bouncedRecipientInfoList(BouncedRecipientInfo... bouncedRecipientInfoList);

        /**
         * <p>
         * A list of recipients of the bounced message, including the information required to create the Delivery Status
         * Notifications (DSNs) for the recipients. You must specify at least one <code>BouncedRecipientInfo</code> in
         * the list.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ses.model.BouncedRecipientInfo.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ses.model.BouncedRecipientInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ses.model.BouncedRecipientInfo.Builder#build()} is called immediately
         * and its result is passed to {@link #bouncedRecipientInfoList(List<BouncedRecipientInfo>)}.
         * 
         * @param bouncedRecipientInfoList
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ses.model.BouncedRecipientInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bouncedRecipientInfoList(java.util.Collection<BouncedRecipientInfo>)
         */
        Builder bouncedRecipientInfoList(Consumer<BouncedRecipientInfo.Builder>... bouncedRecipientInfoList);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the address in the "From" header of the bounce. For
         * more information about sending authorization, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization.html">Amazon SES Developer Guide</a>.
         * </p>
         * 
         * @param bounceSenderArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the address in the "From"
         *        header of the bounce. For more information about sending authorization, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization.html">Amazon SES Developer
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bounceSenderArn(String bounceSenderArn);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String originalMessageId;

        private String bounceSender;

        private String explanation;

        private MessageDsn messageDsn;

        private List<BouncedRecipientInfo> bouncedRecipientInfoList = DefaultSdkAutoConstructList.getInstance();

        private String bounceSenderArn;

        private BuilderImpl() {
        }

        private BuilderImpl(SendBounceRequest model) {
            super(model);
            originalMessageId(model.originalMessageId);
            bounceSender(model.bounceSender);
            explanation(model.explanation);
            messageDsn(model.messageDsn);
            bouncedRecipientInfoList(model.bouncedRecipientInfoList);
            bounceSenderArn(model.bounceSenderArn);
        }

        public final String getOriginalMessageId() {
            return originalMessageId;
        }

        public final void setOriginalMessageId(String originalMessageId) {
            this.originalMessageId = originalMessageId;
        }

        @Override
        public final Builder originalMessageId(String originalMessageId) {
            this.originalMessageId = originalMessageId;
            return this;
        }

        public final String getBounceSender() {
            return bounceSender;
        }

        public final void setBounceSender(String bounceSender) {
            this.bounceSender = bounceSender;
        }

        @Override
        public final Builder bounceSender(String bounceSender) {
            this.bounceSender = bounceSender;
            return this;
        }

        public final String getExplanation() {
            return explanation;
        }

        public final void setExplanation(String explanation) {
            this.explanation = explanation;
        }

        @Override
        public final Builder explanation(String explanation) {
            this.explanation = explanation;
            return this;
        }

        public final MessageDsn.Builder getMessageDsn() {
            return messageDsn != null ? messageDsn.toBuilder() : null;
        }

        public final void setMessageDsn(MessageDsn.BuilderImpl messageDsn) {
            this.messageDsn = messageDsn != null ? messageDsn.build() : null;
        }

        @Override
        public final Builder messageDsn(MessageDsn messageDsn) {
            this.messageDsn = messageDsn;
            return this;
        }

        public final List<BouncedRecipientInfo.Builder> getBouncedRecipientInfoList() {
            List<BouncedRecipientInfo.Builder> result = BouncedRecipientInfoListCopier
                    .copyToBuilder(this.bouncedRecipientInfoList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBouncedRecipientInfoList(Collection<BouncedRecipientInfo.BuilderImpl> bouncedRecipientInfoList) {
            this.bouncedRecipientInfoList = BouncedRecipientInfoListCopier.copyFromBuilder(bouncedRecipientInfoList);
        }

        @Override
        public final Builder bouncedRecipientInfoList(Collection<BouncedRecipientInfo> bouncedRecipientInfoList) {
            this.bouncedRecipientInfoList = BouncedRecipientInfoListCopier.copy(bouncedRecipientInfoList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder bouncedRecipientInfoList(BouncedRecipientInfo... bouncedRecipientInfoList) {
            bouncedRecipientInfoList(Arrays.asList(bouncedRecipientInfoList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder bouncedRecipientInfoList(Consumer<BouncedRecipientInfo.Builder>... bouncedRecipientInfoList) {
            bouncedRecipientInfoList(Stream.of(bouncedRecipientInfoList)
                    .map(c -> BouncedRecipientInfo.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getBounceSenderArn() {
            return bounceSenderArn;
        }

        public final void setBounceSenderArn(String bounceSenderArn) {
            this.bounceSenderArn = bounceSenderArn;
        }

        @Override
        public final Builder bounceSenderArn(String bounceSenderArn) {
            this.bounceSenderArn = bounceSenderArn;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SendBounceRequest build() {
            return new SendBounceRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
