/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.servicediscovery.model.CreateHttpNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreateHttpNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreatePrivateDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreatePrivateDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreatePublicDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreatePublicDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreateServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreateServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CustomHealthNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.DeleteNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeleteNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DeleteServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeleteServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DeregisterInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeregisterInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesRequest;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesResponse;
import software.amazon.awssdk.services.servicediscovery.model.DuplicateRequestException;
import software.amazon.awssdk.services.servicediscovery.model.GetInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetOperationRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetOperationResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.InstanceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.InvalidInputException;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListNamespacesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListNamespacesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListOperationsRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListOperationsResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.servicediscovery.model.NamespaceAlreadyExistsException;
import software.amazon.awssdk.services.servicediscovery.model.NamespaceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.OperationNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.RegisterInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.RegisterInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.RequestLimitExceededException;
import software.amazon.awssdk.services.servicediscovery.model.ResourceInUseException;
import software.amazon.awssdk.services.servicediscovery.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.servicediscovery.model.ResourceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceAlreadyExistsException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceDiscoveryException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.TagResourceRequest;
import software.amazon.awssdk.services.servicediscovery.model.TagResourceResponse;
import software.amazon.awssdk.services.servicediscovery.model.TooManyTagsException;
import software.amazon.awssdk.services.servicediscovery.model.UntagResourceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UntagResourceResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdateHttpNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateHttpNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdateInstanceCustomHealthStatusRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateInstanceCustomHealthStatusResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdatePrivateDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdatePrivateDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdatePublicDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdatePublicDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdateServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateServiceResponse;
import software.amazon.awssdk.services.servicediscovery.transform.CreateHttpNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreatePrivateDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreatePublicDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreateServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeleteNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeleteServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeregisterInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DiscoverInstancesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetInstancesHealthStatusRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetOperationRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListInstancesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListNamespacesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.RegisterInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateHttpNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateInstanceCustomHealthStatusRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdatePrivateDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdatePublicDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateServiceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ServiceDiscoveryClient}.
 *
 * @see ServiceDiscoveryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServiceDiscoveryClient implements ServiceDiscoveryClient {
    private static final Logger log = Logger.loggerFor(DefaultServiceDiscoveryClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ServiceDiscoveryServiceClientConfiguration serviceClientConfiguration;

    protected DefaultServiceDiscoveryClient(ServiceDiscoveryServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an HTTP namespace. Service instances registered using an HTTP namespace can be discovered using a
     * <code>DiscoverInstances</code> request but can't be discovered using DNS.
     * </p>
     * <p>
     * For the current quota on the number of namespaces that you can create using the same Amazon Web Services account,
     * see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map quotas</a> in the
     * <i>Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createHttpNamespaceRequest
     * @return Result of the CreateHttpNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceAlreadyExistsException
     *         The namespace that you're trying to create already exists.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the quota on the number of resources.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws TooManyTagsException
     *         The list of tags on the resource is over the quota. The maximum number of tags that can be applied to a
     *         resource is 50.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.CreateHttpNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreateHttpNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateHttpNamespaceResponse createHttpNamespace(CreateHttpNamespaceRequest createHttpNamespaceRequest)
            throws InvalidInputException, NamespaceAlreadyExistsException, ResourceLimitExceededException,
            DuplicateRequestException, TooManyTagsException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHttpNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateHttpNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHttpNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHttpNamespace");

            return clientHandler.execute(new ClientExecutionParams<CreateHttpNamespaceRequest, CreateHttpNamespaceResponse>()
                    .withOperationName("CreateHttpNamespace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createHttpNamespaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateHttpNamespaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a private namespace based on DNS, which is visible only inside a specified Amazon VPC. The namespace
     * defines your service naming scheme. For example, if you name your namespace <code>example.com</code> and name
     * your service <code>backend</code>, the resulting DNS name for the service is <code>backend.example.com</code>.
     * Service instances that are registered using a private DNS namespace can be discovered using either a
     * <code>DiscoverInstances</code> request or using DNS. For the current quota on the number of namespaces that you
     * can create using the same Amazon Web Services account, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map quotas</a> in the <i>Cloud
     * Map Developer Guide</i>.
     * </p>
     *
     * @param createPrivateDnsNamespaceRequest
     * @return Result of the CreatePrivateDnsNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceAlreadyExistsException
     *         The namespace that you're trying to create already exists.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the quota on the number of resources.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws TooManyTagsException
     *         The list of tags on the resource is over the quota. The maximum number of tags that can be applied to a
     *         resource is 50.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.CreatePrivateDnsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreatePrivateDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePrivateDnsNamespaceResponse createPrivateDnsNamespace(
            CreatePrivateDnsNamespaceRequest createPrivateDnsNamespaceRequest) throws InvalidInputException,
            NamespaceAlreadyExistsException, ResourceLimitExceededException, DuplicateRequestException, TooManyTagsException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePrivateDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePrivateDnsNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPrivateDnsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePrivateDnsNamespace");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePrivateDnsNamespaceRequest, CreatePrivateDnsNamespaceResponse>()
                            .withOperationName("CreatePrivateDnsNamespace").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createPrivateDnsNamespaceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePrivateDnsNamespaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a public namespace based on DNS, which is visible on the internet. The namespace defines your service
     * naming scheme. For example, if you name your namespace <code>example.com</code> and name your service
     * <code>backend</code>, the resulting DNS name for the service is <code>backend.example.com</code>. You can
     * discover instances that were registered with a public DNS namespace by using either a
     * <code>DiscoverInstances</code> request or using DNS. For the current quota on the number of namespaces that you
     * can create using the same Amazon Web Services account, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map quotas</a> in the <i>Cloud
     * Map Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * The <code>CreatePublicDnsNamespace</code> API operation is not supported in the Amazon Web Services GovCloud (US)
     * Regions.
     * </p>
     * </important>
     *
     * @param createPublicDnsNamespaceRequest
     * @return Result of the CreatePublicDnsNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceAlreadyExistsException
     *         The namespace that you're trying to create already exists.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the quota on the number of resources.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws TooManyTagsException
     *         The list of tags on the resource is over the quota. The maximum number of tags that can be applied to a
     *         resource is 50.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.CreatePublicDnsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreatePublicDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePublicDnsNamespaceResponse createPublicDnsNamespace(
            CreatePublicDnsNamespaceRequest createPublicDnsNamespaceRequest) throws InvalidInputException,
            NamespaceAlreadyExistsException, ResourceLimitExceededException, DuplicateRequestException, TooManyTagsException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePublicDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePublicDnsNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPublicDnsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePublicDnsNamespace");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePublicDnsNamespaceRequest, CreatePublicDnsNamespaceResponse>()
                            .withOperationName("CreatePublicDnsNamespace").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createPublicDnsNamespaceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePublicDnsNamespaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a service. This action defines the configuration for the following entities:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For public and private DNS namespaces, one of the following combinations of DNS records in Amazon Route 53:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>A</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AAAA</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>A</code> and <code>AAAA</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SRV</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CNAME</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Optionally, a health check
     * </p>
     * </li>
     * </ul>
     * <p>
     * After you create the service, you can submit a <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a> request,
     * and Cloud Map uses the values in the configuration to create the specified entities.
     * </p>
     * <p>
     * For the current quota on the number of instances that you can register using the same namespace and using the
     * same service, see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map
     * quotas</a> in the <i>Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createServiceRequest
     * @return Result of the CreateService operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the quota on the number of resources.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws ServiceAlreadyExistsException
     *         The service can't be created because a service with the same name already exists.
     * @throws TooManyTagsException
     *         The list of tags on the resource is over the quota. The maximum number of tags that can be applied to a
     *         resource is 50.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.CreateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreateService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateServiceResponse createService(CreateServiceRequest createServiceRequest) throws InvalidInputException,
            ResourceLimitExceededException, NamespaceNotFoundException, ServiceAlreadyExistsException, TooManyTagsException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateService");

            return clientHandler.execute(new ClientExecutionParams<CreateServiceRequest, CreateServiceResponse>()
                    .withOperationName("CreateService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a namespace from the current account. If the namespace still contains one or more services, the request
     * fails.
     * </p>
     *
     * @param deleteNamespaceRequest
     * @return Result of the DeleteNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.DeleteNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeleteNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteNamespaceResponse deleteNamespace(DeleteNamespaceRequest deleteNamespaceRequest) throws InvalidInputException,
            NamespaceNotFoundException, ResourceInUseException, DuplicateRequestException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNamespace");

            return clientHandler.execute(new ClientExecutionParams<DeleteNamespaceRequest, DeleteNamespaceResponse>()
                    .withOperationName("DeleteNamespace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteNamespaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteNamespaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified service. If the service still contains one or more registered instances, the request fails.
     * </p>
     *
     * @param deleteServiceRequest
     * @return Result of the DeleteService operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.DeleteService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeleteService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteServiceResponse deleteService(DeleteServiceRequest deleteServiceRequest) throws InvalidInputException,
            ServiceNotFoundException, ResourceInUseException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteService");

            return clientHandler.execute(new ClientExecutionParams<DeleteServiceRequest, DeleteServiceResponse>()
                    .withOperationName("DeleteService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the Amazon Route 53 DNS records and health check, if any, that Cloud Map created for the specified
     * instance.
     * </p>
     *
     * @param deregisterInstanceRequest
     * @return Result of the DeregisterInstance operation returned by the service.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws InstanceNotFoundException
     *         No instance exists with the specified ID, or the instance was recently registered, and information about
     *         the instance hasn't propagated yet.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.DeregisterInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeregisterInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterInstanceResponse deregisterInstance(DeregisterInstanceRequest deregisterInstanceRequest)
            throws DuplicateRequestException, InvalidInputException, InstanceNotFoundException, ResourceInUseException,
            ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterInstance");

            return clientHandler.execute(new ClientExecutionParams<DeregisterInstanceRequest, DeregisterInstanceResponse>()
                    .withOperationName("DeregisterInstance").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deregisterInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeregisterInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Discovers registered instances for a specified namespace and service. You can use <code>DiscoverInstances</code>
     * to discover instances for any type of namespace. For public and private DNS namespaces, you can also use DNS
     * queries to discover instances.
     * </p>
     *
     * @param discoverInstancesRequest
     * @return Result of the DiscoverInstances operation returned by the service.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws RequestLimitExceededException
     *         The operation can't be completed because you've reached the quota for the number of requests. For more
     *         information, see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/throttling.html">Cloud Map API
     *         request throttling quota</a> in the <i>Cloud Map Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.DiscoverInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DiscoverInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DiscoverInstancesResponse discoverInstances(DiscoverInstancesRequest discoverInstancesRequest)
            throws ServiceNotFoundException, NamespaceNotFoundException, InvalidInputException, RequestLimitExceededException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DiscoverInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DiscoverInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, discoverInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DiscoverInstances");
            String hostPrefix = "data-";
            String resolvedHostExpression = "data-";

            return clientHandler.execute(new ClientExecutionParams<DiscoverInstancesRequest, DiscoverInstancesResponse>()
                    .withOperationName("DiscoverInstances").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(discoverInstancesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DiscoverInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a specified instance.
     * </p>
     *
     * @param getInstanceRequest
     * @return Result of the GetInstance operation returned by the service.
     * @throws InstanceNotFoundException
     *         No instance exists with the specified ID, or the instance was recently registered, and information about
     *         the instance hasn't propagated yet.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetInstanceResponse getInstance(GetInstanceRequest getInstanceRequest) throws InstanceNotFoundException,
            InvalidInputException, ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInstance");

            return clientHandler.execute(new ClientExecutionParams<GetInstanceRequest, GetInstanceResponse>()
                    .withOperationName("GetInstance").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the current health status (<code>Healthy</code>, <code>Unhealthy</code>, or <code>Unknown</code>) of one or
     * more instances that are associated with a specified service.
     * </p>
     * <note>
     * <p>
     * There's a brief delay between when you register an instance and when the health status for the instance is
     * available.
     * </p>
     * </note>
     *
     * @param getInstancesHealthStatusRequest
     * @return Result of the GetInstancesHealthStatus operation returned by the service.
     * @throws InstanceNotFoundException
     *         No instance exists with the specified ID, or the instance was recently registered, and information about
     *         the instance hasn't propagated yet.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetInstancesHealthStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstancesHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInstancesHealthStatusResponse getInstancesHealthStatus(
            GetInstancesHealthStatusRequest getInstancesHealthStatusRequest) throws InstanceNotFoundException,
            InvalidInputException, ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInstancesHealthStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInstancesHealthStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInstancesHealthStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInstancesHealthStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetInstancesHealthStatusRequest, GetInstancesHealthStatusResponse>()
                            .withOperationName("GetInstancesHealthStatus").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getInstancesHealthStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetInstancesHealthStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a namespace.
     * </p>
     *
     * @param getNamespaceRequest
     * @return Result of the GetNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetNamespace" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetNamespaceResponse getNamespace(GetNamespaceRequest getNamespaceRequest) throws InvalidInputException,
            NamespaceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNamespace");

            return clientHandler.execute(new ClientExecutionParams<GetNamespaceRequest, GetNamespaceResponse>()
                    .withOperationName("GetNamespace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getNamespaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetNamespaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about any operation that returns an operation ID in the response, such as a
     * <code>CreateService</code> request.
     * </p>
     * <note>
     * <p>
     * To get a list of operations that match specified criteria, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html">ListOperations</a>.
     * </p>
     * </note>
     *
     * @param getOperationRequest
     * @return Result of the GetOperation operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws OperationNotFoundException
     *         No operation exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetOperationResponse getOperation(GetOperationRequest getOperationRequest) throws InvalidInputException,
            OperationNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOperationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOperation");

            return clientHandler.execute(new ClientExecutionParams<GetOperationRequest, GetOperationResponse>()
                    .withOperationName("GetOperation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getOperationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetOperationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the settings for a specified service.
     * </p>
     *
     * @param getServiceRequest
     * @return Result of the GetService operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetServiceResponse getService(GetServiceRequest getServiceRequest) throws InvalidInputException,
            ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetService");

            return clientHandler
                    .execute(new ClientExecutionParams<GetServiceRequest, GetServiceResponse>().withOperationName("GetService")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getServiceRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists summary information about the instances that you registered by using a specified service.
     * </p>
     *
     * @param listInstancesRequest
     * @return Result of the ListInstances operation returned by the service.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstancesResponse listInstances(ListInstancesRequest listInstancesRequest) throws ServiceNotFoundException,
            InvalidInputException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstances");

            return clientHandler.execute(new ClientExecutionParams<ListInstancesRequest, ListInstancesResponse>()
                    .withOperationName("ListInstances").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listInstancesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists summary information about the namespaces that were created by the current Amazon Web Services account.
     * </p>
     *
     * @param listNamespacesRequest
     * @return Result of the ListNamespaces operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListNamespaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListNamespaces"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNamespacesResponse listNamespaces(ListNamespacesRequest listNamespacesRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNamespacesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListNamespacesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNamespacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNamespaces");

            return clientHandler.execute(new ClientExecutionParams<ListNamespacesRequest, ListNamespacesResponse>()
                    .withOperationName("ListNamespaces").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listNamespacesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListNamespacesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists operations that match the criteria that you specify.
     * </p>
     *
     * @param listOperationsRequest
     * @return Result of the ListOperations operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListOperationsResponse listOperations(ListOperationsRequest listOperationsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOperations");

            return clientHandler.execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                    .withOperationName("ListOperations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listOperationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists summary information for all the services that are associated with one or more specified namespaces.
     * </p>
     *
     * @param listServicesRequest
     * @return Result of the ListServices operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListServicesResponse listServices(ListServicesRequest listServicesRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListServicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");

            return clientHandler.execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                    .withOperationName("ListServices").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listServicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListServicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation can't be completed because the resource was not found.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InvalidInputException, AwsServiceException, SdkClientException,
            ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates one or more records and, optionally, creates a health check based on the settings in a
     * specified service. When you submit a <code>RegisterInstance</code> request, the following occurs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For each DNS record that you define in the service that's specified by <code>ServiceId</code>, a record is
     * created or updated in the hosted zone that's associated with the corresponding namespace.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the service includes <code>HealthCheckConfig</code>, a health check is created based on the settings in the
     * health check configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * The health check, if any, is associated with each of the new or updated records.
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * One <code>RegisterInstance</code> request must complete before you can submit another request and specify the
     * same service ID and instance ID.
     * </p>
     * </important>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html">CreateService</a>.
     * </p>
     * <p>
     * When Cloud Map receives a DNS query for the specified DNS name, it returns the applicable value:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>If the health check is healthy</b>: returns all the records
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If the health check is unhealthy</b>: returns the applicable value for the last healthy instance
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If you didn't specify a health check configuration</b>: returns all the records
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the current quota on the number of instances that you can register using the same namespace and using the
     * same service, see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map
     * quotas</a> in the <i>Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param registerInstanceRequest
     * @return Result of the RegisterInstance operation returned by the service.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the quota on the number of resources.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.RegisterInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/RegisterInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterInstanceResponse registerInstance(RegisterInstanceRequest registerInstanceRequest)
            throws DuplicateRequestException, InvalidInputException, ResourceInUseException, ResourceLimitExceededException,
            ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterInstance");

            return clientHandler.execute(new ClientExecutionParams<RegisterInstanceRequest, RegisterInstanceResponse>()
                    .withOperationName("RegisterInstance").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(registerInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RegisterInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation can't be completed because the resource was not found.
     * @throws TooManyTagsException
     *         The list of tags on the resource is over the quota. The maximum number of tags that can be applied to a
     *         resource is 50.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            TooManyTagsException, InvalidInputException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation can't be completed because the resource was not found.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InvalidInputException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an HTTP namespace.
     * </p>
     *
     * @param updateHttpNamespaceRequest
     * @return Result of the UpdateHttpNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.UpdateHttpNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateHttpNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateHttpNamespaceResponse updateHttpNamespace(UpdateHttpNamespaceRequest updateHttpNamespaceRequest)
            throws InvalidInputException, NamespaceNotFoundException, ResourceInUseException, DuplicateRequestException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateHttpNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateHttpNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHttpNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHttpNamespace");

            return clientHandler.execute(new ClientExecutionParams<UpdateHttpNamespaceRequest, UpdateHttpNamespaceResponse>()
                    .withOperationName("UpdateHttpNamespace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateHttpNamespaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateHttpNamespaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Submits a request to change the health status of a custom health check to healthy or unhealthy.
     * </p>
     * <p>
     * You can use <code>UpdateInstanceCustomHealthStatus</code> to change the status only for custom health checks,
     * which you define using <code>HealthCheckCustomConfig</code> when you create a service. You can't use it to change
     * the status for Route 53 health checks, which you define using <code>HealthCheckConfig</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_HealthCheckCustomConfig.html"
     * >HealthCheckCustomConfig</a>.
     * </p>
     *
     * @param updateInstanceCustomHealthStatusRequest
     * @return Result of the UpdateInstanceCustomHealthStatus operation returned by the service.
     * @throws InstanceNotFoundException
     *         No instance exists with the specified ID, or the instance was recently registered, and information about
     *         the instance hasn't propagated yet.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws CustomHealthNotFoundException
     *         The health check for the instance that's specified by <code>ServiceId</code> and <code>InstanceId</code>
     *         isn't a custom health check.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.UpdateInstanceCustomHealthStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateInstanceCustomHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateInstanceCustomHealthStatusResponse updateInstanceCustomHealthStatus(
            UpdateInstanceCustomHealthStatusRequest updateInstanceCustomHealthStatusRequest) throws InstanceNotFoundException,
            ServiceNotFoundException, CustomHealthNotFoundException, InvalidInputException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateInstanceCustomHealthStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateInstanceCustomHealthStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateInstanceCustomHealthStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInstanceCustomHealthStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateInstanceCustomHealthStatusRequest, UpdateInstanceCustomHealthStatusResponse>()
                            .withOperationName("UpdateInstanceCustomHealthStatus").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateInstanceCustomHealthStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateInstanceCustomHealthStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a private DNS namespace.
     * </p>
     *
     * @param updatePrivateDnsNamespaceRequest
     * @return Result of the UpdatePrivateDnsNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.UpdatePrivateDnsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdatePrivateDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdatePrivateDnsNamespaceResponse updatePrivateDnsNamespace(
            UpdatePrivateDnsNamespaceRequest updatePrivateDnsNamespaceRequest) throws InvalidInputException,
            NamespaceNotFoundException, ResourceInUseException, DuplicateRequestException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePrivateDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdatePrivateDnsNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePrivateDnsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePrivateDnsNamespace");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdatePrivateDnsNamespaceRequest, UpdatePrivateDnsNamespaceResponse>()
                            .withOperationName("UpdatePrivateDnsNamespace").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updatePrivateDnsNamespaceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdatePrivateDnsNamespaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a public DNS namespace.
     * </p>
     *
     * @param updatePublicDnsNamespaceRequest
     * @return Result of the UpdatePublicDnsNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.UpdatePublicDnsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdatePublicDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdatePublicDnsNamespaceResponse updatePublicDnsNamespace(
            UpdatePublicDnsNamespaceRequest updatePublicDnsNamespaceRequest) throws InvalidInputException,
            NamespaceNotFoundException, ResourceInUseException, DuplicateRequestException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePublicDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdatePublicDnsNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePublicDnsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePublicDnsNamespace");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdatePublicDnsNamespaceRequest, UpdatePublicDnsNamespaceResponse>()
                            .withOperationName("UpdatePublicDnsNamespace").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updatePublicDnsNamespaceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdatePublicDnsNamespaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Submits a request to perform the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Update the TTL setting for existing <code>DnsRecords</code> configurations
     * </p>
     * </li>
     * <li>
     * <p>
     * Add, update, or delete <code>HealthCheckConfig</code> for a specified service
     * </p>
     * <note>
     * <p>
     * You can't add, update, or delete a <code>HealthCheckCustomConfig</code> configuration.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * For public and private DNS namespaces, note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you omit any existing <code>DnsRecords</code> or <code>HealthCheckConfig</code> configurations from an
     * <code>UpdateService</code> request, the configurations are deleted from the service.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you omit an existing <code>HealthCheckCustomConfig</code> configuration from an <code>UpdateService</code>
     * request, the configuration isn't deleted from the service.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you update settings for a service, Cloud Map also updates the corresponding settings in all the records and
     * health checks that were created by using the specified service.
     * </p>
     *
     * @param updateServiceRequest
     * @return Result of the UpdateService operation returned by the service.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.UpdateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateServiceResponse updateService(UpdateServiceRequest updateServiceRequest) throws DuplicateRequestException,
            InvalidInputException, ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateService");

            return clientHandler.execute(new ClientExecutionParams<UpdateServiceRequest, UpdateServiceResponse>()
                    .withOperationName("UpdateService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ServiceDiscoveryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestLimitExceeded")
                                .exceptionBuilderSupplier(RequestLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUse")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequest")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceAlreadyExists")
                                .exceptionBuilderSupplier(ServiceAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceNotFound")
                                .exceptionBuilderSupplier(ServiceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InstanceNotFound")
                                .exceptionBuilderSupplier(InstanceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotFound")
                                .exceptionBuilderSupplier(OperationNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInput")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NamespaceAlreadyExists")
                                .exceptionBuilderSupplier(NamespaceAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NamespaceNotFound")
                                .exceptionBuilderSupplier(NamespaceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomHealthNotFound")
                                .exceptionBuilderSupplier(CustomHealthNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceeded")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public final ServiceDiscoveryServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
